package com.zbank.file.common.utils;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import java.io.ByteArrayOutputStream;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * @author daiwj
 * @date 2021/04/25
 * @description:
 */
public class RSAUtils {

    private static final Logger log = LoggerFactory.getLogger(RSAUtils.class);

    private static final String CHARSET = "UTF-8";

    private static final String algorithm = "SHA256withRSA";
    /**
     * RSA最大加密明文大小
     */
    private static final int MAX_ENCRYPT_BLOCK = 117;
    /**
     * RSA最大解密密文大小
     */
    private static final int MAX_DECRYPT_BLOCK = 128;

    /**
     * 请求报文签名
     *
     * @param content 签名原文
     * @return 签名密文, 注意，出错时会返回null，请注意处理
     */
    public static String sign(String content, String privateKey) throws Exception {
        Signature signature = Signature.getInstance(algorithm);
        signature.initSign(convertPrivateKey(privateKey));
        signature.update(content.getBytes(CHARSET));
        return Base64.encodeBase64String(signature.sign());
    }
    /**
     * 返回报文验签
     *
     * @param content 验签原文报文
     * @param signStr 返回签名字符串
     * @return 验签结果，注意，出错时会返回false
     */
    public static boolean verify(String content, String signStr, String publicKey) {
        try{
            Signature signature = Signature.getInstance(algorithm);
            signature.initVerify(convertPublicKey(publicKey));
            signature.update(content.getBytes(CHARSET));
            return signature.verify(Base64.decodeBase64(signStr.getBytes(CHARSET)));
        }catch (Exception e){
            log.error("verify error", e);
            return false;
        }

    }
    /**
     * 对称密钥公钥加密
     *
     * @param content 原文
     * @return 加密密文
     */
    public static String encryptByPublicKey(byte[] content, String publicKeyStr) throws Exception {
        PublicKey publicKey = convertPublicKey(publicKeyStr);
        Cipher cipher = Cipher.getInstance(publicKey.getAlgorithm());
        cipher.init(Cipher.ENCRYPT_MODE, publicKey);
        ByteArrayOutputStream out=null;
        try  {
        	 out = new ByteArrayOutputStream();
            int inputLen = content.length;
            int offSet = 0;
            byte[] cache;
            int i = 0;
            // 对数据分段加密
            while (inputLen - offSet > 0) {
                if (inputLen - offSet > MAX_ENCRYPT_BLOCK) {
                    cache = cipher.doFinal(content, offSet, MAX_ENCRYPT_BLOCK);
                } else {
                    cache = cipher.doFinal(content, offSet, inputLen - offSet);
                }
                out.write(cache, 0, cache.length);
                i++;
                offSet = i * MAX_ENCRYPT_BLOCK;
            }
            byte[] encryptedData = out.toByteArray();
            return Base64.encodeBase64String(encryptedData);
        }finally {
			if(out!=null) {
				try {
					out.close();
				}catch (Exception e) {
					log.error("ByteArrayOutputStream close error", e);
				}
				
			}
		}
    }
    /**
     * 密钥密文解密
     * @param content 密钥密文
     * @return 密钥明文
     */
    public static String decryptByPrivateKey(String content, String privateKeyStr) throws Exception {
        PrivateKey privateKey = convertPrivateKey(privateKeyStr);
        Cipher cipher = Cipher.getInstance(privateKey.getAlgorithm());
        cipher.init(Cipher.DECRYPT_MODE, privateKey);

        byte[] encryptedData = Base64.decodeBase64(content.getBytes(CHARSET));
        int inputLen = encryptedData.length;
        ByteArrayOutputStream out=null;
        try  {
        	 out = new ByteArrayOutputStream();
            int offSet = 0;
            byte[] cache;
            int i = 0;
            // 对数据分段解密
            while (inputLen - offSet > 0) {
                if (inputLen - offSet > MAX_DECRYPT_BLOCK) {
                    cache = cipher.doFinal(encryptedData, offSet, MAX_DECRYPT_BLOCK);
                } else {
                    cache = cipher.doFinal(encryptedData, offSet, inputLen - offSet);
                }
                out.write(cache, 0, cache.length);
                i++;
                offSet = i * MAX_DECRYPT_BLOCK;
            }
            return out.toString(CHARSET);
        }finally {
			if(out!=null) {
				try {
					out.close();
				}catch (Exception e) {
					log.error("ByteArrayOutputStream close error", e);
				}
				
			}
		}
    }

    private static PrivateKey convertPrivateKey(String privateKey) throws Exception {
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(Base64.decodeBase64(privateKey.getBytes(CHARSET)));
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        return keyFactory.generatePrivate(keySpec);
    }
    private static PublicKey convertPublicKey(String publicKey) throws Exception {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKey.getBytes(CHARSET)));
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        return keyFactory.generatePublic(keySpec);
    }
}
