package com.zbank.file.common.utils;


import java.math.BigInteger;

import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.asn1.gm.GMNamedCurves;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.encoders.Hex;


/**
 * @author daiwj
 * @date 2021/05/13
 * @description: SM2加解密工具类
 */
public class SM2Util {
    private static final String SPEC_NAME = "sm2p256v1";
    private static final X9ECParameters x9ECParameters = GMNamedCurves.getByName(SPEC_NAME);
    private static final ECDomainParameters ecDomainParameters = new ECDomainParameters(x9ECParameters.getCurve(), x9ECParameters.getG(), x9ECParameters.getN());
    /**
     * 加密
     * @return
     */
    public static String encrypt(String data, String sm2PubKey) throws Exception{
        String encryptesData = "";
        try {
            if (StringUtils.isBlank(data)) {
                return "";
            }
            byte[] pointBytes = Hex.decode(sm2PubKey);
            ECPoint q = x9ECParameters.getCurve().decodePoint(pointBytes);
            ECPublicKeyParameters ecPublicKeyParameters = new ECPublicKeyParameters(q, ecDomainParameters);
            //初始化加密 SM2Engine 采用和C1 || C3 || C2 的模式
            SM2Engine sm2EngineForEncrypt = new SM2Engine();
            sm2EngineForEncrypt.init(true, new ParametersWithRandom(ecPublicKeyParameters));
            byte[] dataBytes = data.getBytes();
            byte[] cipherBytes = sm2EngineForEncrypt.processBlock(dataBytes, 0, dataBytes.length);
            byte[] c1c3c2Bytes = chagneC1C2C3ToC1C3C2(cipherBytes);
            encryptesData = new String(Base64.encode(c1c3c2Bytes));
            return encryptesData;
        } catch (InvalidCipherTextException e) {
           throw e;
        }
    }

    /**
     * 解密
     */
    public static String decrypt(String data, String sm2PriKey) throws Exception{
        String decryptedData = "";
        try {
            byte[] cipherBytesToDecrypt = Base64.decode(data.getBytes());
            byte[] c1c2c3Bytes = changeC1C3C2ToC1C2C3(cipherBytesToDecrypt);
            BigInteger d = new BigInteger(sm2PriKey, 16);
            ECPrivateKeyParameters ecPrivateKeyParameters = new ECPrivateKeyParameters(d, ecDomainParameters);
            // 初始化解密 SM2Engine
            SM2Engine sm2EngineForDecrypt = new SM2Engine();
            sm2EngineForDecrypt.init(false, ecPrivateKeyParameters);
            byte[] decryptedBytes = sm2EngineForDecrypt.processBlock(c1c2c3Bytes, 0, c1c2c3Bytes.length);
            decryptedData = new String(decryptedBytes);
            return decryptedData;
        } catch (InvalidCipherTextException e) {
           throw e;
        }
    }

    /**
     * C1C2C3切换C1C3C2
     */
    private static byte[] chagneC1C2C3ToC1C3C2(byte[] c1c2c3) {
        final int c1Len = (x9ECParameters.getCurve().getFieldSize() + 7) / 8 * 2 + 1;
        final int c3Len = 32;
        byte[] resule = new byte[c1c2c3.length];
        //c1
        System.arraycopy(c1c2c3, 0, resule, 0, c1Len);
        //c3
        System.arraycopy(c1c2c3, c1c2c3.length - c3Len, resule, c1Len, c3Len);
        //c2
        System.arraycopy(c1c2c3, c1Len, resule, c1Len + c3Len, c1c2c3.length - c1Len - c3Len);
        return resule;

    }
    /**
     * C1C3C2切换C1C2C3
     */
    private static byte[] changeC1C3C2ToC1C2C3(byte[] c1c3c2) {
        //sm2p256v1的这个固定65。可看GMNamedCurves、ECCurve代码。
        final int c1Len = (x9ECParameters.getCurve().getFieldSize() + 7) / 8 * 2 + 1;
        final int c3Len = 32;
        byte[] result = new byte[c1c3c2.length];
        //c1: 0->65
        System.arraycopy(c1c3c2, 0, result, 0, c1Len);
        //c2
        System.arraycopy(c1c3c2, c1Len + c3Len, result, c1Len, c1c3c2.length - c1Len - c3Len);
        //c3
        System.arraycopy(c1c3c2, c1Len, result, c1c3c2.length - c3Len, c3Len);
        return result;
    }
}
