package com.zbank.file.sdk;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zbank.file.api.APIService;
import com.zbank.file.bean.FileDownLoadInfo;
import com.zbank.file.bean.FileQueryRequest;
import com.zbank.file.bean.FileQueryResponse;
import com.zbank.file.bean.FileSyncDownloadRequest;
import com.zbank.file.bean.FileSyncDownloadResponse;
import com.zbank.file.bean.ImageDownloadRequest;
import com.zbank.file.bean.ImageDownloadResponse;
import com.zbank.file.bean.ImageInfo;
import com.zbank.file.bean.StreamDownLoadInfo;
import com.zbank.file.common.http.config.HttpConfig;
import com.zbank.file.common.utils.Md5EncodeUtil;
import com.zbank.file.common.utils.SM4Utils;
import com.zbank.file.common.utils.StringUtil;
import com.zbank.file.constants.DealCode;
import com.zbank.file.exception.EmptyFileException;
import com.zbank.file.exception.SDKException;
import com.zbank.file.sdk.download.split.FileSplitDownLoadTask;
import com.zbank.file.secure.IPackSecure;

public abstract class BaseSDK {
	private static final Logger log = LoggerFactory.getLogger(BaseSDK.class);
	/**
	 * 用于设置多线程分片下载(本地磁盘合并)的线程池最大线程数，考虑到综合原因（网络带宽以及内存占用）
	 */
	public static int DOWNLOAD_THREADS = 4;
	/**
	 * 用于在设置多线程分片上传/下载的任务数量限定值，考虑到综合原因（网络带宽以及内存占用）
	 */
	public static int PERMIT_MAX = 4;
	/**
	 * 用于设定启动多线程上传或者下载的分片数量阀值
	 */
	public static int START_CURRENT_THRESHHOLD = 32;
	
	protected APIService apiService;

	protected HttpConfig httpConfig = HttpConfig.defaultConfig;
	protected IPackSecure packScure;
	
	/**
	 * 文件下载缓存，用于避免多线程同时在同一个目录下下载同一个文件。
	 */
	private static Map<String,File> downloadingFileCache = new ConcurrentHashMap<String,File>();
	
	protected BaseSDK(){
		if (System.getProperty("filesdk.download.threads") != null && !"".equals(System.getProperty("filesdk.download.threads"))) {
			DOWNLOAD_THREADS = Integer.parseInt(System.getProperty("filesdk.download.threads"));
			if (DOWNLOAD_THREADS > 10) {
				DOWNLOAD_THREADS = 10;
			}
		}
		if (System.getProperty("filesdk.permit.max") != null && !"".equals(System.getProperty("filesdk.permit.max"))) {
			PERMIT_MAX = Integer.parseInt(System.getProperty("filesdk.permit.max"));
			if (PERMIT_MAX > 10) {
				PERMIT_MAX = 10;
			}
		}
		if (System.getProperty("filesdk.start.concurrent.threshhold") != null && !"".equals(System.getProperty("filesdk.start.concurrent.threshhold"))) {
			START_CURRENT_THRESHHOLD = Integer.parseInt(System.getProperty("filesdk.start.concurrent.threshhold"));
			if (START_CURRENT_THRESHHOLD < 10) {
				START_CURRENT_THRESHHOLD = 10;
			}
		}
	}
	/**
	 * 从影像平台下载文件
	 * 
	 * @param fileId
	 * @param channelId
	 * @param seqNo
	 * @return
	 * @throws SDKException
	 */
	protected ImageDownloadResponse downloadImage(String fileId, String channelId, String seqNo) throws SDKException {
		ImageDownloadRequest request = new ImageDownloadRequest();
		request.setChannelId(channelId);
		request.setFileId(fileId);
		request.setSeqNo(seqNo);
		ImageDownloadResponse response = this.apiService.downloadFromImageSystem(request, httpConfig, packScure);
		if (!DealCode.SUCCESS.getCode().equals(response.getCode())) {
			throw new SDKException(response.getCode(), response.getMessage());
		}
		if (response.getImageInfo() == null) {
			throw new SDKException("文件" + fileId + "下载失败:未查到对应的文件信息!!");
		}
		if (response.getImageInfo().getBytes() == null || response.getImageInfo().getBytes().length == 0) {
			throw new SDKException("文件" + fileId + "下载失败:未查到对应的文件数据!!");
		}
		if (StringUtil.isEmpty(response.getImageInfo().getFileMd5())) {
			throw new SDKException("文件" + fileId + "下载失败:沒有MD5值!!");
		}
		ImageInfo fileInfo = response.getImageInfo();
		fileInfo.setFileSize(fileInfo.getBytes().length);
		fileInfo.setSplitSum(1);
		fileInfo.setFileName(response.getImageInfo().getFileName());
		return response;
	}

	/**
	 * 文件分片下载
	 * @param fileResponse
	 * @param fileId
	 * @param channelId
	 * @param destDir
	 * @param seqNo
	 * @param isParallel
	 * @return
	 * @throws SDKException
	 */
	protected FileDownLoadInfo downloadFile(FileQueryResponse fileResponse, String fileId, String channelId, String destDir,
			String seqNo, boolean isParallel) throws SDKException {
		String fileName = fileResponse.getFileInfo().getFileName();
		String suffix = "";
		if(fileName.contains(".")){
			suffix = fileName.substring(fileName.indexOf('.'));
		}
		File destFile = new File(destDir + File.separator + fileId + suffix);
		ThreadPoolExecutor exService = null;
		try {
			if (destFile.exists()) {
				FileDownLoadInfo fileDownLoadeInfo = new FileDownLoadInfo();
				fileDownLoadeInfo.setDestFile(destFile);
				fileDownLoadeInfo.setFileId(fileResponse.getFileInfo().getFileId());
				fileDownLoadeInfo.setFileMd5(fileResponse.getFileInfo().getFileMd5());
				fileDownLoadeInfo.setFileType(fileResponse.getFileInfo().getFileType());
				fileDownLoadeInfo.setFileSize(fileResponse.getFileInfo().getFileSize());
				fileDownLoadeInfo.setFileName(fileName);
				for (int i = 0; i < fileResponse.getFileInfo().getSplitSum(); i++) {
					FileSplitDownLoadTask.deleteFile(new File(destDir + File.separator + fileId + "_" + i + ".temp"), 0);
				}
				return fileDownLoadeInfo;
			}
			
			if(!downloadLock(destFile.getAbsolutePath(),destFile)){
				throw new SDKException(DealCode.FILE_DOWNLOADING.getCode(),"文件["+destFile.getAbsolutePath()+"]已经在下载中，请稍后再试！！");
			}
			int splitSum = fileResponse.getFileInfo().getSplitSum();
			CountDownLatch countDown = null;
			//在这里加一个限定条件，当分片数量大于START_CURRENT_THRESHHOLD的时候，我们认为文件下载的时间会比较长，这时采用多线程并发下载合理减少下载时间
			if (isParallel && splitSum > START_CURRENT_THRESHHOLD) {
				countDown = new CountDownLatch(splitSum);
				exService = new ThreadPoolExecutor(2, DOWNLOAD_THREADS, 120, TimeUnit.SECONDS,new LinkedBlockingQueue<Runnable>(splitSum));
			}else {
				isParallel = false;
			}
			byte[] downloadResultMap = new byte[splitSum];
			for (int i = 0; i < downloadResultMap.length; i++) {
				downloadResultMap[i] = (byte)0;
			}
			for (int i = 0; i < splitSum; i++) {
				FileSplitDownLoadTask task = new FileSplitDownLoadTask(destDir, fileResponse.getFileInfo(), i, seqNo,
						downloadResultMap, countDown, httpConfig, packScure, apiService,channelId);
				if (!isParallel) {
					task.run();
				} else {
					exService.submit(task);
				}
			}
			if (countDown != null) {
				countDown.await();
			}
			boolean splitFull = true;
			for (int i = 0; i < downloadResultMap.length; i++) {
				if (downloadResultMap[i] == (byte)0) {
					splitFull = false;
					break;
				}
			}
			if (splitFull) {
				File tempFile = new File(
						destDir + File.separator + fileId + suffix + UUID.randomUUID().toString().replaceAll("-", ""));
				BufferedOutputStream bou = null;
				BufferedInputStream bf = null;
				try {
					bou = new BufferedOutputStream(new FileOutputStream(tempFile));
					for (int i = 0; i < downloadResultMap.length; i++) {
						bf = new BufferedInputStream(new FileInputStream(destDir + File.separator + fileId + "_" + i + ".temp"));
						int len = -1;
						byte[] bytes = new byte[bf.available()];
						while ((len = bf.read(bytes)) != -1) {
							bou.write(bytes, 0, len);
							bou.flush();
						}
						bf.close();
					}
				} catch (Exception e) {
					if (bf != null) {
						try {
							bf.close();
						} catch (Exception e1) {}
					}
					throw new SDKException("文件" + fileId + "下载失败!!请重新下载");
				} finally {
					if (bou != null) {
						try {
							bou.close();
						} catch (Exception e) {}
					}
				}

				if (tempFile.exists() && Md5EncodeUtil.encode(tempFile).equals(fileResponse.getFileInfo().getFileMd5())
						&& FileSplitDownLoadTask.renameFile(tempFile, destFile, 0)) {
					FileDownLoadInfo fileDownLoadeInfo = new FileDownLoadInfo();
					fileDownLoadeInfo.setDestFile(destFile);
					fileDownLoadeInfo.setFileId(fileResponse.getFileInfo().getFileId());
					fileDownLoadeInfo.setFileMd5(fileResponse.getFileInfo().getFileMd5());
					fileDownLoadeInfo.setFileType(fileResponse.getFileInfo().getFileType());
					fileDownLoadeInfo.setFileSize((int) destFile.length());
					fileDownLoadeInfo.setFileName(fileName);
					for (int i = 0; i < downloadResultMap.length; i++) {
						FileSplitDownLoadTask.deleteFile(new File(destDir + File.separator + fileId + "_" + i + ".temp"), 0);
					}
					return fileDownLoadeInfo;
				}else{
					FileSplitDownLoadTask.deleteFile(tempFile, 0);
				}
			}
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException)e;
			} else {
				throw new SDKException(DealCode.SDK_ERROR.getCode(), "文件" + fileId + "下载失败!!请重新下载" + e.getMessage(), e);
			}
		} finally {
			if(destFile != null){
				downloadUnLock(destFile.getAbsolutePath(),destFile);
			}
			if (exService != null) {
				exService.shutdown();
			}
		}
		throw new SDKException("文件" + fileId + "下载失败!!请重新下载");
	}
	
	/**
	 * 加锁下载线程
	 * 
	 * @param filePath
	 * @param thread
	 * @return
	 */
	protected static boolean downloadLock(String filePath,File lockFile){
		if(downloadingFileCache.get(filePath) != null){
			return false;
		}
		synchronized(BaseSDK.class){
			try{
				if(downloadingFileCache.get(filePath) != null){
					return false;
				}else{
					downloadingFileCache.put(filePath, lockFile);
					return true;
				}
			}catch(Exception e){
				return false;
			}
		}
	}
	
	/**
	 * 解锁下载线程
	 * @param filePath
	 */
	protected static void downloadUnLock(String filePath,File unlockFile) {
		synchronized (BaseSDK.class) {
			try {
				if(downloadingFileCache.get(filePath) == unlockFile){
					downloadingFileCache.remove(filePath);
				}
			} catch (Exception e) {
			}
		}
	}
	
	/**
	 * 安全传输的服务端流
	 * 
	 * @param fileId
	 * @param channelId
	 * @param seqNo
	 * @return
	 * @throws SDKException
	 * @throws EmptyFileException
	 */
	protected StreamDownLoadInfo downloadStreamSecureTransfer(final String fileId, final String channelId, final String seqNo)throws SDKException, EmptyFileException {
		if (StringUtil.isEmpty(fileId)) {
			throw new SDKException("参数fileId不能为空!!");
		}
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("参数seqNo不能为空!!");
		}
		try {
			if (fileId.trim().contains(" ") || fileId.trim().contains("/")) {
				throw new SDKException("请使用ImgSDK.downloadSmallFileFromImageSys()方法进行下载!");
			} else {
				FileQueryRequest qryRequest = new FileQueryRequest();
				qryRequest.setChannelId(channelId);
				qryRequest.setFileId(fileId);
				qryRequest.setSeqNo(seqNo);
				FileQueryResponse fileResponse = this.apiService.fileQuery(qryRequest, this.httpConfig, packScure);
				if (!DealCode.SUCCESS.getCode().equals(fileResponse.getCode())) {
					throw new SDKException(fileResponse.getCode(), fileResponse.getMessage());
				}
				// 处理大小为0 的文件
				if (fileResponse.getFileInfo().getFileSize() == 0) {
					throw new EmptyFileException("文件" + fileResponse.getFileInfo().getFileId() + "为空文件,请处理!!");
				}
				final PipedOutputStream pipeOutputStream = new PipedOutputStream();
				PipedInputStream pipeInputStream = new PipedInputStream();
				pipeOutputStream.connect(pipeInputStream);
				StreamDownLoadInfo result = new StreamDownLoadInfo();
				result.setInputStream(pipeInputStream);
				result.setFileId(fileId);
				result.setFileName(fileResponse.getFileInfo().getFileName());
				result.setFileMd5(fileResponse.getFileInfo().getFileMd5());
				result.setFileType(fileResponse.getFileInfo().getFileType());
				result.setFileSize(fileResponse.getFileInfo().getFileSize());
				
				new Thread(new Runnable(){
					@Override
					public void run() {
						try{
							FileSyncDownloadRequest request = new FileSyncDownloadRequest();
							request.setChannelId(channelId);
							request.setFileId(fileId);
							request.setSeqNo(seqNo);
							request.setFileStreamEncrypt("true");
							FileSyncDownloadResponse response = apiService.fileSyncDownLoad(request, httpConfig,packScure);
							if (!DealCode.SUCCESS.getCode().equals(response.getCode())) {
								throw new SDKException(response.getCode(), response.getMessage());
							}
							StreamDownLoadInfo unEncrypted = new StreamDownLoadInfo();
							unEncrypted.setClient(response.getSyncDownLoadInfo().getClient());
							unEncrypted.setResp(response.getSyncDownLoadInfo().getResp());
							streamDownLoadInfo2Dest(unEncrypted, pipeOutputStream);
						}catch(Exception e){
							log.error("文件下载失败!!",e);
						}
					}
					
				}).start();
				return result;
			}
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException) e;
			}
			throw new SDKException(DealCode.SDK_ERROR.getCode(), e);
		} 
	}
	
	/**
	 * 针对info数据写入dest
	 * 
	 * 
	 * @param info
	 * @param dest
	 * @param packScure
	 * @throws SDKException
	 */
	@SuppressWarnings("deprecation")
	protected void streamDownLoadInfo2Dest(StreamDownLoadInfo info ,OutputStream dest) throws SDKException{
		try{
			if(this.packScure != null && Pattern.matches("^SM2SM4$", this.packScure.getEncryptType())){
				SM4Utils.decryptNetStream(info.getInputStream(), dest,  this.packScure.getUnEncryptKey());
			} else {
				int byteCount = 0;
				byte[] buffer = new byte[8 * 1024];
				while ((byteCount = info.read(buffer)) > 0) {
					dest.write(buffer, 0, byteCount);
					dest.flush();
				}
			}
		}catch(Exception e){
			throw new SDKException(DealCode.SDK_ERROR.getCode(),"下载文件失败！！",e);
		}finally{
			if(info != null){
				info.releaseResouces();
			}
			if(dest != null){
				try {
					dest.close();
				} catch (IOException e) {
				}
			}
		}
	}
}
