package com.zbank.file.sdk;

import java.io.File;
import java.util.List;

import com.zbank.file.bean.FileDownLoadInfo;
import com.zbank.file.bean.FileInfo;
import com.zbank.file.bean.StreamDownLoadInfo;
import com.zbank.file.bean.UploadInfo;
import com.zbank.file.common.http.config.HttpConfig;
import com.zbank.file.exception.EmptyFileException;
import com.zbank.file.exception.SDKException;
import com.zbank.file.sdk.download.sync.SplitIterator;
import com.zbank.file.secure.IPackSecure;

/**
 * 众邦分布式文件系统上传下载接口
 * @author zhanglulu
 *
 */
public interface IFileSDK {

	/**
	 * 文件上传，支持断点续传。该上传方式是将文件按照一定的大小分成若干片，上传至服务器。如果isParallel为true，将开启多线程上传功能。
	 * @param file 即将上传的文件
	 * @param channelId 渠道编号
	 * @param seqNo 流水号
	 * @param isParallel 是否采用并行上传，即：多线程上传分片。
	 * @return
	 * @throws SDKException
	 */
	public UploadInfo upload(File file,String channelId,String seqNo,boolean isParallel)throws SDKException;
	
	/**
	 * 文件字节数组上传，支持断点续传。该上传方式是将文件按照一定的大小分成若干片，上传至服务器。如果isParallel为true，将开启多线程上传功能。
	 * @param bytes 即将进行上传的文件全部字节数组
	 * @param fileName 即将上传的文件名称
	 * @param channelId 渠道编号
	 * @param seqNo 流水号
	 * @param isParallel 是否采用并行上传，即：多线程上传。
	 * @return
	 * @throws SDKException
	 */
	public UploadInfo upload(byte[] bytes,String fileName,String channelId,String seqNo,boolean isParallel)throws SDKException;
	
	/**
	 * 文件下载，支持断点续传。该方式是将文件的分片下载到本地磁盘之后，再合并成完整的文件，然后再做MD5校验保证文件的完整性!!<b>
	 * @param fileId 即将下载的文件唯一标识
	 * @param channelId 渠道编号
	 * @param destDir 文件下载之后的保存目录
	 * @param seqNo 流水号
	 * @param allowMakeDir 是否允许在destDir下创建以fileId为名称的目录，建议设置为true。
	 * @param isParallel 是否采用并发下载，即：多线程下载分片。
	 * @return
	 * @throws SDKException
	 * @throws EmptyFileException 当所下载的文件大小为0(即当初所上传的文件为空文件)时，将抛出该异常。
	 */
	public FileDownLoadInfo downloadFile(String fileId,String channelId,String destDir,String seqNo,boolean allowMakeDir,boolean isParallel)throws SDKException,EmptyFileException;
	
	/**
	 * 文件流下载，该下载方式是在服务端将分片构成网络流，并返回给调用者。基本用法如下
	 * <pre>{@code
	 * SDK sdk = SDK.build(....);
	 * StreamDownLoadInfo downloadInfo = null;
	 * try{
	 * 	downloadInfo = sdk.downloadStream(....);
	 * 	byte[] bytes = new byte[...]
	 * 	//以下为循环读取输入流中的数据，直到读取结束为止
	 *	int n = downloadInfo.read(bytes);
	 * 	File destFile = ....
	 * 	while(n != -1){
	 *	   //TODO 你自己的代码，对读取的数据进行处理
	 *	   n = downloadInfo.read(bytes);
	 * 	}
	 * 	String md5 = Md5EncodeUtil.encode(destFile);
	 * 	if(md5.equals(downloadInfo.getFileMD5())){
	 *  	//文件下载成功
	 *  	//TODO 下载成功后的处理逻辑
	 * 	}else{
	 *  	//文件下载失败
	 *  	//TODO 下载失败后的处理逻辑
	 * 	} 
	 * }catch(Exception e){
	 * 	//TODO 这里是你的异常处理逻辑....
	 * }finally{
	 *	<b>//完成之后一定要调用StreamDownLoadInfo.releaseResouces();方法释放资源!!!!</b>
	 * 	if(downloadInfo != null){
	 *	   downloadInfo.releaseResouces();
	 * 	}
	 * }
	 
	 * </pre>
	 * <b>注意：在取完之后最好做MD5校验，保证文件的正确性!!</br></br>
	 * @param fileId 即将下载的文件唯一标识
	 * @param channelId 渠道编号
	 * @param seqNo 流水号
	 * @return
	 * @throws SDKException
	 * @throws EmptyFileException 当所下载的文件大小为0(即当初所上传的文件为空文件)时，将抛出该异常。
	 */
	public StreamDownLoadInfo downloadStream(String fileId,String channelId,String seqNo)throws SDKException,EmptyFileException;
	
	/**
	 * 文件分片迭代器下载，构造一个文件分片的迭代器，调用者可根据需要处理。基本用法如下：
	 * <pre>{@code
	 * SDK sdk = SDK.build(....);
	 * SplitIterator ite = sdk.genSplitIterator(....);
	 * File destFile = ....
	 * while(ite.hasNext()){
	 *  byte[] bytes ;
	 *  try{
	 *    bytes = ite.next();//获取下一文件分片
	 *  }catch(SDKException e){
	 *    bytes = ite.retry();//如果获取失败，可以重新获取
	 *  }
	 *  //将bytes写进destFile
	 * }
	 * String md5 = Md5EncodeUtil.encode(destFile);
	 * if(md5.equals(ite.getFileInfo().getFileMd5())){
	 *  //文件下载成功
	 *  //TODO 下载成功后的处理逻辑
	 * }else{
	 *  //文件下载失败
	 *  //TODO 下载失败后的处理逻辑
	 * }
	 * </pre>
	 * <b>注意：在取完之后最好做MD5校验，保证文件的正确性!!</br></br>
	 * 
	 * @param fileId 文件唯一标识
	 * @param channelId 渠道编号
	 * @param seqNo 流水号
	 * @return
	 */
	public SplitIterator genSplitIterator(String fileId,String channelId,String seqNo) throws SDKException,EmptyFileException;
	
	/**
	 * 文件查询接口，用于查询指定时间段内的指定文件名称的文件列表，按照创建时间降序排列。<b>
	 * @param channelId     渠道编号
	 * @param fileName  	要查询的文件名称
	 * @param slotKey 	             要查询的文件所属分区值（众邦银行提供）
	 * @param beginDate     查询的开始日期  格式为yyyy-MM-dd HH:mm:ss
	 * @param endDate       查询的截止日期  格式为yyyy-MM-dd HH:mm:ss
	 * @param pageNo     	查询第几页的数据，该查询是分页查询，每页20条数据。
	 * @param seqNo     	三方流水号
	 * @return
	 * @throws SDKException
	 */
	public List<FileInfo> queryFileList(String channelId,String fileName,String slotKey,String beginDate,String endDate,int pageNo,String seqNo)throws SDKException;

	
	/**
	 * 配置SDK的http选项
	 * 
	 * @param httpConfig
	 * @return
	 */
	public IFileSDK config(HttpConfig httpConfig) ;

	/**
	 * 配置SDK的加密方案实例
	 * 
	 * @param packScure
	 * @return
	 */
	public IFileSDK config(IPackSecure packScure) ;

}
