package com.zbank.file.sdk;

import java.io.File;
import java.io.FileInputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zbank.file.api.APIServiceImpl;
import com.zbank.file.bean.ImageDownloadResponse;
import com.zbank.file.bean.ImageInfo;
import com.zbank.file.bean.StreamDownLoadInfo;
import com.zbank.file.bean.UploadLimitConfig;
import com.zbank.file.common.http.config.HttpConfig;
import com.zbank.file.common.utils.Md5EncodeUtil;
import com.zbank.file.common.utils.StringUtil;
import com.zbank.file.constants.DealCode;
import com.zbank.file.exception.SDKException;
import com.zbank.file.secure.IPackSecure;

/**
 * 众邦影像平台文件上传下载接口实现
 * 
 * @author zhanglulu
 *
 */
public class ImgSDK extends BaseSDK implements IImgSDK{

	private static final Logger log = LoggerFactory.getLogger(ImgSDK.class);
	
	protected ImgSDK(String url) {
		this.apiService = new APIServiceImpl(url);
	}

	public static ImgSDK build(String url) {
		return new ImgSDK(url);
	}

	public ImgSDK config(HttpConfig httpConfig) {
		this.httpConfig = httpConfig;
		return this;
	}

	public ImgSDK config(IPackSecure packScure) {
		this.packScure = packScure;
		return this;
	}

	@Override
	public Map<String, Object> uploadBigFile2ImageSys(String channelId, File file,HashMap<String, Object> uploadParam) throws SDKException {
		if ("".equals(channelId) || null == channelId) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if(file.getName().indexOf(".") < 0){
			throw new SDKException("文件名称必须带有后缀!!当前文件名为："+file.getName());
		}
		FileInputStream fileIn = null;
		try {
			fileIn = new FileInputStream(file);
			long offset = 0;
			byte[] ts = new byte[this.apiService.queryFileUploadLimitConfig(channelId, this.httpConfig, this.packScure).getFileSplitSizeLimit()/2];
			uploadParam.put("channelId", channelId);
			uploadParam.put("tranCode", "2009");
			uploadParam.put("totalFileSize", String.valueOf(fileIn.available()));
			uploadParam.put("fileType", file.getName().substring(file.getName().lastIndexOf(".")+1));
			uploadParam.put("fileName", file.getName());
			for (int len = 0; (len = fileIn.read(ts)) != -1;) {
				byte[] fileBytes;
				if (len < ts.length) {
					fileBytes = Arrays.copyOfRange(ts, 0, len);
					uploadParam.put("splitSize", String.valueOf(len));
				} else {
					fileBytes = ts;
					uploadParam.put("splitSize", String.valueOf(ts.length));
				}
				uploadParam.put("offset", String.valueOf(offset));
				for (int i = 0; i < 4; i++) {
					try {
						Map<String, Object> mp = this.apiService.upload2009Notice(uploadParam, fileBytes, this.httpConfig,this.packScure);
						if (DealCode.SUCCESS.getCode().equals(mp.get("code")) && mp.get("result")!= null && "00000000000".equals(((Map<String,Object>)mp.get("result")).get("tradeResult"))) {
							break;
						}
					} catch (SDKException e) {
						Thread.sleep(1000L);
						log.error("上传文件" + file.getAbsolutePath() + "分片失败，进行第" + (i + 1) + "次重试!!", e);
					}
				}
				offset += len;
			}
			
			HashMap<String, Object> maphead = new HashMap<String, Object>();
			maphead.put("channelId", channelId);
			maphead.put("tranCode", "2109");
			maphead.put("busiNo", uploadParam.get("busiNo"));
			maphead.put("operId", uploadParam.get("operId"));
			maphead.put("branchNo", uploadParam.get("branchNo"));
			maphead.put("fileType", uploadParam.get("fileType"));
			maphead.put("billType", uploadParam.get("billType"));
			maphead.put("totalFileSize", uploadParam.get("totalFileSize"));
			maphead.put("fileName", uploadParam.get("fileName"));
			return this.apiService.query2109Rst(maphead, this.httpConfig, this.packScure);
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException) e;
			}
			throw new SDKException(DealCode.SDK_ERROR.getCode(), e);
		} finally {
			try {
				if (fileIn != null) {
					fileIn.close();
				}
			} catch (Exception e) {
			}
		}
	}

	@Override
	public Map<String, Object> uploadBigFile2ImageSys(String channelId, byte[] fileData, String fileName, HashMap<String, Object> uploadParam) throws SDKException {
		if ("".equals(channelId) || null == channelId) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if ("".equals(fileName) || null == fileName) {
			throw new SDKException("参数fileName不能为空!!");
		}
		if(fileName.indexOf(".") < 0){
			throw new SDKException("文件名称必须带有后缀!!当前文件名为："+fileName);
		}
		try {
			int splitSize = this.apiService.queryFileUploadLimitConfig(channelId, this.httpConfig, this.packScure).getFileSplitSizeLimit()/2;
			uploadParam.put("channelId", channelId);
			uploadParam.put("tranCode", "2009");
			uploadParam.put("totalFileSize", String.valueOf(fileData.length));
			uploadParam.put("fileType", fileName.substring(fileName.lastIndexOf(".")+1));
			uploadParam.put("fileName", fileName);
			int splitCnt = fileData.length / splitSize;
			if(fileData.length  % splitSize !=0){
				splitCnt += 1;
			}
			for (int k = 0; k < splitCnt; k++) {
				int offset = splitSize * k;
				uploadParam.put("offset", String.valueOf(offset));
				byte[] fileBytes;
				if (fileData.length - offset >= splitSize) {
					fileBytes = Arrays.copyOfRange(fileData, offset, offset+splitSize);
					uploadParam.put("splitSize", String.valueOf(splitSize));
				} else {
					fileBytes = Arrays.copyOfRange(fileData, offset, fileData.length);
					uploadParam.put("splitSize", String.valueOf(fileData.length));
				}
				
				for (int i = 0; i < 4; i++) {
					try {
						Map<String, Object> mp = this.apiService.upload2009Notice(uploadParam, fileBytes, this.httpConfig,this.packScure);
						if (DealCode.SUCCESS.getCode().equals(mp.get("code")) && mp.get("result")!= null && "00000000000".equals(((Map<String,Object>)mp.get("result")).get("tradeResult"))) {
							break;
						}
					} catch (SDKException e) {
						Thread.sleep(1000L);
						log.error("上传文件" + fileName + "分片失败，进行第" + (i + 1) + "次重试!!", e);
					}
				}
				
			}
			
			HashMap<String, Object> maphead = new HashMap<String, Object>();
			maphead.put("channelId", channelId);
			maphead.put("tranCode", "2109");
			maphead.put("busiNo", uploadParam.get("busiNo"));
			maphead.put("operId", uploadParam.get("operId"));
			maphead.put("branchNo", uploadParam.get("branchNo"));
			maphead.put("fileType", uploadParam.get("fileType"));
			maphead.put("billType", uploadParam.get("billType"));
			maphead.put("totalFileSize", uploadParam.get("totalFileSize"));
			maphead.put("fileName", uploadParam.get("fileName"));
			return this.apiService.query2109Rst(maphead, this.httpConfig, this.packScure);
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException) e;
			}
			throw new SDKException(DealCode.SDK_ERROR.getCode(), e);
		}
	
	}

	@Override
	public StreamDownLoadInfo downloadBigFileFromImageSys(final String channelId, final String fileUrl) throws SDKException {
		if ("".equals(channelId) || null == channelId) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if ("".equals(fileUrl) || null == fileUrl) {
			throw new SDKException("参数fileUrl不能为空!!");
		}
		try {
			final HashMap<String, Object> maphead = new HashMap<String, Object>();
			maphead.put("channelId", channelId);
			maphead.put("fileUrl", URLEncoder.encode(fileUrl, "utf-8"));
			if(Pattern.matches("^SM2SM4$", this.packScure.getEncryptType())){
				maphead.put("fileStreamEncrypt", "true");
				final PipedOutputStream pipeOutputStream = new PipedOutputStream();
				PipedInputStream pipeInputStream = new PipedInputStream();
				pipeOutputStream.connect(pipeInputStream);
				final StreamDownLoadInfo result = new StreamDownLoadInfo();
				result.setInputStream(pipeInputStream);
				new Thread(new Runnable(){
					@Override
					public void run() {
						try{
							StreamDownLoadInfo downloadInfo =  apiService.downloadBigImage(maphead, httpConfig, packScure);
							result.setFileId(downloadInfo.getFileId());
							result.setFileName(downloadInfo.getFileName());
							result.setFileMd5(downloadInfo.getFileMd5());
							result.setFileType(downloadInfo.getFileType());
							result.setFileSize(downloadInfo.getFileSize());
							streamDownLoadInfo2Dest(downloadInfo, pipeOutputStream);
						}catch(Exception e){
							log.error("文件下载失败!!",e);
						}
					}
					
				}).start();
				return result;
			}else{
				return apiService.downloadBigImage(maphead, httpConfig, packScure);
			}
		} catch (Exception e) {
			throw new SDKException(DealCode.SDK_ERROR.getCode(), e);
		}
	}

	@Override
	public ImageInfo downloadSmallFileFromImageSys(String channelId, String fileHandle) throws SDKException {
		ImageDownloadResponse response = downloadImage(fileHandle, channelId, UUID.randomUUID().toString());
		if(!Md5EncodeUtil.encode(response.getImageInfo().getBytes()).equals(response.getImageInfo().getFileMd5())){
			throw new SDKException(DealCode.SDK_ERROR.getCode(),"文件的Md5值不匹配！！");
		}
		return response.getImageInfo();
	}

	@Override
	public String uploadSmallFile2ImageSys(String channelId, String fileType, String busiNo, String billType,
			byte[] fileBytes, Map<String, String> extInfo) throws SDKException {
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if (StringUtil.isEmpty(fileType)) {
			throw new SDKException("参数fileType不能为空!!");
		}
		if (StringUtil.isEmpty(busiNo)) {
			throw new SDKException("参数busiNo不能为空!!");
		}
		if (StringUtil.isEmpty(billType)) {
			throw new SDKException("参数billType不能为空!!");
		}
		if (fileBytes == null) {
			throw new SDKException("参数fileBytes不能为空!!");
		}

		UploadLimitConfig config = this.apiService.queryFileUploadLimitConfig(channelId, this.httpConfig,
				this.packScure);
		if (config.getFileSplitSizeLimit2001() < fileBytes.length) {
			throw new SDKException(DealCode.SDK_ERROR.getCode(),
					"上传的文件大小超过上限" + config.getFileSplitSizeLimit2001() + "字节！！");
		}

		HashMap<String, Object> maphead = new HashMap<String, Object>();
		maphead.put("channelId", channelId);
		maphead.put("fileType", fileType);
		maphead.put("busiNo", busiNo);
		maphead.put("billType", billType);
		maphead.put("fileSize", String.valueOf(fileBytes.length));
		if (extInfo != null) {
			maphead.putAll(extInfo);
		}

		Map<String, Object> rm = this.apiService.upload2001(maphead, fileBytes, this.httpConfig, this.packScure);
		Map<String, Object> map = ((Map<String, Object>) rm.get("result"));
		if (DealCode.SUCCESS.getCode().equals(rm.get("code")) && rm.get("result") != null && "00000000000".equals(map.get("tradeResult"))) {
			return map.get("fileId") + " " + busiNo;
		} else {
			throw new SDKException(rm.get("code").toString(), rm.get("msg").toString());
		}
	}
}
