package com.zbank.file.sdk.download.async.impl1;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.zbank.file.api.APIService;
import com.zbank.file.bean.AsyncDownLoadInfo;
import com.zbank.file.bean.FileAsyncDownLoadRequest;
import com.zbank.file.bean.FileQueryResponse;
import com.zbank.file.bean.SplitDownLoadRequest;
import com.zbank.file.bean.SplitDownLoadResponse;
import com.zbank.file.common.http.config.HttpConfig;
import com.zbank.file.common.utils.JsonUtil;
import com.zbank.file.constants.DealCode;
import com.zbank.file.exception.EmptyFileException;
import com.zbank.file.exception.SDKException;
import com.zbank.file.sdk.download.async.IAsyncDownLoad;
import com.zbank.file.secure.IPackSecure;

/**
 * 
 * @author zhanglulu
 *
 */
public class DownLoadWithMultipleTask implements IAsyncDownLoad {

	static final Logger log = LoggerFactory.getLogger(DownLoadWithMultipleTask.class);
	
	private APIService apiService;
	private int concurrency;
	private IPackSecure packScure;
	private HttpConfig httpConfig;

	public DownLoadWithMultipleTask(APIService apiService, int concurrency, HttpConfig httpConfig,
			IPackSecure packScure) {
		this.apiService = apiService;
		this.concurrency = concurrency;
		this.packScure = packScure;
		this.httpConfig = httpConfig;
	}

	public AsyncDownLoadInfo download(FileAsyncDownLoadRequest request) throws SDKException, EmptyFileException {

		FileQueryResponse fileInfo = this.apiService.fileQuery(request, httpConfig, packScure);
		if (!DealCode.SUCCESS.getCode().equals(fileInfo.getCode())) {
			String reqJson = null;
			try {
				reqJson = JsonUtil.MAPPER.writeValueAsString(request);
			} catch (JsonProcessingException e) {
				log.error("序列化json异常{}", e.getMessage());
			}
			throw new SDKException("根据参数" + reqJson + "查询文件信息失败，无法进行下载操作", new Throwable());
		}
		//处理大小为0 的文件
		if(fileInfo.getFileInfo().getFileSize() == 0){
			throw new EmptyFileException("文件"+fileInfo.getFileInfo().getFileId()+"为空文件,请处理!!");
		}
		//在这里需要判断concurrency与分片的数量的大小，如果分片数量小于concurrency，则将concurrency设置为分片数量，尽量减少线程创建数量
		if(fileInfo.getFileInfo().getSplitSum() < this.concurrency) {
			this.concurrency = fileInfo.getFileInfo().getSplitSum();
		}
		CachedFileInputStream inputStream = new CachedFileInputStream(concurrency, fileInfo.getFileInfo());
		for (int i = 0; i < inputStream.getCacheSize(); i++) {
			new Thread(
					new WriteTask(inputStream.getSpecifiedElement(i), apiService, request, httpConfig, packScure,inputStream)).start();
		}
		AsyncDownLoadInfo response = new AsyncDownLoadInfo();
		response.setFileInfo(fileInfo.getFileInfo());
		response.setInputStream(inputStream);
		return response;

	}

	static class WriteTask implements Runnable {

		SplitInfo bean;
		APIService apiService;
		FileAsyncDownLoadRequest info;
		IPackSecure packScure;
		HttpConfig httpConfig;
		CachedFileInputStream inputStream;
		
		WriteTask(SplitInfo bean, APIService apiService, FileAsyncDownLoadRequest info, HttpConfig httpConfig,
				IPackSecure packScure,CachedFileInputStream inputStream) {
			this.bean = bean;
			this.apiService = apiService;
			this.info = info;
			this.packScure = packScure;
			this.httpConfig = httpConfig;
			this.inputStream = inputStream;
		}

		@Override
		public void run() {
			for (int i = bean.initIndex; i < bean.maxIndex; i += bean.step) {
				synchronized (bean.lock) {
					try {
						if (!inputStream.isOver) {
							while (!bean.writable) {
								try {
									log.info("----任务被阻塞,initIndex={},readable={},writable={}",bean.initIndex,bean.readable,bean.writable);
									bean.lock.wait();
								} catch (Exception e) {
									log.error(e.getMessage());
								}
							}
							SplitDownLoadRequest request = new SplitDownLoadRequest();
							request.setChannelId(info.getChannelId());
							request.setFileId(info.getFileId());
							request.setCurrSplitNo(String.valueOf(i));
							SplitDownLoadResponse response = apiService.splitDownLoad(request, httpConfig, packScure);
							if (response != null && DealCode.SUCCESS.getCode().equals(response.getCode())) {
								bean.dataPackage = response.getSplitDownLoadInfo();
							}else {
								inputStream.isOver = true;
							}
						}
					} catch (Exception e) {
						log.error(e.getMessage(),e);
						inputStream.isOver = true;
						bean.dataPackage = null;
					} finally {
						bean.readable = true;
						bean.writable = false;
						try {
							bean.lock.notifyAll();
						} catch (Exception e) {
							log.error(e.getMessage());
						}
					}
				}
			}
		}

	}

}
