package com.zbank.file.sdk.download.async.impl2;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.zbank.file.api.APIService;
import com.zbank.file.bean.AsyncDownLoadInfo;
import com.zbank.file.bean.FileAsyncDownLoadRequest;
import com.zbank.file.bean.FileQueryResponse;
import com.zbank.file.bean.SplitDownLoadInfo;
import com.zbank.file.bean.SplitDownLoadRequest;
import com.zbank.file.bean.SplitDownLoadResponse;
import com.zbank.file.common.http.config.HttpConfig;
import com.zbank.file.common.utils.JsonUtil;
import com.zbank.file.constants.DealCode;
import com.zbank.file.exception.EmptyFileException;
import com.zbank.file.exception.SDKException;
import com.zbank.file.sdk.download.async.IAsyncDownLoad;
import com.zbank.file.secure.IPackSecure;

/**
 * 
 * @author zhanglulu
 *
 */
public class DownLoadStreamWithPictureBuffer implements IAsyncDownLoad {

	static final Logger log = LoggerFactory.getLogger(DownLoadStreamWithPictureBuffer.class);
	private APIService apiService;
	private int concurrency;
	private IPackSecure packScure;
	private HttpConfig httpConfig;

	public DownLoadStreamWithPictureBuffer(APIService apiService, int concurrency, HttpConfig httpConfig,
			IPackSecure packScure) {
		this.apiService = apiService;
		this.concurrency = concurrency;
		this.packScure = packScure;
		this.httpConfig = httpConfig;
	}

	@Override
	public AsyncDownLoadInfo download(final FileAsyncDownLoadRequest request) throws SDKException,EmptyFileException{
		final FileQueryResponse fileInfo = this.apiService.fileQuery(request, httpConfig, packScure);
		if (!DealCode.SUCCESS.getCode().equals(fileInfo.getCode())) {
			String reqJson = null;
			try {
				reqJson = JsonUtil.MAPPER.writeValueAsString(request);
			} catch (JsonProcessingException e) {
				log.error("序列化json异常{}", e.getMessage());
			}
			throw new SDKException("根据参数" + reqJson + "查询文件信息失败，无法进行下载操作", new Throwable());
		}
		//处理大小为0 的文件
		if(fileInfo.getFileInfo().getFileSize() == 0){
			throw new EmptyFileException("文件"+fileInfo.getFileInfo().getFileId()+"为空文件,请处理!!");
		}
		final Status status = new Status();
		status.isOver = false;
		final Permit permit = new Permit(concurrency, concurrency);
		final CachedFileInputStream inputStream = new CachedFileInputStream(fileInfo.getFileInfo(), permit, status);
		new Thread(new Runnable() {

			@Override
			public void run() {

				for (int i = 0; i < fileInfo.getFileInfo().getSplitSum(); i++) {
					permit.decrease();
					WriteTask writeTask = new WriteTask(i, apiService, request, httpConfig, packScure);
					new Thread(new ExecuteTask(writeTask,inputStream.getCache(),status)).start();
				}
			}
		}).start();
		AsyncDownLoadInfo response = new AsyncDownLoadInfo();
		response.setFileInfo(fileInfo.getFileInfo());
		response.setInputStream(inputStream);
		return response;
	}
	
	static class ExecuteTask implements Runnable{

		WriteTask writeTask;
		SplitInfo[] buffer;
		Status status;
		
		ExecuteTask(WriteTask writeTask,SplitInfo[] buffer,Status status){
			this.buffer = buffer;
			this.status = status;
			this.writeTask = writeTask;
		}
		@Override
		public void run() {
			writeTask.write(buffer, status);
		}
		
	}

	static class WriteTask {

		int currNo;
		APIService apiService;
		FileAsyncDownLoadRequest info;
		IPackSecure packScure;
		HttpConfig httpConfig;

		WriteTask(int currNo, APIService apiService, FileAsyncDownLoadRequest info, HttpConfig httpConfig,
				IPackSecure packScure) {
			this.currNo = currNo;
			this.apiService = apiService;
			this.info = info;
			this.packScure = packScure;
			this.httpConfig = httpConfig;
		}

		void write(SplitInfo[] buffer, Status status) {
			synchronized (buffer[currNo].lock) {
				try {
					SplitDownLoadResponse response = null;
					if (!status.isOver) {
						// 向服务端发送请求获取分片信息。。。。。。。
						SplitDownLoadRequest request = new SplitDownLoadRequest();
						request.setChannelId(info.getChannelId());
						request.setFileId(info.getFileId());
						request.setCurrSplitNo(String.valueOf(currNo));
						response = apiService.splitDownLoad(request, httpConfig, this.packScure);
					}
					if (response != null && DealCode.SUCCESS.getCode().equals(response.getCode())) {
						buffer[currNo].dataPackage = response.getSplitDownLoadInfo();
					} else {
						SplitDownLoadInfo dataPackage = new SplitDownLoadInfo();
						buffer[currNo].dataPackage = dataPackage;
						status.isOver = true;
					}
				} catch (Exception e) {
					log.error(e.getMessage(), e);
					SplitDownLoadInfo dataPackage = new SplitDownLoadInfo();
					buffer[currNo].dataPackage = dataPackage;
					status.isOver = true;
				} finally {
					try {
						buffer[currNo].lock.notifyAll();
					} catch (Exception e) {
						log.error(e.getMessage());
					}
				}
			}
		}
	}

}
