package com.zbank.file.secure;

import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zbank.file.common.utils.JsonUtil;
import com.zbank.file.common.utils.SM2Util;
import com.zbank.file.common.utils.SM4Utils;
import com.zbank.file.common.utils.Utils;
import com.zbank.file.constants.DealCode;
import com.zbank.file.exception.SDKException;

public class SM2SM4PackSecure extends AbstractPackScure{
	
	private static final Logger log = LoggerFactory.getLogger(SM2SM4PackSecure.class);
	/**
	 * SM2加密用的加密密钥
	 */
	private String sm2KeyString;
	/**
	 * 校验和字符串
	 */
	private String cksStr;
	
	/**
	 * SM4密钥
	 */
	private byte[] sm4Key;
	
	/***
	 * sm4向量
	 */
	private byte[] iv;
	
	
	public SM2SM4PackSecure(String encryptKeyString, String cksStr) {
		this.sm2KeyString = encryptKeyString;
		try {
			this.sm4Key = SM4Utils.getRawKey(Utils.genRandomKey(32));
			this.iv = Utils.hexToByte(UUID.randomUUID().toString().replaceAll("-", ""));
			this.cksStr = Base64.encodeBase64String(SM4Utils.sm4Encrypt(cksStr.getBytes(Charset.forName("UTF-8")), this.sm4Key, this.iv));
		} catch (Exception e) {
			log.error("构造加密器失败", e);
			throw new RuntimeException(e);
		}
	}

	@Override
	public String getEncryptType() {
		// TODO Auto-generated method stub
		return "SM2SM4";
	}

	@Override
	public String encryptAndSign(Map<String, Object> param) throws SDKException {
		try {
			String json = JsonUtil.MAPPER.writeValueAsString(param);
			log.info("加密前报文：{}", json);
			String encrypt = Utils.byteToHex(SM4Utils.sm4Encrypt(json.getBytes(Charset.forName("UTF-8")), this.sm4Key, this.iv));
			log.info("加密后报文：{}", encrypt);
			return encrypt;
		} catch (Exception e) {
			log.error("报文AES加密异常", e);
			throw new SDKException(e.getMessage(), e.getCause());
		}
	}

	@Override
	public Map<String, Object> decryptAndSignCheck(String pack) throws SDKException {
		log.info("解密前报文：{}", pack);
		try {
			// 解密
			byte[] outputData = SM4Utils.sm4Decrypt(Utils.hexToByte(pack), this.sm4Key, this.iv);
			String decrypt = new String(outputData,"UTF-8");
			log.info("解密后报文：{}", decrypt);
			return JsonUtil.MAPPER.readValue(decrypt, HashMap.class);
		} catch (Exception e) {
			log.error("报文AES解密异常", e);
			throw new SDKException(e.getMessage(), e.getCause());
		}
	}

	@Override
	public String getKey() throws SDKException{
		try{
			return SM2Util.encrypt(Utils.byteToHex(this.sm4Key), this.sm2KeyString);
		}catch(Exception e){
			throw new SDKException(DealCode.SDK_ERROR.getCode(),e);
		}
		
		
	}

	@Override
	public String getChkStr() throws SDKException{
		return this.cksStr;
	}

	@Override
	public String getIV() throws SDKException {
		try {
			return SM2Util.encrypt(Utils.byteToHex(this.iv), this.sm2KeyString);
		} catch (Exception e) {
			throw new SDKException(DealCode.SDK_ERROR.getCode(), e);
		}

	}
	
	@Override
	public byte[] getUnEncryptKey() throws SDKException{
		// TODO Auto-generated method stub
		return this.sm4Key;
	}

}
