package com.zbank.file.api;

import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;
import com.zbank.file.bean.FileInfo;
import com.zbank.file.bean.FileQueryRequest;
import com.zbank.file.bean.FileQueryResponse;
import com.zbank.file.bean.FileSyncDownloadRequest;
import com.zbank.file.bean.FileSyncDownloadResponse;
import com.zbank.file.bean.FileUploadConfig;
import com.zbank.file.bean.ImageDownloadRequest;
import com.zbank.file.bean.ImageDownloadResponse;
import com.zbank.file.bean.ImageInfo;
import com.zbank.file.bean.PreUploadRequest;
import com.zbank.file.bean.PreUploadResponse;
import com.zbank.file.bean.SplitDownLoadInfo;
import com.zbank.file.bean.SplitDownLoadRequest;
import com.zbank.file.bean.SplitDownLoadResponse;
import com.zbank.file.bean.StreamDownLoadInfo;
import com.zbank.file.bean.SyncDownLoadInfo;
import com.zbank.file.bean.UploadLimitConfig;
import com.zbank.file.bean.UploadRequest;
import com.zbank.file.bean.UploadResponse;
import com.zbank.file.common.http.HttpClientUtil;
import com.zbank.file.common.http.bean.BinaryDataRequest;
import com.zbank.file.common.http.bean.BinaryDataResponse;
import com.zbank.file.common.http.bean.Request;
import com.zbank.file.common.http.bean.Response;
import com.zbank.file.common.http.config.HttpConfig;
import com.zbank.file.common.utils.JsonUtil;
import com.zbank.file.common.utils.StringUtil;
import com.zbank.file.constants.DealCode;
import com.zbank.file.exception.SDKException;
import com.zbank.file.secure.IPackSecure;

/**
 * @author daiwj
 * @date 2021/04/27
 * @description: API层实现：服务端对应接口的调用
 */
public class APIServiceImpl extends AbstractAPIServiceImpl{
	private static final Logger log = LoggerFactory.getLogger(APIServiceImpl.class);
	/**
	 * 接口地址url:域名/ip+port
	 */
	private String serverUrl;

	public APIServiceImpl(String serverUrl) {
		this.serverUrl = serverUrl;
	}

	@Override
	public PreUploadResponse preUpload(PreUploadRequest request, HttpConfig config, IPackSecure packSecure) throws SDKException {
		Request req = new Request();
		//请求参数放在http header中进行传输
		req.setHeaderData(this.convertToHeader(request, packSecure));

		req.setServiceId(ServiceEnum.PRE_FILE_UPLOAD.getServiceId());
		req.setUrl(serverUrl + ServiceEnum.PRE_FILE_UPLOAD.getUri());
		log.info("预上传接口请求数据：url:{}，Headers:{}， Entity:{}", req.getUrl(), req.getHeaderData(), req.getEntityData());
		Response response = HttpClientUtil.post(req, config);

		PreUploadResponse preResponse = new PreUploadResponse();
		preResponse.setCode(response.getCode());
		preResponse.setMessage(response.getMessage());

		Object result = response.getResult();
		if(result != null && !StringUtil.isEmpty(result.toString())){
			if (packSecure != null){
				result = packSecure.decryptAndSignCheck(result.toString());
			}
			FileUploadConfig uploadConfig = JsonUtil.MAPPER.convertValue(result, FileUploadConfig.class);
			preResponse.setUploadConfig(uploadConfig);
		}
		log.info("预上传接口获取响应结果成功。");
		return preResponse;
	}

	@Override
	public UploadResponse splitUpload(UploadRequest request, HttpConfig config, IPackSecure packSecure) throws SDKException {
		BinaryDataRequest binaryReq = new BinaryDataRequest();
		byte[] fileBytes = request.getFileBytes();
		binaryReq.setBinaryData(this.encryptFileBytes(fileBytes, packSecure));
		//将文件字节置空，避免被加密
		request.setFileBytes(null);
		binaryReq.setHeaderData(this.convertToHeader(request, packSecure));
		//回置文件字节，因SDK有重试机制
		request.setFileBytes(fileBytes);
		binaryReq.setServiceId(ServiceEnum.FILE_UPLOAD.getServiceId());
		binaryReq.setUrl(serverUrl + ServiceEnum.FILE_UPLOAD.getUri());
		log.info("文件上传接口请求数据：url:{}，Headers:{}, binaryData.length:{}", binaryReq.getUrl(),
				binaryReq.getHeaderData(), binaryReq.getBinaryData().length);

		Response response = HttpClientUtil.uploadBinaryData(binaryReq, config);

		UploadResponse uploadResponse = new UploadResponse();
		uploadResponse.setCode(response.getCode());
		uploadResponse.setMessage(response.getMessage());

		//明文这里是一个map，密文是字符串
		Object result = response.getResult();
		if(result != null && !StringUtil.isEmpty(result.toString())){
			if (packSecure != null){
				result = packSecure.decryptAndSignCheck(result.toString());
			}
			Map<String, Object> resultMap = (Map<String, Object>) result;
			uploadResponse.setFileId(resultMap.get("fileId") == null ? "" : resultMap.get("fileId").toString());
			uploadResponse.setExtraData(resultMap.get("extraData") == null ? "" : resultMap.get("extraData").toString());
		}
		log.info("文件上传接口获取响应结果成功。");
		return uploadResponse;
	}

	@Override
	public FileQueryResponse fileQuery(FileQueryRequest request, HttpConfig config, IPackSecure packSecure) throws SDKException {
		Request req = new Request();
		//请求参数放在http header中进行传输
		req.setHeaderData(this.convertToHeader(request, packSecure));

		req.setServiceId(ServiceEnum.FILE_INFO_QUERY.getServiceId());
		req.setUrl(serverUrl + ServiceEnum.FILE_INFO_QUERY.getUri());
		log.info("文件查询接口请求数据：url:{}，Headers:{}， Entity:{}", req.getUrl(), req.getHeaderData(), req.getEntityData());
		Response response = HttpClientUtil.post(req, config);

		FileQueryResponse fileQueryResponse = new FileQueryResponse();
		fileQueryResponse.setCode(response.getCode());
		fileQueryResponse.setMessage(response.getMessage());
		//获取响应报文result节点，明文时为map，密文时为字符串
		Object result = response.getResult();
		if(result != null && !StringUtil.isEmpty(result.toString())){
			if (packSecure != null){
				result = packSecure.decryptAndSignCheck(result.toString());
			}
			FileInfo fileInfo = JsonUtil.MAPPER.convertValue(result, FileInfo.class);
			fileQueryResponse.setFileInfo(fileInfo);
		}
		log.info("文件查询接口获取响应结果成功。");
		return fileQueryResponse;
	}

	@Override
	public SplitDownLoadResponse splitDownLoad(SplitDownLoadRequest request, HttpConfig config, IPackSecure packSecure)
			throws SDKException {
		Request req = new Request();
		//请求参数放在http header中进行传输
		req.setHeaderData(this.convertToHeader(request, packSecure));

		req.setServiceId(ServiceEnum.SPLIT_FILE_DOWNLOAD.getServiceId());
		req.setUrl(serverUrl + ServiceEnum.SPLIT_FILE_DOWNLOAD.getUri());
		log.info("分片文件下载接口请求数据：url:{}，Headers:{}， Entity:{}", req.getUrl(), req.getHeaderData(), req.getEntityData());

		BinaryDataResponse binaryResponse = HttpClientUtil.downloadBinaryData(req, config);

		SplitDownLoadResponse response = new SplitDownLoadResponse();
		response.setCode(binaryResponse.getCode());
		response.setMessge(binaryResponse.getMessage());
		SplitDownLoadInfo splitInfo = new SplitDownLoadInfo();
		splitInfo.setSplitBytes(this.decryptFileBytes(binaryResponse.getBinaryData(), packSecure));
		//获取响应报文result节点，明文时为map，密文时为字符串
		Object result = binaryResponse.getResult().get("result");
		if(result != null && !StringUtil.isEmpty(result.toString())){
			if (packSecure != null){
				result = packSecure.decryptAndSignCheck(result.toString());
			}
			Map<String, Object> resultMap = (Map<String, Object>) result;
			splitInfo.setSplitMd5(resultMap.get("splitMd5") == null ? "" : resultMap.get("splitMd5").toString());
			response.setSplitDownLoadInfo(splitInfo);
		}
		return response;
	}

	@Override
	public FileSyncDownloadResponse fileSyncDownLoad(FileSyncDownloadRequest request, HttpConfig config, IPackSecure packSecure)
			throws SDKException {
		Request req = new Request();
		//请求参数放在http header中进行传输
		req.setHeaderData(this.convertToHeader(request, packSecure));
		req.setServiceId(ServiceEnum.FILE_DOWNLOAD.getServiceId());
		req.setUrl(serverUrl + ServiceEnum.FILE_DOWNLOAD.getUri());
		log.info("文件下载接口请求数据：url:{}，Headers:{}， Entity:{}", req.getUrl(), req.getHeaderData(), req.getEntityData());
		BinaryDataResponse binaryResponse = HttpClientUtil.downloadInputStream(req, config);

		FileSyncDownloadResponse response = new FileSyncDownloadResponse();
		response.setCode(binaryResponse.getCode());
		response.setMessage(binaryResponse.getMessage());
		//获取响应报文result节点，明文时为map，密文时为字符串
		Object result = binaryResponse.getResult().get("result");
		if(result != null && !StringUtil.isEmpty(result.toString())){
			if (packSecure != null){
				result = packSecure.decryptAndSignCheck(result.toString());
			}
			SyncDownLoadInfo syncDownLoadInfo = JsonUtil.MAPPER.convertValue(result, SyncDownLoadInfo.class);
			syncDownLoadInfo.setResp(binaryResponse.getResp());
			syncDownLoadInfo.setClient(binaryResponse.getClient());
			response.setSyncDownLoadInfo(syncDownLoadInfo);
		}
		log.info("文件下载接口获取响应结果成功。");
		return response;
	}
	

	@Override
	public ImageDownloadResponse downloadFromImageSystem(ImageDownloadRequest request, HttpConfig config, IPackSecure packSecure)
			throws SDKException {
		String fileId = request.getFileId().trim();
		if(fileId.contains(" ")){
			request.setTranCode("2002");
		}else {
			request.setTranCode("L2005");
		}
		Request req = new Request();
		//请求参数
		req.setHeaderData(this.convertToHeader(request, packSecure));
		req.setServiceId(ServiceEnum.IMG_SYSTEM_FILE_DOWNLOAD.getServiceId());
		req.setUrl(serverUrl + ServiceEnum.IMG_SYSTEM_FILE_DOWNLOAD.getUri());
		log.info("文件查询接口请求数据：url[{}]，Headers[{}]， Entity[{}]", req.getUrl(), req.getHeaderData(), req.getEntityData());

		BinaryDataResponse response = HttpClientUtil.downloadBinaryData(req, config);

		ImageDownloadResponse imageResp = new ImageDownloadResponse();
		imageResp.setCode(response.getCode());
		imageResp.setMessage(response.getMessage());
		ImageInfo imageInfo = new ImageInfo();
		imageInfo.setBytes(this.decryptFileBytes(response.getBinaryData(), packSecure));
		//获取响应报文result节点
		Object result = response.getResult().get("result");
		if(result != null){
			if (packSecure != null){
				result = packSecure.decryptAndSignCheck(result.toString());
			}
			Map<String, Object> resultMap = (Map<String, Object>) result;
			imageInfo.setFileMd5(resultMap.get("fileMd5") == null ? "" : resultMap.get("fileMd5").toString());
			imageInfo.setFileName(resultMap.get("fileName") == null ? "" : resultMap.get("fileName").toString());
			imageInfo.setBillType(resultMap.get("billType") == null ? "" : resultMap.get("billType").toString());
			imageInfo.setBusiDate(resultMap.get("busiDate") == null ? "" : resultMap.get("busiDate").toString());
		}
		imageResp.setImageInfo(imageInfo);
		return imageResp;
	}

	@Override
	public List<FileInfo> queryFileList(FileQueryRequest request, HttpConfig config, IPackSecure packSecure)
			throws SDKException {
		Request req = new Request();
		// 请求参数放在http header中进行传输
		req.setHeaderData(this.convertToHeader(request, packSecure));

		req.setServiceId(ServiceEnum.QUERY_FILE_BATCH.getServiceId());
		req.setUrl(serverUrl + ServiceEnum.QUERY_FILE_BATCH.getUri());
		log.info("文件批量查询接口请求数据：url:{}，Headers:{}， Entity:{}", req.getUrl(), req.getHeaderData(), req.getEntityData());
		Response response = HttpClientUtil.post(req, config);
		if(!DealCode.SUCCESS.getCode().equals(response.getCode())){
			throw new SDKException(response.getCode(),response.getMessage());
		}

		// 获取响应报文result节点，明文时为map，密文时为字符串
		Object result = response.getResult();
		if (result != null && !StringUtil.isEmpty(result.toString())) {
			if (packSecure != null) {
				result = packSecure.decryptAndSignCheck(result.toString());
			}
			log.info("文件批量查询接口获取响应结果成功。");
			Map<String,List<FileInfo>> resultMap = JsonUtil.MAPPER.convertValue(result, new TypeReference<Map<String,List<FileInfo>>>() {});
			return resultMap.get("fileInfoArry");
		}

		return new ArrayList<FileInfo>();
	}


	@Override
	public Map<String, Object> upload2009Notice(HashMap<String, Object> maphead, byte[] fileBytes, HttpConfig config, IPackSecure packSecure)
			throws SDKException  {
		try {
			UploadLimitConfig limitConf = this.queryFileUploadLimitConfig(maphead.get("channelId").toString(),config,packSecure);
			int actualSize = Integer.parseInt(maphead.get("splitSize").toString());
			if(actualSize > limitConf.getFileSplitSizeLimit()){
				throw new SDKException(DealCode.SPLIT_SIZE_OVERFLOW.getCode(),DealCode.SPLIT_SIZE_OVERFLOW.getMsg()+"限制大小为:"+limitConf.getFileSplitSizeLimit()+",实际大小为:"+actualSize);
			}
			BinaryDataRequest binaryReq = new BinaryDataRequest();
			binaryReq.setBinaryData(this.encryptFileBytes(fileBytes, packSecure));
			binaryReq.setHeaderData(this.convertNewToHeader(maphead, packSecure));
			binaryReq.setServiceId(ServiceEnum.FILE_2009_NOTICE.getServiceId());
			binaryReq.setUrl(serverUrl + ServiceEnum.FILE_2009_NOTICE.getUri());
			log.info("文件上传接口请求数据：url:{}，Headers:{}, binaryData.length:{}", binaryReq.getUrl(),
					binaryReq.getHeaderData(), fileBytes.length);
			Response response = HttpClientUtil.uploadBinaryData(binaryReq, config);
			log.info("传输完成通知接口请求数据：url:{}，Headers:{}", binaryReq.getUrl(), binaryReq.getHeaderData());
			if(!DealCode.SUCCESS.getCode().equals(response.getCode())) {
				throw new SDKException(response.getCode(),response.getMessage());
			}
			Map<String, Object> resultMap=response.getResultMap();
			Object result = response.getResult();
			if(result != null && !StringUtil.isEmpty(result.toString())){
				if (packSecure != null){
					result = packSecure.decryptAndSignCheck(result.toString());
				}
				Map<String, Object> rstMap=(Map<String, Object>)result;

				String encodeUrl=(String)rstMap.get("fileUrl");
				if(StringUtils.isNotEmpty(encodeUrl)) {
					try {
						rstMap.put("fileUrl", URLDecoder.decode(encodeUrl,"utf-8"));
					} catch (Exception e) {
						throw new SDKException(response.getCode(), response.getMessage());
					}
				}
				resultMap.put("result", rstMap);
			}
			log.info("文件上传通知接口获取响应结果成功"+resultMap+"");
			return resultMap;
		}catch(Exception e) {
			if(e instanceof SDKException) {
				throw (SDKException)e;
			}
			throw new SDKException(DealCode.SDK_ERROR.getCode(),e.getMessage());
		}

	}

	@Override
	public Map<String, Object> query2109Rst(HashMap<String, Object> maphead, HttpConfig config, IPackSecure packSecure)
			throws SDKException  {
		try {
			Request req = new Request();
			// 请求参数放在http header中进行传输
			req.setHeaderData(this.convertNewToHeader(maphead, packSecure));

			req.setServiceId(ServiceEnum.FILE_2109_RESULT.getServiceId());
			req.setUrl(serverUrl + ServiceEnum.FILE_2109_RESULT.getUri());
			log.info("查询大文件影像上传结果请求数据：url:{}，Headers:{}", req.getUrl(), req.getHeaderData());
			Response response = HttpClientUtil.post(req, config);
			if(!DealCode.SUCCESS.getCode().equals(response.getCode())) {
				throw new SDKException(response.getCode(),response.getMessage());
			}
			Map<String, Object> resultMap=response.getResultMap();
			Object result = response.getResult();
			if(result != null && !StringUtil.isEmpty(result.toString())){
				if (packSecure != null){
					result = packSecure.decryptAndSignCheck(result.toString());
				}
				Map<String, Object> rstMap=(Map<String, Object>)result;
				String encodeUrl=(String)rstMap.get("fileUrl");
				if(StringUtils.isNotEmpty(encodeUrl)) {
					try {
						rstMap.put("fileUrl", URLDecoder.decode(encodeUrl,"utf-8"));
					} catch (Exception e) {
						throw new SDKException(response.getCode(), response.getMessage());
					}
				}
				resultMap.put("result", rstMap);
			}
			log.info("查询大文件影像上传结果接口获取响应结果成功"+resultMap+"");
			return resultMap;
		}catch(Exception e) {
			if(e instanceof SDKException) {
				throw (SDKException)e;
			}
			throw new SDKException(DealCode.SDK_ERROR.getCode(),e.getMessage());
		}
	}

	
	
	@Override
	public StreamDownLoadInfo downloadBigImage(HashMap<String, Object> maphead, HttpConfig config, IPackSecure packSecure) throws SDKException {
		try {
			Request req = new Request();
			//请求参数
			req.setHeaderData(this.convertNewToHeader(maphead, packSecure));
			req.setServiceId(ServiceEnum.IMG_BIG_FILE_DOWNLOAD.getServiceId());
			req.setUrl(serverUrl + ServiceEnum.IMG_BIG_FILE_DOWNLOAD.getUri());
			log.info("大文件下载请求数据：url[{}]，Headers[{}]， Entity[{}]", req.getUrl(), req.getHeaderData(), req.getEntityData());
			
			BinaryDataResponse binaryResponse = HttpClientUtil.downloadInputStream(req, config);
			
//			FileSyncDownloadResponse response = new FileSyncDownloadResponse();
//			response.setCode(binaryResponse.getCode());
//			response.setMessage(binaryResponse.getMessage());
			//获取响应报文result节点，明文时为map，密文时为字符串
			StreamDownLoadInfo rst = new StreamDownLoadInfo();
			rst.setResp(binaryResponse.getResp());
			rst.setClient(binaryResponse.getClient());
			log.info("文件下载接口获取响应结果成功。");
			return rst;
		} catch (Exception e) {
			if(e instanceof SDKException) {
				throw (SDKException)e;
			}
			throw new SDKException(DealCode.SDK_ERROR.getCode(),e.getMessage());
		}
	}

	@Override
	public UploadLimitConfig queryFileUploadLimitConfig(String channelId, HttpConfig config, IPackSecure packSecure)
			throws SDKException {
		try {
			Request req = new Request();
			Map<String, Object> maphead = new HashMap<String, Object>();
			maphead.put("channelId", channelId);
			// 请求参数放在http header中进行传输
			req.setHeaderData(this.convertNewToHeader(maphead, packSecure));

			req.setServiceId(ServiceEnum.FILE_UPLOAD_CONFIG.getServiceId());
			req.setUrl(serverUrl + ServiceEnum.FILE_UPLOAD_CONFIG.getUri());
			log.info("获取渠道{}上传限制参数接口请求数据：url:{}，Headers:{}", channelId, req.getUrl(), req.getHeaderData());
			Response response = HttpClientUtil.post(req, config);
			if (!DealCode.SUCCESS.getCode().equals(response.getCode())) {
				throw new SDKException(response.getCode(), response.getMessage());
			}
			Map<String, Object> resultMap = response.getResultMap();
			if (response.getResult() != null && !"".equals(response.getResult()) && packSecure != null) {
				resultMap.put("result", packSecure.decryptAndSignCheck(response.getResult().toString()));
			}
			Map<String, Object> rm = (Map<String, Object>) resultMap.get("result");
			UploadLimitConfig limitConfig = new UploadLimitConfig();
			limitConfig.setFileSizeLimit(Integer.parseInt(rm.get("fileSizeLimit").toString()));
			limitConfig.setFileSplitSizeLimit(Integer.parseInt(rm.get("fileSplitSizeLimit").toString()));
			limitConfig.setFileSplitSizeLimit2001(Integer.parseInt(rm.get("fileSizeLimit2001").toString()));
			log.info("获取渠道{}上传限制参数接口请求结果成功!!",channelId);
			return limitConfig;
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException) e;
			}
			throw new SDKException(DealCode.SDK_ERROR.getCode(), e.getMessage());
		}
	}
	
	@Override
	public Map<String, Object> upload2001(HashMap<String, Object> maphead, byte[] fileBytes,HttpConfig config, IPackSecure packSecure) throws SDKException {
		try {
			
			BinaryDataRequest binaryReq = new BinaryDataRequest();
			binaryReq.setBinaryData(this.encryptFileBytes(fileBytes, packSecure));
			binaryReq.setHeaderData(this.convertNewToHeader(maphead, packSecure));
			binaryReq.setServiceId(ServiceEnum.IMG_BIG_FILE_UPLOAD_2001.getServiceId());
			binaryReq.setUrl(serverUrl + ServiceEnum.IMG_BIG_FILE_UPLOAD_2001.getUri());
			log.info("文件上传接口请求数据：url:{}，Headers:{}, binaryData.length:{}", binaryReq.getUrl(),
					binaryReq.getHeaderData(), fileBytes.length);
			Response response = HttpClientUtil.uploadBinaryData(binaryReq, config);
			log.info("传输完成通知接口请求数据：url:{}，Headers:{}", binaryReq.getUrl(), binaryReq.getHeaderData());
			if (!DealCode.SUCCESS.getCode().equals(response.getCode())) {
				throw new SDKException(response.getCode(), response.getMessage());
			}
			Object result = response.getResult();
			if (result != null && !StringUtil.isEmpty(result.toString())) {
				if (packSecure != null) {
					result = packSecure.decryptAndSignCheck(result.toString());
				}
				response.getResultMap().put("result", result);
			}
			return (Map<String, Object>) response.getResultMap();
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException) e;
			}
			throw new SDKException(DealCode.SDK_ERROR.getCode(), e.getMessage());
		}
		
	}
	
	public String getServerUrl() {
		return serverUrl;
	}

	public APIServiceImpl setServerUrl(String serverUrl) {
		this.serverUrl = serverUrl;
		return this;
	}

}
