package com.zbank.file.api;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

import com.zbank.file.common.utils.BeanUtils;
import com.zbank.file.common.utils.JsonUtil;
import com.zbank.file.common.utils.SM4Utils;
import com.zbank.file.exception.SDKException;
import com.zbank.file.secure.IPackSecure;

public abstract class AbstractAPIServiceImpl implements APIService {

	
	@Override
	public Map<String, String> convertToHeader(Object obj, IPackSecure packSecure) throws SDKException {
		//请求参数放在http header中进行传输
		try {
			String headerData;
			if (packSecure == null) {
				headerData = JsonUtil.MAPPER.writeValueAsString(obj);
				return Collections.singletonMap(PARAM_HEADER, URLEncoder.encode(headerData, "UTF-8"));
			}
			//加解密类不为null时 进行加密
			Map<String, Object> reqParam = BeanUtils.beanToMap(obj);
			headerData = packSecure.encryptAndSign(reqParam);
			Map<String, String> header = new HashMap<String, String>();
			header.put(PARAM_HEADER, headerData);
			header.put(ENCRYPT_TYPE_HEADER, packSecure.getEncryptType());
			if(Pattern.matches("^SM2AES|RSAAES|SM2SM4$", packSecure.getEncryptType())){
				header.put(ENCRYPT_KEY, packSecure.getKey());
				header.put(CKS_STR, packSecure.getChkStr());
			}
			if(Pattern.matches("^SM2SM4$", packSecure.getEncryptType())){
				header.put(IV, packSecure.getIV());
			}
			return header;
		}catch (Exception e){
			throw new SDKException("请求报文转换为http header发生异常", e);
		}
	}



	public  Map<String, String> convertNewToHeader(Map<String, Object> reqParam, IPackSecure packSecure) throws SDKException {
		//请求参数放在http header中进行传输
		try {
			String headerData;
			if (packSecure == null) {
				headerData=mapToString(reqParam);
				return Collections.singletonMap(PARAM_HEADER, URLEncoder.encode(headerData, "UTF-8"));
			}
			//加解密类不为null时 进行加密
			headerData = packSecure.encryptAndSign(reqParam);
			Map<String, String> header = new HashMap<String, String>();
			header.put(PARAM_HEADER, headerData);
			header.put(ENCRYPT_TYPE_HEADER, packSecure.getEncryptType());
			if(Pattern.matches("^SM2AES|RSAAES|SM2SM4$", packSecure.getEncryptType())){
				header.put(ENCRYPT_KEY, packSecure.getKey());
				header.put(CKS_STR, packSecure.getChkStr());
			}
			if(Pattern.matches("^SM2SM4$", packSecure.getEncryptType())){
				header.put(IV, packSecure.getIV());
			}
			return header;
		}catch (Exception e){
			throw new SDKException("请求报文转换为http header发生异常", e);
		}
	}


	/**
	 * 将Map内容转换成字符串，用于日志打印输出。
	 * @param map
	 * @return
	 */
	public static String mapToString(Map<String, Object> map) {
		if(map == null) {
			return "{}";
		}
		StringBuffer sb = new StringBuffer();
		sb.append("{ ");
		for(String s : map.keySet()) {
			sb.append("\""+s+"\":\""+map.get(s)+"\",");
		}
		sb.replace(sb.length()-1, sb.length(), "}");
		return sb.toString();
	}
	
	@Override
	public byte[] encryptFileBytes(byte[] fileBytes, IPackSecure packSecure) throws SDKException {
		if (packSecure == null || fileBytes == null || fileBytes.length==0) {
			return fileBytes;
		}
		ByteArrayOutputStream bou = new ByteArrayOutputStream();
		if(Pattern.matches("^SM2SM4$", packSecure.getEncryptType())){
			try{
				SM4Utils.encrypt(new ByteArrayInputStream(fileBytes),bou , packSecure.getUnEncryptKey());
			}catch(Exception e){
				new SDKException("文件加密出错",e);
			}
			return bou.toByteArray();
		}
		return fileBytes;
	}

	@Override
	public byte[] decryptFileBytes(byte[] fileBytes, IPackSecure packSecure) throws SDKException {
		if (packSecure == null || fileBytes == null ||fileBytes.length==0) {
			return fileBytes;
		}
		ByteArrayOutputStream bou = new ByteArrayOutputStream();
		if(Pattern.matches("^SM2SM4$", packSecure.getEncryptType())){
			try{
				SM4Utils.decrypt(new ByteArrayInputStream(fileBytes),bou , packSecure.getUnEncryptKey());
			}catch(Exception e){
				new SDKException("文件加密出错",e);
			}
			return bou.toByteArray();
		}
		return fileBytes;
	}
}
