package com.zbank.file.common.utils;


import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.Security;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.util.UUID;

import javax.crypto.Cipher;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;

import org.bouncycastle.jce.provider.BouncyCastleProvider;

/**
 * 
 * @author zhanglulu
 *
 */
public class SM4Utils {

	private static final String PADDING_MODE = "SM4/ECB/PKCS5Padding";
	
	private static final String PADDING_MODE_CBC = "SM4/CBC/PKCS5Padding";  

	private static final String FILE_MODE_READ = "r";

	private static final String FILE_MODE_READ_WRITE = "rw";

	private static final String PBK_SHA1 = "PBKDF2WithHmacSHA1";

	private static final String ALGORITHM_SM4 = "SM4";

	private static final int KEY_DEFAULT_SIZE = 128;

	private static final int ENCRYPT_BUFFER_SIZE = 1024;

	private static final int DECRYPT_BUFFER_SIZE = 1040;

	static {
		try {
			Security.addProvider(new BouncyCastleProvider());
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * 文件加密
	 * 
	 * @param sourceFile 源文件
	 * @param destFile 加密后文件
	 * @param key 密钥
	 * @throws Exception
	 */
	public static void encrypt(File sourceFile, File destFile, byte[] key) throws Exception {
		sm4Cipher(Cipher.ENCRYPT_MODE, sourceFile, destFile, key);
	}

	/**
	 * 文件解密
	 * @param sourceFile 源文件
	 * @param destFile 解密后文件
	 * @param key 密钥
	 * @throws Exception
	 */
	public static void decrypt(File sourceFile, File destFile, byte[] key) throws Exception {
		sm4Cipher(Cipher.DECRYPT_MODE, sourceFile, destFile, key);
	}
	
	/**
	 * 文件加密
	 * 
	 * @param sourceFile 源文件
	 * @param dest 目标输出流
	 * @param key 密钥
	 * @throws Exception
	 */
	public static void encrypt(File sourceFile, OutputStream dest, byte[] key) throws Exception {
		sm4Cipher(Cipher.ENCRYPT_MODE, sourceFile, dest, key);
	}

	/**
	 * 文件解密
	 * @param sourceFile 源文件
	 * @param dest 目标输出流
	 * @param key 密钥
	 * @throws Exception
	 */
	public static void decrypt(File sourceFile, OutputStream dest, byte[] key) throws Exception {
		sm4Cipher(Cipher.DECRYPT_MODE, sourceFile, dest, key);
	}

	/**
	 * 流加密
	 * @param src 源
	 * @param dest 目标
	 * @param key 密钥
	 * @throws Exception
	 */
	public static void encrypt(InputStream src, OutputStream dest, byte[] key) throws Exception {
		sm4Cipher(Cipher.ENCRYPT_MODE, src, dest, key);
	}

	/**
	 * 流解密
	 * @param src 源
	 * @param dest 目标
	 * @param key 密钥
	 * @throws Exception
	 */
	public static void decrypt(InputStream src, OutputStream dest, byte[] key) throws Exception {
		sm4Cipher(Cipher.DECRYPT_MODE, src, dest, key);
	}
	
	
	/**
	 * 对流进行加解密/解密处理。
	 * 
	 * @param cipherMode 加密/解密模式。Cipher.ENCRYPT_MODE 或者 Cipher.DECRYPT_MODE
	 * @param src 源输入流
	 * @param dest 目的输出流
	 * @param key 密钥
	 * @throws Exception
	 */
	private static void sm4Cipher(int cipherMode, InputStream src, OutputStream dest, byte[] key) throws Exception {

		try {
			Cipher mCipher = generateEcbCipher(cipherMode, key);

			int bufferSize = Cipher.ENCRYPT_MODE == cipherMode ? ENCRYPT_BUFFER_SIZE : DECRYPT_BUFFER_SIZE;
			byte[] byteData = new byte[bufferSize];
			int n = src.read(byteData);
			while (n != -1) {
				byte[] rsult = mCipher.doFinal(byteData, 0, n);
				dest.write(rsult);
				dest.flush();
				n = src.read(byteData);
			}
		} finally {
			if (src != null) {
				try {
					src.close();
				} catch (Exception e) {}

			}
			if (dest != null) {
				try {
					dest.close();
				} catch (Exception e) {}
			}
		}

	}
	
	/**
	 * 流加密
	 * @param src 源
	 * @param dest 目标
	 * @param key 密钥
	 * @throws Exception
	 */
	public static void encryptNetStream(InputStream src, OutputStream dest, byte[] key) throws Exception {
		sm4Cipher(Cipher.ENCRYPT_MODE, src, dest, key,true);
	}

	/**
	 * 流解密
	 * @param src 源
	 * @param dest 目标
	 * @param key 密钥
	 * @throws Exception
	 */
	public static void decryptNetStream(InputStream src, OutputStream dest, byte[] key) throws Exception {
		sm4Cipher(Cipher.DECRYPT_MODE, src, dest, key,true);
	}
	
	/**
	 * 对流进行加解密/解密处理。
	 * 
	 * @param cipherMode 加密/解密模式。Cipher.ENCRYPT_MODE 或者 Cipher.DECRYPT_MODE
	 * @param src 源输入流
	 * @param dest 目的输出流
	 * @param key 密钥
	 * @throws Exception
	 */
	private static void sm4Cipher(int cipherMode, InputStream src, OutputStream dest, byte[] key,boolean isNetStream) throws Exception {
		if(!isNetStream){
			SM4Utils.sm4Cipher(cipherMode, src, dest, key);
		}else{
			try {
				Cipher mCipher = generateEcbCipher(cipherMode, key);
				int bufferSize = Cipher.ENCRYPT_MODE == cipherMode ? ENCRYPT_BUFFER_SIZE : DECRYPT_BUFFER_SIZE;
				ByteBuffer buffer = ByteBuffer.allocate(8 * bufferSize);
				byte[] byteData = new byte[8 * bufferSize];
				int n = src.read(byteData);
				while (n != -1) {
					int readLength;
					if (buffer.remaining() >= n) {
						readLength = n;
					} else {
						readLength = buffer.remaining();
					}
					buffer.put(byteData, 0, readLength);
					if (!buffer.hasRemaining()) {
						buffer.flip();
						byte[] b = new byte[bufferSize];
						for (int i = 0; i < 8; i++) {
							buffer.get(b);
							dest.write(mCipher.doFinal(b, 0, bufferSize));
							dest.flush();
						}
						buffer.clear();
					}
					if (readLength < n) {
						buffer.put(byteData, readLength, n - readLength);
					}
					n = src.read(byteData);
				}
				buffer.flip();
				if (buffer.hasRemaining()) {
					int remainging = buffer.remaining();
					int loopTimes = remainging % bufferSize == 0 ? (remainging / bufferSize): (remainging / bufferSize + 1);
					byte[] b = new byte[bufferSize];
					int getLength;
					for (int i = 0; i < loopTimes; i++) {
						if(i == loopTimes - 1){
							getLength = remainging - bufferSize * i;
						}else{
							getLength = bufferSize;
						}
						buffer.get(b, 0, getLength);
						dest.write(mCipher.doFinal(b, 0, getLength));
						dest.flush();
					}
				}
			} finally {
				if (src != null) {
					try {
						src.close();
					} catch (Exception e) {}

				}
				if (dest != null) {
					try {
						dest.close();
					} catch (Exception e) {}
				}
			}
		}
	}

	/**
	 * 文件加/解密
	 * @param cipherMode 加/解密
	 * @param sourceFile 源文件
	 * @param targetFile 目标文件
	 * @param key 密钥
	 * @throws Exception
	 */
	@SuppressWarnings("resource")
	private static void sm4Cipher(int cipherMode, File sourceFile, File targetFile, byte[] key) throws Exception {
		FileChannel sourcefc = null;
		FileChannel targetfc = null;

		try {
			Cipher mCipher = generateEcbCipher(cipherMode, key);
			sourcefc = new RandomAccessFile(sourceFile, FILE_MODE_READ).getChannel();
			targetfc = new RandomAccessFile(targetFile, FILE_MODE_READ_WRITE).getChannel();

			int bufferSize = Cipher.ENCRYPT_MODE == cipherMode ? ENCRYPT_BUFFER_SIZE : DECRYPT_BUFFER_SIZE;
			ByteBuffer byteData = ByteBuffer.allocate(bufferSize);
			while (sourcefc.read(byteData) != -1) {
				// 通过通道读写交叉进行。
				// 将缓冲区准备为数据传出状态
				byteData.flip();

				byte[] byteList = new byte[byteData.remaining()];
				byteData.get(byteList, 0, byteList.length);
				// 此处，若不使用数组加密解密会失败，因为当byteData达不到1024个时，加密方式不同对空白字节的处理也不相同，从而导致成功与失败。
				byte[] bytes = mCipher.doFinal(byteList);
				targetfc.write(ByteBuffer.wrap(bytes));
				byteData.clear();
			}
		} catch (Exception e) {
			throw e;
		} finally {
			if (sourcefc != null) {
				try {
					sourcefc.close();
				} catch (Exception e) {}

			}
			if (targetfc != null) {
				try {
					targetfc.close();
				} catch (Exception e) {}
			}
		}
	}
	
	/**
	 * 文件加/解密
	 * @param cipherMode 加/解密
	 * @param sourceFile 源文件
	 * @param targetFile 目标文件
	 * @param key 密钥
	 * @throws Exception
	 */
	@SuppressWarnings("resource")
	private static void sm4Cipher(int cipherMode, File sourceFile, OutputStream dest, byte[] key) throws Exception {
		FileChannel sourcefc = null;

		try {
			Cipher mCipher = generateEcbCipher(cipherMode, key);
			sourcefc = new RandomAccessFile(sourceFile, FILE_MODE_READ).getChannel();

			int bufferSize = Cipher.ENCRYPT_MODE == cipherMode ? ENCRYPT_BUFFER_SIZE : DECRYPT_BUFFER_SIZE;
			ByteBuffer byteData = ByteBuffer.allocate(bufferSize);
			while (sourcefc.read(byteData) != -1) {
				// 通过通道读写交叉进行。
				// 将缓冲区准备为数据传出状态
				byteData.flip();

				byte[] byteList = new byte[byteData.remaining()];
				byteData.get(byteList, 0, byteList.length);
				// 此处，若不使用数组加密解密会失败，因为当byteData达不到1024个时，加密方式不同对空白字节的处理也不相同，从而导致成功与失败。
				byte[] bytes = mCipher.doFinal(byteList);
				dest.write(bytes);
				dest.flush();
				byteData.clear();
			}
		} catch (Exception e) {
			throw e;
		} finally {
			if (sourcefc != null) {
				try {
					sourcefc.close();
				} catch (Exception e) {}

			}
			if (dest != null) {
				try {
					dest.close();
				} catch (Exception e) {}
			}
		}
	}

	/**
	 * 大字符串分段加密
	 * 
	 * @param content 需要加密的内容
	 * @param key 密钥
	 * @return 加密后内容
	 * @throws Exception 异常
	 */
	public static String sm4EncryptLarge(String content, byte[] key) throws Exception {
		byte[] data = content.getBytes(Charset.forName("UTF-8"));
		Cipher mCipher = generateEcbCipher(Cipher.ENCRYPT_MODE, key);
		int inputLen = data.length;
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		int offSet = 0;
		byte[] cache;
		int i = 0;
		// 对数据分段解密
		while (inputLen - offSet > 0) {
			if (inputLen - offSet > ENCRYPT_BUFFER_SIZE) {
				cache = mCipher.doFinal(data, offSet, ENCRYPT_BUFFER_SIZE);
			} else {
				cache = mCipher.doFinal(data, offSet, inputLen - offSet);
			}
			out.write(cache, 0, cache.length);
			i++;
			offSet = i * ENCRYPT_BUFFER_SIZE;
		}
		byte[] encryptedData = out.toByteArray();
		out.close();
		return Utils.byteToHex(encryptedData);
	}

	/**
	 * 大字符串分段解密
	 * 
	 * @param content 密文
	 * @param key 密钥
	 * @return 解密后内容
	 * @throws Exception 异常
	 */
	public static String sm4DecryptLarge(String content, byte[] key) throws Exception {
		byte[] data = Utils.hexToByte(content);
		Cipher mCipher = generateEcbCipher(Cipher.DECRYPT_MODE, key);
		int inputLen = data.length;
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		int offSet = 0;
		byte[] cache;
		int i = 0;
		// 对数据分段解密
		while (inputLen - offSet > 0) {
			if (inputLen - offSet > DECRYPT_BUFFER_SIZE) {
				cache = mCipher.doFinal(data, offSet, DECRYPT_BUFFER_SIZE);
			} else {
				cache = mCipher.doFinal(data, offSet, inputLen - offSet);
			}
			out.write(cache, 0, cache.length);
			i++;
			offSet = i * DECRYPT_BUFFER_SIZE;
		}
		byte[] decryptedData = out.toByteArray();
		out.close();
		return new String(decryptedData, Charset.forName("UTF-8"));
	}

	/**
	 * 字符串加密  CBC模式
	 * 
	 * @param data 字符串
	 * @param key 密钥
	 * @return 加密后内容
	 * @throws Exception 异常
	 */
	public static byte[] sm4Encrypt(byte[] data, byte[] key,byte[] iv) throws Exception {
		Cipher mCipher = generateCbcCipher(Cipher.ENCRYPT_MODE, key,iv);
		return mCipher.doFinal(data);
	}

	/**
	 * 字符串解密 CBC模式
	 * 
	 * @param data 加密字符串
	 * @param key 密钥
	 * @return 解密后内容
	 * @throws Exception 异常
	 */
	public static byte[] sm4Decrypt(byte[] data, byte[] key,byte[] iv) throws Exception {
		Cipher mCipher = generateCbcCipher(Cipher.DECRYPT_MODE, key,iv);
		return mCipher.doFinal(data);
	}
	
	/**
	 * 字符串加密  ECB模式
	 * 
	 * @param data 字符串
	 * @param key 密钥
	 * @return 加密后内容
	 * @throws Exception 异常
	 */
	public static String sm4Encrypt(String data, byte[] key) throws Exception {
		Cipher mCipher = generateEcbCipher(Cipher.ENCRYPT_MODE, key);
		byte[] bytes = mCipher.doFinal(data.getBytes(Charset.forName("UTF-8")));
		return Utils.byteToHex(bytes);
	}

	/**
	 * 字符串解密 ECB模式
	 * 
	 * @param data 加密字符串
	 * @param key 密钥
	 * @return 解密后内容
	 * @throws Exception 异常
	 */
	public static String sm4Decrypt(String data, byte[] key) throws Exception {
		Cipher mCipher = generateEcbCipher(Cipher.DECRYPT_MODE, key);
		byte[] bytes = mCipher.doFinal(Utils.hexToByte(data));
		return new String(bytes, Charset.forName("UTF-8"));
	}

	/**
	 * 使用一个安全的随机数来产生一个密匙,密匙加密使用的
	 * 
	 * @param seed 种子
	 * @return 随机数组
	 * @throws NoSuchAlgorithmException 模式错误
	 */
	public static byte[] getRawKey(String seed) throws NoSuchAlgorithmException, InvalidKeySpecException {

		int count = 1000;
		int keyLen = KEY_DEFAULT_SIZE;
		int saltLen = keyLen / 8;
		SecureRandom random = new SecureRandom();
		byte[] salt = new byte[saltLen];
		random.setSeed(seed.getBytes());
		random.nextBytes(salt);
		KeySpec keySpec = new PBEKeySpec(seed.toCharArray(), salt, count, keyLen);
		SecretKeyFactory secretKeyFactory = SecretKeyFactory.getInstance(PBK_SHA1);
		return secretKeyFactory.generateSecret(keySpec).getEncoded();
	}

	/**
	 * 生成ecb模式密码工具
	 * 
	 * @param mode 模式
	 * @param key 密钥
	 * @return 密码工具
	 * @throws Exception 异常
	 */
	private static Cipher generateEcbCipher(int mode, byte[] key) throws Exception {
		Cipher cipher = Cipher.getInstance(PADDING_MODE, BouncyCastleProvider.PROVIDER_NAME);
		Key sm4Key = new SecretKeySpec(key, ALGORITHM_SM4);
		cipher.init(mode, sm4Key);
		return cipher;
	}
	
	/**
	 * 生成ccb模式密码工具
	 * 
	 * @param mode 模式
	 * @param key 密钥
	 * @return 密码工具
	 * @throws Exception 异常
	 */
	private static Cipher generateCbcCipher(int mode, byte[] key,byte[] iv) throws Exception {
		Cipher cipher = Cipher.getInstance(PADDING_MODE_CBC, BouncyCastleProvider.PROVIDER_NAME);
		Key sm4Key = new SecretKeySpec(key, ALGORITHM_SM4);
        IvParameterSpec ivParameterSpec = new IvParameterSpec(iv);
        cipher.init(mode, sm4Key, ivParameterSpec);
        return cipher;
	}

	public static void main(String ... ars) throws Exception {
		Security.addProvider(new BouncyCastleProvider());
		System.out.println(UUID.randomUUID().toString().replaceAll("-", ""));
		byte[] sm4Key = getRawKey(Utils.genRandomKey(32));
		byte[] iv = Utils.hexToByte(UUID.randomUUID().toString().replaceAll("-", ""));
		System.out.println("zhagnsanasdfasdfasfjasgfgas");
		byte[] pack = SM4Utils.sm4Encrypt("zhagnsanasdfasdfasfjasgfgas".getBytes(), sm4Key,iv);
		System.out.println(new String(SM4Utils.sm4Decrypt(pack, sm4Key,iv),"UTF-8"));
		SM4Utils.sm4Cipher(Cipher.ENCRYPT_MODE, new FileInputStream("H:/FeiQ.exe"), new FileOutputStream("H:/test/encrypt/FeiQ.exe"), sm4Key);
		SM4Utils.sm4Cipher(Cipher.DECRYPT_MODE, new FileInputStream("H:/test/encrypt/FeiQ.exe"), new FileOutputStream("H:/test/decrypt/FeiQ.exe"), sm4Key);
		System.out.println(Md5EncodeUtil.encode(new File("H:/FeiQ.exe")).equals(Md5EncodeUtil.encode(new File("H:/test/decrypt/FeiQ.exe"))));
	}

}