package com.zbank.file.sdk;


import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zbank.file.api.APIServiceImpl;
import com.zbank.file.bean.AsyncDownLoadInfo;
import com.zbank.file.bean.FileAsyncDownLoadRequest;
import com.zbank.file.bean.FileDownLoadInfo;
import com.zbank.file.bean.FileInfo;
import com.zbank.file.bean.FileQueryRequest;
import com.zbank.file.bean.FileQueryResponse;
import com.zbank.file.bean.FileSyncDownloadRequest;
import com.zbank.file.bean.FileSyncDownloadResponse;
import com.zbank.file.bean.PreUploadRequest;
import com.zbank.file.bean.PreUploadResponse;
import com.zbank.file.bean.StreamDownLoadInfo;
import com.zbank.file.bean.UploadInfo;
import com.zbank.file.bean.UploadRequest;
import com.zbank.file.common.http.config.HttpConfig;
import com.zbank.file.common.utils.Md5EncodeUtil;
import com.zbank.file.common.utils.StringUtil;
import com.zbank.file.constants.DealCode;
import com.zbank.file.exception.EmptyFileException;
import com.zbank.file.exception.SDKException;
import com.zbank.file.sdk.deprecated.IAsyncDownLoadStreamSDK;
import com.zbank.file.sdk.deprecated.IInputStreamUploadSDK;
import com.zbank.file.sdk.download.async.impl1.DownLoadWithMultipleTask;
import com.zbank.file.sdk.download.async.impl2.DownLoadStreamWithPictureBuffer;
import com.zbank.file.sdk.download.async.impl2.Permit;
import com.zbank.file.sdk.download.sync.SplitIterator;
import com.zbank.file.sdk.upload.SplitUploadTask;
import com.zbank.file.secure.IPackSecure;

/**
 * 众邦分布式文件系统上传下载接口实现
 * 
 * @author zhanglulu
 *
 */
public class FileSDK extends BaseSDK implements IFileSDK,IAsyncDownLoadStreamSDK,IInputStreamUploadSDK{
	private static final Logger log = LoggerFactory.getLogger(FileSDK.class);
	
	protected FileSDK(String url) {
		super();
		this.apiService = new APIServiceImpl(url);
	}

	public static FileSDK build(String url) {
		return new FileSDK(url);
	}

	public FileSDK config(HttpConfig httpConfig) {
		this.httpConfig = httpConfig;
		return this;
	}

	public FileSDK config(IPackSecure packScure) {
		this.packScure = packScure;
		return this;
	}

	@Override
	@Deprecated
	public AsyncDownLoadInfo fileAsyncDownLoadWithPicBuffer(String fileId, String channelId, int concurrency, String seqNo)
			throws SDKException, EmptyFileException {

		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("channelId不能为空!!");
		}
		if (StringUtil.isEmpty(fileId)) {
			throw new SDKException("fileId不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("seqNo不能为空!!");
		}
		if(concurrency < 1) {
			throw new SDKException("concurrency不能小于1!!");
		}
		if(concurrency > BaseSDK.PERMIT_MAX) {
			throw new SDKException("concurrency不能大于"+BaseSDK.PERMIT_MAX+"!!");
		}
		if (fileId.trim().contains(" ") || fileId.trim().contains("/")) {
			throw new SDKException("请使用ImgSDK.downloadSmallFileFromImageSys()方法进行下载!");
		} else {
			FileAsyncDownLoadRequest request = new FileAsyncDownLoadRequest();
			request.setChannelId(channelId);
			request.setFileId(fileId);
			request.setSeqNo(seqNo);
			return new DownLoadStreamWithPictureBuffer(apiService, concurrency, httpConfig, packScure).download(request);
		}
	}

	@Override
	@Deprecated
	public AsyncDownLoadInfo fileAsyncDownLoadWithMutiTask(String fileId, String channelId, int concurrency, String seqNo)
			throws SDKException, EmptyFileException {
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("channelId不能为空!!");
		}
		if (StringUtil.isEmpty(fileId)) {
			throw new SDKException("fileId不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("seqNo不能为空!!");
		}
		if(concurrency < 1) {
			throw new SDKException("concurrency不能小于1!!");
		}
		if(concurrency > BaseSDK.PERMIT_MAX) {
			throw new SDKException("concurrency不能大于"+BaseSDK.PERMIT_MAX+"!!");
		}
		if (fileId.trim().contains(" ") || fileId.trim().contains("/")) {
			throw new SDKException("请使用ImgSDK.downloadSmallFileFromImageSys()方法进行下载!");
		} else {
			FileAsyncDownLoadRequest request = new FileAsyncDownLoadRequest();
			request.setChannelId(channelId);
			request.setFileId(fileId);
			request.setSeqNo(seqNo);
			return new DownLoadWithMultipleTask(apiService, concurrency, httpConfig, packScure).download(request);
		}
	}

	@Override
	public UploadInfo upload(File file, String channelId, String seqNo, boolean isParallel) throws SDKException {
		InputStream inputStream = null;
		if (file == null) {
			throw new SDKException("参数file不能为空!!");
		}
		if (!file.exists()) {
			throw new SDKException("文件" + file.getAbsolutePath() + "不存在!!");
		}
		if (file.isDirectory()) {
			throw new SDKException("文件" + file.getAbsolutePath() + "不能为目录!!");
		}
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("参数seqNo不能为空!!");
		}
		try {
			inputStream = new BufferedInputStream(new FileInputStream(file));
			String md5 = Md5EncodeUtil.encode(file);
			return this.upload(inputStream, md5, file.getName(), file.length(), channelId, seqNo, isParallel, true);
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException)e;
			} else {
				throw new SDKException(DealCode.SDK_ERROR.getCode(), e.getMessage(), e);
			}
		}
	}

	@Override
	@Deprecated
	public UploadInfo upload(InputStream in, String md5, String fileName, long fileSize, String channelId, String seqNo,
			boolean isParallel, boolean closeStream) throws SDKException {
		if (in == null) {
			throw new SDKException("参数in不能为空!!");
		}
		if (StringUtil.isEmpty(md5)) {
			throw new SDKException("参数md5不能为空!!");
		}
		if (StringUtil.isEmpty(fileName)) {
			throw new SDKException("参数fileName不能为空!!");
		}
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("参数seqNo不能为空!!");
		}
		//增加 校验上传文件流和文件大小是否一致
		try {
			if(in.available() != fileSize){
				throw new SDKException("参数fileSize大小和实际输入流大小不一致!!");
			}
		} catch (Exception e) {
			if(e instanceof SDKException){
				throw (SDKException)e;
			}
			throw new SDKException("输入流in读取异常!!",e);
		}
		PreUploadRequest request = new PreUploadRequest();
		request.setChannelId(channelId);
		request.setFileMd5(md5);
		request.setFileName(fileName);
		request.setFileSize(String.valueOf(fileSize));
		request.setSeqNo(seqNo);
		PreUploadResponse response = this.apiService.preUpload(request, this.httpConfig, this.packScure);
		if (DealCode.REPEAT.getCode().equals(response.getCode())) {
			log.info("文件{}已经上传过，请勿重复上传。fileId={}", fileName, response.getUploadConfig().getFileId());
			UploadInfo result = new UploadInfo();
			result.setFileId(response.getUploadConfig().getFileId());
			return result;
		}else if (!DealCode.SUCCESS.getCode().equals(response.getCode())
				&& !DealCode.CONTINUE_UPLOAD.getCode().equals(response.getCode())) {
			throw new SDKException(response.getCode(), response.getMessage());
		}else if(DealCode.DATE_LEN.getCode().equals(response.getCode())){//重新上传延长有效期
			log.info("[{}]{}", response.getUploadConfig().getFileId(), response.getMessage());
			UploadInfo result = new UploadInfo();
			result.setFileId(response.getUploadConfig().getFileId());
			return result;
		}
		
		ThreadPoolExecutor exService = null;
		Permit permit = null;
		try {
			int fileSizeLimit = Integer.parseInt(response.getUploadConfig().getFileSizeLimit());
			if (fileSizeLimit < fileSize) {
				throw new SDKException("所上传的文件大小不能超过" + fileSizeLimit + "字节!!");
			}
			int splitSizeConf = Integer.parseInt(response.getUploadConfig().getFileSplitSize());
			int splitSum = 0;
			if (fileSize % splitSizeConf == 0) {
				splitSum = (int)(fileSize / splitSizeConf);
			} else {
				splitSum = (int)(fileSize / splitSizeConf) + 1;
			}

			char[] faultPic = StringUtil.isEmpty(response.getUploadConfig().getFaultPictrue()) ? null
					: response.getUploadConfig().getFaultPictrue().toCharArray();
			byte[] buffer;
			byte[] uploadResultMap = new byte[splitSum];
			for (int i = 0; i < uploadResultMap.length; i++) {
				uploadResultMap[i] = (byte)0;
			}
			CountDownLatch countDownLatch = null;
			//在这里加一个限定条件，当分片数量大于AbstractSDK.START_CURRENT_THRESHHOLD的时候，我们认为文件上传的时间会比较长，这时采用多线程并发下载合理减少下载时间 
			if (isParallel && splitSum > BaseSDK.START_CURRENT_THRESHHOLD) {
				countDownLatch = new CountDownLatch(splitSum);
				permit = new Permit(BaseSDK.PERMIT_MAX,BaseSDK.PERMIT_MAX);
				exService = new ThreadPoolExecutor(2, BaseSDK.PERMIT_MAX, 120, TimeUnit.SECONDS,new LinkedBlockingQueue<Runnable>());
			}else {
				isParallel = false;
			}
			
			for (int i = 0; i < splitSum; i++) {
				if (faultPic != null && faultPic[i] == '1') {
					try {
						in.skip(splitSizeConf);
					} catch (Exception e) {
						in.read(new byte[splitSizeConf]);
					}
					if(countDownLatch != null){
						countDownLatch.countDown();
					}
					uploadResultMap[i] = (byte)1;
				} else {
					buffer = new byte[splitSizeConf];
					int len = in.read(buffer);
					UploadRequest splitUp = new UploadRequest();
					splitUp.setFileId(response.getUploadConfig().getFileId());
					splitUp.setChannelId(channelId);
					splitUp.setCurrSplitNo(String.valueOf(i));
					if (len < buffer.length) {
						byte[] splitBytes = Arrays.copyOfRange(buffer, 0, len);
						splitUp.setSplitMd5(Md5EncodeUtil.encode(splitBytes));
						splitUp.setFileBytes(splitBytes);
					} else {
						splitUp.setSplitMd5(Md5EncodeUtil.encode(buffer));
						splitUp.setFileBytes(buffer);
					}
					splitUp.setIsRetry(faultPic == null ? "0" : "1");
					splitUp.setSeqNo(seqNo);

					splitUp.setSplitSize(String.valueOf(len));
					splitUp.setSplitSum(String.valueOf(splitSum));
					SplitUploadTask task = new SplitUploadTask(splitUp, uploadResultMap, this.httpConfig, this.packScure,
							this.apiService, countDownLatch,permit);
					if (!isParallel) {
						task.run();
					} else {
						permit.decrease();
						exService.submit(task);
					}
				}
			}

			if (countDownLatch != null) {
				countDownLatch.await();
			}
			boolean isSuccess = true;
			for (int i = 0; i < uploadResultMap.length; i++) {
				if (uploadResultMap[i] == (byte)0) {
					isSuccess = false;
					break;
				}
			}
			if (!isSuccess) {
				throw new SDKException("文件" + fileName + "上传失败，请重新上传！！");
			}
			UploadInfo result = new UploadInfo();
			result.setFileId(response.getUploadConfig().getFileId());
			return result;
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException)e;
			} else {
				throw new SDKException(DealCode.SDK_ERROR.getCode(), e.getMessage(), e);
			}
		} finally {
			if (closeStream && in != null) {
				try {
					in.close();
				} catch (Exception e) {}
			}
			if(exService != null) {
				exService.shutdown();
			}
		}
	}

	@Override
	public UploadInfo upload(byte[] bytes, String fileName, String channelId, String seqNo, boolean isParallel)
			throws SDKException {
		if (bytes == null) {
			throw new SDKException("参数bytes不能为空!!");
		}
		if (StringUtil.isEmpty(fileName)) {
			throw new SDKException("参数fileName不能为空!!");
		}
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("参数seqNo不能为空!!");
		}
		String md5 = Md5EncodeUtil.encode(bytes);
		PreUploadRequest request = new PreUploadRequest();
		request.setChannelId(channelId);
		request.setFileMd5(md5);
		request.setFileName(fileName);
		request.setFileSize(String.valueOf(bytes.length));
		request.setSeqNo(seqNo);
		PreUploadResponse response = this.apiService.preUpload(request, this.httpConfig, this.packScure);

		if (DealCode.REPEAT.getCode().equals(response.getCode())) {
			log.info("文件{}已经上传过，请勿重复上传。fileId={}", fileName, response.getUploadConfig().getFileId());
			UploadInfo result = new UploadInfo();
			result.setFileId(response.getUploadConfig().getFileId());
			return result;
		}else if (!DealCode.SUCCESS.getCode().equals(response.getCode())
				&& !DealCode.CONTINUE_UPLOAD.getCode().equals(response.getCode())) {
			throw new SDKException(response.getCode(), response.getMessage());
		}else if(DealCode.DATE_LEN.getCode().equals(response.getCode())){//重新上传延长有效期
			log.info("[{}]{}", response.getUploadConfig().getFileId(), response.getMessage());
			UploadInfo result = new UploadInfo();
			result.setFileId(response.getUploadConfig().getFileId());
			return result;
		}
		ThreadPoolExecutor exService = null;
		Permit permit = null;
		try {
			int fileSizeLimit = Integer.parseInt(response.getUploadConfig().getFileSizeLimit());
			if (fileSizeLimit < bytes.length) {
				throw new SDKException("所上传的文件大小不能超过" + fileSizeLimit + "字节!!");
			}
			int splitSizeConf = Integer.parseInt(response.getUploadConfig().getFileSplitSize());
			int splitSum = 0;
			if (bytes.length % splitSizeConf == 0) {
				splitSum = (int)(bytes.length / splitSizeConf);
			} else {
				splitSum = (int)(bytes.length / splitSizeConf) + 1;
			}

			char[] faultPic = StringUtil.isEmpty(response.getUploadConfig().getFaultPictrue()) ? null
					: response.getUploadConfig().getFaultPictrue().toCharArray();
			byte[] uploadResultMap = new byte[splitSum];
			for (int i = 0; i < uploadResultMap.length; i++) {
				uploadResultMap[i] = (byte)0;
			}
			CountDownLatch countDownLatch = null;
			//在这里加一个限定条件，当分片数量大于AbstractSDK.START_CURRENT_THRESHHOLD的时候，我们认为文件上传的时间会比较长，这时采用多线程并发下载合理减少下载时间 
			if (isParallel && splitSum > BaseSDK.START_CURRENT_THRESHHOLD) {
				countDownLatch = new CountDownLatch(splitSum);
				permit = new Permit(BaseSDK.PERMIT_MAX,BaseSDK.PERMIT_MAX);
				exService = new ThreadPoolExecutor(2, BaseSDK.PERMIT_MAX, 120, TimeUnit.SECONDS,new LinkedBlockingQueue<Runnable>());
			}else {
				isParallel = false;
			}
			for (int i = 0; i < splitSum; i++) {
				if (faultPic != null && faultPic[i] == '1') {
					uploadResultMap[i] = (byte)1;
					if(countDownLatch != null){
						countDownLatch.countDown();
					}
				} else {
					UploadRequest splitUp = new UploadRequest();
					splitUp.setFileId(response.getUploadConfig().getFileId());
					splitUp.setChannelId(channelId);
					splitUp.setCurrSplitNo(String.valueOf(i));
					int start = splitSizeConf * i;
					byte[] splitBytes = new byte[0];
					if (bytes.length - start >= splitSizeConf) {
						splitBytes = Arrays.copyOfRange(bytes, start, start + splitSizeConf);
					} else {
						splitBytes = Arrays.copyOfRange(bytes, start, bytes.length);
					}
					splitUp.setFileBytes(splitBytes);
					splitUp.setSplitMd5(Md5EncodeUtil.encode(splitBytes));
					splitUp.setFileBytes(splitBytes);
					splitUp.setIsRetry(faultPic == null ? "0" : "1");
					splitUp.setSeqNo(seqNo);
					splitUp.setSplitSize(String.valueOf(splitUp.getFileBytes().length));
					splitUp.setSplitSum(String.valueOf(splitSum));
					SplitUploadTask task = new SplitUploadTask(splitUp, uploadResultMap, this.httpConfig, this.packScure,
							this.apiService, countDownLatch,permit);
					if (!isParallel) {
						task.run();
					} else {
						permit.decrease();
						exService.submit(task);
					}
				}
			}

			if (countDownLatch != null) {
				countDownLatch.await();
			}
			boolean isSuccess = true;
			for (int i = 0; i < uploadResultMap.length; i++) {
				if (uploadResultMap[i] == (byte)0) {
					isSuccess = false;
					break;
				}
			}
			if (!isSuccess) {
				throw new SDKException("文件" + fileName + "上传失败，请重新上传！！");
			}
			UploadInfo result = new UploadInfo();
			result.setFileId(response.getUploadConfig().getFileId());
			return result;
		} catch (Exception e) {
			if (e instanceof SDKException) {
				throw (SDKException)e;
			} else {
				throw new SDKException(DealCode.SDK_ERROR.getCode(), e.getMessage(), e);
			}
		}finally {
			if(exService != null) {
				exService.shutdown();
			}
		}
	}

	@Override
	public FileDownLoadInfo downloadFile(String fileId, String channelId, String destDir, String seqNo, boolean allowMakeDir,
			boolean isParallel) throws SDKException, EmptyFileException {
		if (StringUtil.isEmpty(fileId)) {
			throw new SDKException("参数fileId不能为空!!");
		}
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if (StringUtil.isEmpty(destDir)) {
			throw new SDKException("参数destFilePath不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("参数seqNo不能为空!!");
		}
		File dir = new File(destDir);
		if (!dir.exists()) {
			throw new SDKException("目录"+destDir+"不存在!!");
		}
		if (!dir.isDirectory()) {
			throw new SDKException("文件"+destDir+"不是一个目录!!");
		}
		if (fileId.trim().contains(" ") || fileId.trim().contains("/")) {
			throw new SDKException("请使用ImgSDK.downloadSmallFileFromImageSys()方法进行下载!");
		} else {
			FileQueryRequest request = new FileQueryRequest();
			request.setChannelId(channelId);
			request.setFileId(fileId);
			request.setSeqNo(seqNo);
			FileQueryResponse fileResponse = this.apiService.fileQuery(request, this.httpConfig, packScure);
			if (!DealCode.SUCCESS.getCode().equals(fileResponse.getCode())) {
				throw new SDKException(fileResponse.getCode(), fileResponse.getMessage());
			}
			//处理大小为0 的文件
			if(fileResponse.getFileInfo().getFileSize() == 0){
				throw new EmptyFileException("文件"+fileResponse.getFileInfo().getFileId()+"为空文件,请处理!!");
			}
			if (!allowMakeDir) {
				return downloadFile(fileResponse, fileId, channelId, destDir, seqNo, isParallel);
			} else {
				try {
					destDir = destDir + File.separator + fileId;
					File parentDir = new File(destDir);
					if (!parentDir.exists()) {
						parentDir.mkdirs();
					}
					return downloadFile(fileResponse, fileId, channelId, destDir, seqNo, isParallel);
				} catch (Exception e) {
					if (e instanceof SDKException) {
						throw (SDKException)e;
					} else {
						throw new SDKException(DealCode.SDK_ERROR.getCode(), "文件" + fileId + "下载失败!!请重新下载" + e.getMessage(), e);
					}
				}
			}
		}
	}

	@Override
	public List<FileInfo> queryFileList(String channelId, String fileName, String slotKey, String beginDate,
			String endDate,int pageNo, String seqNo) throws SDKException {
		if (StringUtil.isEmpty(fileName)) {
			throw new SDKException("参数fileName不能为空!!");
		}
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("参数seqNo不能为空!!");
		}
		if (StringUtil.isEmpty(slotKey)) {
			throw new SDKException("参数slotKey不能为空!!");
		}
		if (StringUtil.isEmpty(beginDate)) {
			throw new SDKException("参数beginDate不能为空!!");
		}
		if (StringUtil.isEmpty(endDate)) {
			throw new SDKException("参数endDate不能为空!!");
		}
		if (pageNo < 1) {
			pageNo = 1;
		}
		FileQueryRequest request = new FileQueryRequest();
		request.setChannelId(channelId);
		request.setFileName(fileName);
		request.setBeginDate(beginDate);
		request.setEndDate(endDate);
		request.setSlotKey(slotKey);
		request.setPageNo(String.valueOf(pageNo));
		request.setSeqNo(seqNo);
		return this.apiService.queryFileList(request, this.httpConfig, this.packScure);
	}

	@Override
	public SplitIterator genSplitIterator(String fileId, String channelId, String seqNo) throws SDKException,EmptyFileException{
		if (StringUtil.isEmpty(fileId)) {
			throw new SDKException("参数fileId不能为空!!");
		}
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("参数seqNo不能为空!!");
		}
		
		if (fileId.trim().contains(" ") || fileId.trim().contains("/")) {
			throw new SDKException("请使用ImgSDK.downloadSmallFileFromImageSys()方法进行下载!");
		}
		
		FileQueryRequest request = new FileQueryRequest();
		request.setChannelId(channelId);
		request.setFileId(fileId);
		request.setSeqNo(seqNo);
		FileQueryResponse fileResponse = this.apiService.fileQuery(request, this.httpConfig, packScure);
		if (!DealCode.SUCCESS.getCode().equals(fileResponse.getCode())) {
			throw new SDKException(fileResponse.getCode(), fileResponse.getMessage());
		}
		//处理大小为0 的文件
		if(fileResponse.getFileInfo().getFileSize() == 0){
			throw new EmptyFileException("文件"+fileResponse.getFileInfo().getFileId()+"为空文件,请处理!!");
		}
		return new SplitIterator(fileResponse, apiService, httpConfig, packScure, channelId);
	}
	
	@Override
	public StreamDownLoadInfo downloadStream(String fileId, String channelId, String seqNo) throws SDKException, EmptyFileException {
		if (StringUtil.isEmpty(fileId)) {
			throw new SDKException("参数fileId不能为空!!");
		}
		if (StringUtil.isEmpty(channelId)) {
			throw new SDKException("参数channelId不能为空!!");
		}
		if (StringUtil.isEmpty(seqNo)) {
			throw new SDKException("参数seqNo不能为空!!");
		}
		if (fileId.trim().contains(" ") || fileId.trim().contains("/")) {
			throw new SDKException("请使用ImgSDK.downloadSmallFileFromImageSys()方法进行下载!");
		} else {
			if(Pattern.matches("^SM2SM4$", this.packScure.getEncryptType())){
				return this.downloadStreamSecureTransfer(fileId, channelId, seqNo);
			}
			FileQueryRequest qryRequest = new FileQueryRequest();
			qryRequest.setChannelId(channelId);
			qryRequest.setFileId(fileId);
			qryRequest.setSeqNo(seqNo);
			FileQueryResponse fileResponse = this.apiService.fileQuery(qryRequest, this.httpConfig, packScure);
			if (!DealCode.SUCCESS.getCode().equals(fileResponse.getCode())) {
				throw new SDKException(fileResponse.getCode(), fileResponse.getMessage());
			}
			//处理大小为0 的文件
			if(fileResponse.getFileInfo().getFileSize() == 0){
				throw new EmptyFileException("文件"+fileResponse.getFileInfo().getFileId()+"为空文件,请处理!!");
			}
			
			FileSyncDownloadRequest request = new FileSyncDownloadRequest();
			request.setChannelId(channelId);
			request.setFileId(fileId);
			request.setSeqNo(seqNo);
			FileSyncDownloadResponse response = this.apiService.fileSyncDownLoad(request, this.httpConfig, packScure);
			if (!DealCode.SUCCESS.getCode().equals(response.getCode())) {
				throw new SDKException(response.getCode(), response.getMessage());
			}
			StreamDownLoadInfo result = new StreamDownLoadInfo();
			result.setFileId(fileId);
			result.setFileName(response.getSyncDownLoadInfo().getFileName());
			result.setFileMd5(response.getSyncDownLoadInfo().getFileMd5());
			result.setFileType(response.getSyncDownLoadInfo().getFileType());
			result.setResp(response.getSyncDownLoadInfo().getResp());
			result.setClient(response.getSyncDownLoadInfo().getClient());
			result.setFileSize(response.getSyncDownLoadInfo().getFileSize());
			return result;
		}
	}
}
