package com.zbank.file.secure;

import java.util.HashMap;
import java.util.Map;

import javax.crypto.SecretKey;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zbank.file.common.utils.JsonUtil;
import com.zbank.file.common.utils.RSAAESUtils;
import com.zbank.file.exception.SDKException;

public class RSAAESPackSecure extends AbstractPackScure{
	private static final Logger log = LoggerFactory.getLogger(RSAAESPackSecure.class);

	/**
	 * aesKey加密用的加密密钥
	 */
	private String encryptKeyString;
	/**
	 * 校验和字符串
	 */
	private String cksStr;
	private SecretKey aesKey;
	
	/**
	 * 
	 * @param encryptKeyString 用于aesKey加密用的加密密钥（众邦提供）
	 * @param cksStr 校验和（众邦提供）
	 */
	public RSAAESPackSecure(String encryptKeyString,String cksStr){
		this.encryptKeyString = encryptKeyString;
		try{
			this.aesKey = RSAAESUtils.initKey();
			this.cksStr = Base64.encodeBase64String(RSAAESUtils.encrypt(cksStr.getBytes("UTF-8"), this.aesKey));
		}catch(Exception e){
			log.error("构造加密器失败",e);
			throw new RuntimeException(e);
		}
	}
	
	
	@Override
	public String getKey() {
		try {
			return RSAAESUtils.wrap(this.aesKey, encryptKeyString);
		} catch (Exception e) {
			log.error("aesKey包装加密报错", e);
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public String getChkStr(){
		return this.cksStr;
	}

	@Override
	public String getEncryptType() {
		return "RSAAES";
	}
	

	@Override
	public String encryptAndSign(Map<String, Object> param) throws SDKException {
		try {
			String json = JsonUtil.MAPPER.writeValueAsString(param);
			log.info("加密前报文：{}", json);
			String encrypt = Base64.encodeBase64String(RSAAESUtils.encrypt(json.getBytes("UTF-8"), this.aesKey));
			log.info("加密后报文：{}", encrypt);
			return encrypt;
		} catch (Exception e) {
			log.error("报文AES加密异常", e);
			throw new SDKException(e.getMessage(), e.getCause());
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public Map<String, Object> decryptAndSignCheck(String pack) throws SDKException {
		log.info("解密前报文：{}", pack);
		try {
			// 解密
			byte[] outputData = RSAAESUtils.decrypt(Base64.decodeBase64(pack),this.aesKey);
			String decrypt = new String(outputData,"UTF-8");
			log.info("解密后报文：{}", decrypt);
			return JsonUtil.MAPPER.readValue(decrypt, HashMap.class);
		} catch (Exception e) {
			log.error("报文AES解密异常", e);
			throw new SDKException(e.getMessage(), e.getCause());
		}
	}
	
	@Override
	public byte[] getUnEncryptKey() {
		// TODO Auto-generated method stub
		return this.aesKey.getEncoded();
	}
}
