package com.zbank.file.secure;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zbank.file.bean.SecureConfig;
import com.zbank.file.common.utils.JsonUtil;
import com.zbank.file.common.utils.RSAUtils;
import com.zbank.file.exception.SDKException;

/**
 * @description: 数据加解密默认实现：RSA公钥加密，私钥解密
 */
public class RSAPackSecure extends AbstractPackScure {

	private static final Logger log = LoggerFactory.getLogger(RSAPackSecure.class);

	private SecureConfig secureConfig;

	public RSAPackSecure(SecureConfig secureConfig) {
		this.secureConfig = secureConfig;
	}

	@Override
	public String getEncryptType() {
		return "RSA";
	}

	@Override
	public String encryptAndSign(Map<String, Object> param) throws SDKException {
		try {
			String json = JsonUtil.MAPPER.writeValueAsString(param);
			log.info("加密前报文：{}", json);
			String encrypt = RSAUtils.encryptByPublicKey(json.getBytes("UTF-8"), secureConfig.getServerPubKey());
			log.info("加密后报文：{}", encrypt);
			return encrypt;
		} catch (Exception e) {
			log.error("报文加密加签异常", e);
			throw new SDKException(e.getMessage(), e.getCause());
		}
	}

	@Override
	public Map<String, Object> decryptAndSignCheck(String pack) throws SDKException {

		log.info("解密前报文：{}", pack);
		try {
			String decrypt = RSAUtils.decryptByPrivateKey(pack, secureConfig.getPriKey());
			log.info("解密后报文：{}", decrypt);
			return JsonUtil.MAPPER.readValue(decrypt, HashMap.class);
		} catch (Exception e) {
			log.error("RSA解密异常", e);
			throw new SDKException(e.getMessage(), e.getCause());
		}
	}

	public SecureConfig getSecureConfig() {
		return secureConfig;
	}

	public RSAPackSecure setSecureConfig(SecureConfig secureConfig) {
		this.secureConfig = secureConfig;
		return this;
	}

	
}
