/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.adaptor.interceptor;

import javax.management.MBeanServer;
import javax.management.ObjectName;

import mx4j.log.Log;
import mx4j.log.Logger;

/**
 * The base adaptor interceptor implementation
 *
 * @version $Revision: 1.3 $
 */
public class AdaptorInterceptor implements Interceptor, AdaptorInterceptorMBean
{
   private MBeanServer m_server;
   private Interceptor m_nextInterceptor;
   private boolean m_enabled;
   private ObjectName m_name;

   public MBeanServer getMBeanServer()
   {
      return m_server;
   }

   public void setMBeanServer(MBeanServer server)
   {
      m_server = server;
   }

   public void setNext(Interceptor interceptor)
   {
      m_nextInterceptor = interceptor;
   }

   protected Interceptor getNext()
   {
      return m_nextInterceptor;
   }

   /**
    * Explicitely sets the object name of this interceptor in case it is registered with an MBeanServer. <br>
    * If not set, the object name will be deducted from the adaptor's object name and from
    * this interceptor's name.
    *
    * @see #getObjectName
    * @see #getType
    */
   public void setObjectName(ObjectName name)
   {
      m_name = name;
   }

   /**
    * Returns the object name explicitely set by {@link #setObjectName}
    */
   public ObjectName getObjectName()
   {
      return m_name;
   }

   /**
    * Called before the real invocation on the interceptor chain. <br>
    * By default does nothing; exception thrown by this method will result in
    * {@link #doInvoke} not being called.
    * Subclasses normally override this method to perform some operation before the real invocation
    * on the interceptor chain.
    *
    * @see #postInvoke
    */
   protected void preInvoke(Invocation invocation) throws Exception
   {
   }

   public InvocationResult invoke(Invocation invocation) throws Exception
   {
      Logger logger = getLogger();
      boolean trace = logger.isEnabledFor(Logger.TRACE);
      boolean debug = logger.isEnabledFor(Logger.DEBUG);

      boolean isEnabled = isEnabled();
      if (debug) logger.debug("Adaptor Interceptor '" + getType() + "' is enabled: " + isEnabled);

      try
      {
         if (isEnabled)
         {
            if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', preInvoke...");
            try
            {
               preInvoke(invocation);
               if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', preInvoke completed, no exceptions");
            }
            catch (AdaptorInterceptorException x)
            {
               throw x;
            }
            catch (Exception x)
            {
               if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', preInvoke caught exception: " + x);
               throw new AdaptorInterceptorException(x);
            }
         }

         if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', doInvoke...");
         try
         {
            InvocationResult result = doInvoke(invocation);
            if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', doInvoke completed, no exceptions");
            return result;
         }
         catch (AdaptorInterceptorException x)
         {
            throw x;
         }
         catch (Exception x)
         {
            if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', doInvoke caught exception: " + x);
            throw new AdaptorInterceptorException(x);
         }
      }
      finally
      {
         if (isEnabled)
         {
            if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', postInvoke...");
            try
            {
               postInvoke(invocation);
               if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', postInvoke completed, no exceptions");
            }
            catch (AdaptorInterceptorException x)
            {
               throw x;
            }
            catch (Exception x)
            {
               if (trace) logger.trace("Adaptor Interceptor '" + getType() + "', postInvoke caught exception: " + x);
               throw new AdaptorInterceptorException(x);
            }
         }
      }
   }

   /**
    * The real invocation on the interceptor chain. <br>
    * By default calls the next interceptor in the chain.
    *
    * @see #setNext
    * @see #preInvoke
    * @see #postInvoke
    */
   protected InvocationResult doInvoke(Invocation invocation) throws Exception
   {
      return getNext().invoke(invocation);
   }

   /**
    * Called after the real invocation in the interceptor chain. <br>
    * By default does nothing; exception thrown by this method will be just logged.
    * This method is always called, even in case of exception thrown by {@link #preInvoke}
    * or by {@link #doInvoke}
    * Subclasses normally override this method to perform some operation after the real invocation
    * on the interceptor chain.
    *
    * @see #preInvoke
    */
   protected void postInvoke(Invocation invocation) throws Exception
   {
   }

   public String getType()
   {
      return "standard";
   }

   public void setEnabled(boolean value)
   {
      m_enabled = value;
   }

   public boolean isEnabled()
   {
      return m_enabled;
   }

   protected Logger getLogger()
   {
      return Log.getLogger(getClass().getName());
   }

   public String toString()
   {
      return "Adaptor interceptor [" + getType() + "]";
   }
}
