package com.aliyun.sdk.gateway.pop.interceptor.httpRequest;

import com.aliyun.auth.signature.SignerParams;
import com.aliyun.core.http.*;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.sdk.gateway.pop.auth.SignatureComposer;
import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.provider.AnonymousCredentialProvider;
import com.aliyun.sdk.gateway.pop.auth.signer.PopSigner;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.ParseUtil;
import com.aliyun.core.utils.StringUtils;
import darabonba.core.RequestStyle;
import darabonba.core.TeaConfiguration;
import darabonba.core.TeaPair;
import darabonba.core.TeaRequest;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientOption;
import darabonba.core.exception.TeaException;
import darabonba.core.interceptor.HttpRequestInterceptor;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.utils.CommonUtil;
import darabonba.core.utils.ModelUtil;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.aliyun.core.utils.EncodeUtil.hexEncode;

public class HttpReqInterceptor implements HttpRequestInterceptor {
    private final ClientLogger logger = new ClientLogger(HttpReqInterceptor.class);

    @Override
    public HttpRequest modifyHttpRequest(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        HttpRequest httpRequest = context.httpRequest();
        TeaConfiguration configuration = context.configuration();
        ClientConfiguration clientConfiguration = configuration.clientConfiguration();
        PopSigner signer = (PopSigner) clientConfiguration.option(ClientOption.SIGNER);
        if (signer.getSignerVersion() != SignatureVersion.V3
                && signer.getSignerVersion() != SignatureVersion.V4
                && !request.style().equals(RequestStyle.SSE)) {
            return httpRequest;
        }
        String date = CommonUtil.getTimestamp();
        HttpMethod method = Optional.ofNullable(configuration.method()).orElseGet(request::method);
        Map<String, String> headers = CommonUtil.merge(String.class, CommonUtil.buildMap(
                        new TeaPair("x-acs-signature-nonce", CommonUtil.getNonce()),
                        new TeaPair("x-acs-date", date)
                ),
                request.headers().toMap()
        );
        Map<String, String> query = request.query();
        InputStream bodyStream = null;
        byte[] body = null;
        String hashedRequestPayload = hexEncode(signer.hash(StringUtils.toBytes("")));
        if (!CommonUtil.isUnset(request.stream())) {
            body = ParseUtil.readAsBytes(request.stream());
            hashedRequestPayload = hexEncode(signer.hash(body));
            headers.put("content-type", FormatType.RAW);
            bodyStream = request.stream();
            if (bodyStream.markSupported()) {
                try {
                    bodyStream.reset();
                } catch (IOException e) {
                    throw new TeaException(e);
                }
            } else {
                bodyStream = new ByteArrayInputStream(body);
            }
        } else if (!CommonUtil.isUnset(request.body())) {
            if (request.reqBodyType().equals(BodyType.JSON)) {
                if (request.body() instanceof byte[]) {
                    body = (byte[]) request.body();
                } else {
                    body = StringUtils.toBytes(ParseUtil.toJSONString(request.body()));
                }
                hashedRequestPayload = hexEncode(signer.hash(body));
                headers.put("content-type", FormatType.JSON);
            } else {
                if (request.body() instanceof byte[]) {
                    body = (byte[]) request.body();
                } else {
                    body = StringUtils.toBytes(Objects.requireNonNull(ModelUtil.toFormString(ModelUtil.query(CommonUtil.assertAsMap(request.body())))));
                }
                hashedRequestPayload = hexEncode(signer.hash(body));
                headers.put("content-type", FormatType.FORM);
            }
        }
        headers.put(signer.getContent(), hashedRequestPayload);
        if (!(configuration.credentialProvider() instanceof AnonymousCredentialProvider)) {
            ICredential credential = configuration.credentialProvider().getCredentials();
            String accessKeyId = credential.accessKeyId();
            String securityToken = credential.securityToken();
            if (!StringUtils.isEmpty(securityToken)) {
                headers.put("x-acs-accesskey-id", accessKeyId);
                headers.put("x-acs-security-token", securityToken);
            }
            String strToSign = SignatureComposer.composeStringToSign(method, query, headers, request.pathname(), hashedRequestPayload);
            logger.verbose("The string to sign is: {}", strToSign);
            strToSign = signer.getSignerName().algorithmName() + "\n" + hexEncode(signer.hash(StringUtils.toBytes(strToSign)));
            String authorization = "";
            if (signer.getSignerVersion() == SignatureVersion.V4) {
                String region = this.getRegion(request.product(), configuration.endpoint());
                byte[] signingKey = this.getV4SigningKey(signer, credential.accessKeySecret(), request.product(), region, date.substring(0, 10).replaceAll("-", ""));
                authorization = signer.getSignerName().algorithmName()
                        + " Credential="
                        + accessKeyId
                        + "/"
                        + date
                        + "/"
                        + region
                        + "/"
                        + request.product()
                        + "/aliyun_v4_request,SignedHeaders="
                        + SignatureComposer.buildSignedHeaders(headers)
                        + ",Signature="
                        + hexEncode(signer.signString(strToSign, signingKey));
            } else {
                SignerParams params = SignerParams.create();
                params.setCredentials(credential);
                authorization = signer.getSignerName().algorithmName() + " Credential=" + accessKeyId
                        + ",SignedHeaders=" + SignatureComposer.buildSignedHeaders(headers)
                        + ",Signature=" + signer.signString(strToSign, params);
            }
            headers.put("Authorization", authorization);
            logger.verbose("Authorization value is: {}", authorization);
        }
        HttpHeaders httpHeaders = new HttpHeaders(headers);
        httpRequest = new HttpRequest(method,
                ModelUtil.composeUrl(configuration.endpoint(), query, configuration.protocol(), request.pathname()));
        httpRequest.setHeaders(httpHeaders);
        if (!CommonUtil.isUnset(bodyStream)) {
            httpRequest.setStreamBody(bodyStream);
        } else if (!CommonUtil.isUnset(body)) {
            httpRequest.setBody(body);
        }
        return httpRequest;
    }

    private byte[] getV4SigningKey(PopSigner signer, String secret, String product, String region, String date) {
        byte[] sc2 = signer.signString(date, "aliyun_v4" + secret);
        byte[] sc3 = signer.signString(region, sc2);
        byte[] sc4 = signer.signString(product, sc3);
        return signer.signString("aliyun_v4_request", sc4);
    }

    private String getRegion(String product, String endpoint) {
        if (StringUtils.isEmpty(product) || StringUtils.isEmpty(endpoint)) {
            return "center";
        }
        String region = endpoint.replace(product.toLowerCase(), "")
                .replace("aliyuncs.com", "")
                .replace(".", "");
        if (!StringUtils.isEmpty(region)) {
            return region;
        }
        return "center";
    }
}
