package com.els.base.autoupdate.utils;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.ibatis.io.Resources;
import org.apache.ibatis.jdbc.ScriptRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.util.FileCopyUtils;

import com.els.base.core.web.context.support.McodingBeanDefinitionReader;

public class UpgradeLogUtils {

	protected static final Logger logger = LoggerFactory.getLogger(UpgradeLogUtils.class);

	// 暂时用着引用项目当前路径
	// //也可以是McodingBeanDefinitionReader.class.getClassLoader().getResource("").getPath()
//	public static final String PARENT_DIR = UpgradeLogUtils.class.getClassLoader().getResource("/sql").getPath();
//	public static final String PARSE_PATTERN = "yyyyMMddHHmm";
	public static final SimpleDateFormat PARSE_PATTERN = new SimpleDateFormat("yyyyMMddHHmm");
	public static final Pattern FILE_NAME_PATTERN = Pattern.compile("^[\\w+\\-\\_]+\\-(\\d{12})\\.sql$");

	/**
	 * 获取上一次到现在更新的sql文件
	 * @param date 上一次更新的时间
	 * @return List<File> 返回一个List的文件对象
	 * @throws ParseException
	 * @throws IOException 
	 */
	public static List<Resource> findUpdateSqlFile(Date date) throws ParseException, IOException {
		if (date == null) {
			throw new IllegalArgumentException("The date must not be null");
		}
		List<Resource> fileList = new ArrayList<>();
		
		ResourcePatternResolver resourceResolver = new PathMatchingResourcePatternResolver();
		Resource[] resources = resourceResolver.getResources("classpath*:/sql/*.sql");
		
		long time2 = date.getTime();// 更新日期
		for(Resource f: resources){
			String fileName = f.getFilename();
//			File f = resource.getFile();
			//判断一下格式对不对
//			String fileName = f.getName();
			Matcher matcher = FILE_NAME_PATTERN.matcher(fileName);
			if (!matcher.find()) {
				throw new IllegalArgumentException("无法进行系统升级,sql升级文件格式异常，文件名【"+fileName+"】");
			}
			
			String fileDate = matcher.group(1);
			long time1 = PARSE_PATTERN.parse(fileDate).getTime(); // 文件日期
			// 跟传入的日期进行比较，如果大于该日期则返回该对象
			if (time1 > time2) {
				fileList.add(f);
			}
		}

		sort(fileList);
		return fileList;
	}
	
	public static void sort(List<Resource> fileList)  {
		Collections.sort(fileList, new Comparator<Resource>() {

			@Override
			public int compare(Resource file1, Resource file2) {
				String fileName1 = file1.getFilename();
				Matcher matcher1 = FILE_NAME_PATTERN.matcher(fileName1);
				matcher1.find();
				String fDate1 = matcher1.group(1);
				
				String fileName2 = file2.getFilename();
				Matcher matcher2 = FILE_NAME_PATTERN.matcher(fileName2);
				matcher2.find();
				String fDate2 = matcher2.group(1);
				try {
					Long time1 = PARSE_PATTERN.parse(fDate1).getTime();
					Long time2 = PARSE_PATTERN.parse(fDate2).getTime();
					return  time1.compareTo(time2);
					
				} catch (ParseException e) {
					e.printStackTrace();
					return 0;
				}
				
			}

		});
	}

	/**
	 * 读取指定目录的sql文件内容，并以分号进行分割
	 * @param file
	 * @return
	 * @throws IOException 
	 * @throws Exception 
	 */
	public static List<String> readSQLContent(List<Resource> sqlFileList) throws IOException {
		if (CollectionUtils.isEmpty(sqlFileList)) {
			return null;
		}
		List<String> sqlList = new ArrayList<String>();

		for (Resource file : sqlFileList) {
			List<String> tempSqlList = readSQLContent(file);
			if (CollectionUtils.isNotEmpty(tempSqlList)) {
				sqlList.addAll(tempSqlList);
			}
		}
		return sqlList;
	}
	
	public static List<String> readSQLContent(Resource sqlFile) throws IOException{
		List<String> sqlList = new ArrayList<String>();
		
//		String sql1 = FileUtils.readFileToString(sqlFile, CharEncoding.UTF_8);
//		ByteArrayInputStream inputStream = new Byt
		
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream(1024);
		InputStream inputStream = sqlFile.getInputStream();
		FileCopyUtils.copy(inputStream, outputStream);
		outputStream.flush();
		outputStream.close();
		
		String sql1 = new String(outputStream.toByteArray(), Charset.forName("UTF-8"));
		// 去除sql语句注释
		sql1 = UpgradeLogUtils.removeSQLComment(sql1);
		
		String[] sql = sql1.split(";");
		for (String string : sql) {
			if (StringUtils.isBlank(string)) {
				continue;
			}

			sqlList.add(string.trim());
		}
		return sqlList;
	}

	/**
	 * 批量执行sql文件
	 */
	public static void runSqlScript(List<File> sqlFileList) {
		try {
			Properties dbProps = new Properties();
			// 取配置文件可以根据实际的不同修改
			try {
				dbProps.load(McodingBeanDefinitionReader.class.getClassLoader().getResourceAsStream("conf.properties"));
				McodingBeanDefinitionReader.class.getClassLoader().getResource("").getPath();
			} catch (IOException e) {
				e.printStackTrace();
			}
			String driveClassName = dbProps.getProperty("db.driver.class");
			String url = dbProps.getProperty("db.url");
			String username = dbProps.getProperty("db.username");
			String password = dbProps.getProperty("db.password");

			Class.forName(driveClassName).newInstance();
			Connection conn = DriverManager.getConnection(url, username, password);
			ScriptRunner runner = new ScriptRunner(conn);
			Resources.setCharset(Charset.forName("UTF-8")); // 设置字符集,不然中文乱码插入错误
			PrintWriter logWriter = new PrintWriter(System.out);
			runner.setLogWriter(logWriter);// 设置是否输出日志,如果不输出，可以设置为null，也可以设置输出某个文件，如下：
			/*
			 * OutputStream os=new FileOutputStream("C:\\log.txt"); PrintWriter 
			 * pw=new PrintWriter(os);
			 */
			runner.setAutoCommit(false);
			runner.setErrorLogWriter(logWriter);// 错误日志
			runner.setStopOnError(true);
			// 循环设置
			for (File file : sqlFileList) {
				runner.runScript(new FileReader(file));
				// runner.setSendFullScript(true);
			}
			runner.closeConnection();
			conn.close();
		} catch (IOException | InstantiationException | IllegalAccessException | ClassNotFoundException
				| SQLException e) {
			e.printStackTrace();
		}

	}

	/**
	 * JDBC形式批量执行sql语句
	 * @param sqlFileList
	 * @throws SQLException
	 */
	public static void runSqlString(List<String> sqlFileList) throws SQLException {
		if (CollectionUtils.isNotEmpty(sqlFileList)) {
			Connection conn = null;
			Statement stmt = null;
			try {
				Properties dbProps = new Properties();
				// 取配置文件可以根据实际的不同修改
				dbProps.load(McodingBeanDefinitionReader.class.getClassLoader().getResourceAsStream("conf.properties"));
				McodingBeanDefinitionReader.class.getClassLoader().getResource("").getPath();
				String driveClassName = dbProps.getProperty("db.driver.class");
				String url = dbProps.getProperty("db.url");
				String username = dbProps.getProperty("db.username");
				String password = dbProps.getProperty("db.password");

				Class.forName(driveClassName).newInstance();
				conn = DriverManager.getConnection(url, username, password);
				conn.setAutoCommit(false);// 关闭自动提交事务
				stmt = conn.createStatement();
				try {
					for (String sql : sqlFileList) {
						if (StringUtils.isNotBlank(sql)) {
							logger.info("添加语句到批处理开始");
							logger.info(sql);
							logger.info("添加语句到批处理结束");
							stmt.addBatch(sql);
						}
					}
					stmt.executeBatch();
					conn.commit();
				} catch (Exception e) {
					conn.rollback();
					conn.close();
					stmt.close();
					e.printStackTrace();
				}
			} catch (Exception e) {
				conn.rollback();
				stmt.close();
				conn.close();
				e.printStackTrace();
			}
		}
	}
	

	/**
	 * 去除sql注释
	 * @param filePath
	 * @param charSet
	 * @throws Exception
	 */
	public static String removeSQLComment(String sql) {
//		logger.info("sql去除注释前：" + sql);
		Pattern pattern = Pattern.compile("(-{2,}.*?\\n)|(\\/\\*.*?\\*\\/)", Pattern.DOTALL);
		Matcher matcher = pattern.matcher(sql);
		sql = matcher.replaceAll("");
//		logger.info("sql去除注释后：" + sql);
		return sql;
	}
	
//	public static void main(String[] args) {
//		Matcher matcher = FILE_NAME_PATTERN.matcher("abc-201707070909.sql");
//		System.out.println(matcher.find());
//	}
	
}
