package com.els.base.autoupdate.event.listener;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;

import com.els.base.autoupdate.entity.UpgradeLog;
import com.els.base.autoupdate.service.UpgradeLogService;
import com.els.base.autoupdate.utils.UpgradeLogUtils;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;

@Service
public class UpgradeLogListener implements ApplicationContextAware {

	protected static final Logger logger = LoggerFactory.getLogger(UpgradeLogListener.class);

	@Value("${auto.upgrade.is.enable:false}")
	private String isEnable;
	
	@Autowired
	private UpgradeLogService upgradeLogService;

	public void onApplicationEvent(ContextRefreshedEvent event) {
		
		if (!isEnable.equalsIgnoreCase("true")) {
			logger.debug("自动升级功能，已经禁用。启用请配置 auto.upgrade.is.enable 为true");
			return;
		}
		
		/*
		 * 会存在一个问题，在web 项目中（spring mvc），系统会存在两个容器， 一个是root application context ,
		 * 另一个就是我们自己的 projectName-servlet context（作为root application
		 * context的子容器）。 这种情况下，就会造成onApplicationEvent方法被执行两次。
		 * 为了避免上面提到的问题，我们可以只在root application
		 * context初始化完成后调用逻辑代码，其他的容器的初始化完成，则不做任何处理
		 */
		// 防止重复执行。
//		if (event.getApplicationContext().getParent() != null) {
//			return;
//		}

		logger.info("###############执行升级脚本开始#######################");
		// 查询上一次的更新时间
		UpgradeLog lastSuccessLog = upgradeLogService.findLastUpdateLog();
		if (lastSuccessLog == null) {
			// 考虑初次升级，所以默认一个项目上线时间
			// Calendar c=Calendar.getInstance();
			// c.set(2016, 6, 15, 16, 16, 44);
			// sqlFileList = UpgradeLogUtils.findUpdateSqlFile(c.getTime());
			this.upgradeLogService.initNewUpdateLog();
			return;
		}


		List<Resource> sqlFileList = new ArrayList<>();
		try {
			sqlFileList = UpgradeLogUtils.findUpdateSqlFile(lastSuccessLog.getLastUpdateTime());
		} catch (Exception e1) {
			logger.error("升级失败", e1);
			throw new CommonException("升级失败，原因:" + e1.getMessage());
		}

		if (CollectionUtils.isEmpty(sqlFileList)) {
			logger.info("###############执行升级脚本结束,没有可执行的sql#######################");
			return;
		}
		// 先更新语句

		for (Resource sqlFile : sqlFileList) {
			try {
				if (this.upgradeLogService.hasSuccessRecord(sqlFile.getFilename())) {
					continue;
				}
				
				List<String> list = UpgradeLogUtils.readSQLContent(sqlFile);
				if (CollectionUtils.isEmpty(list)) {
					continue;
				}

				logger.info("=========================执行sql开始==========================");
				// JDBC形式执行sql语句
				// UpgradeLogUtils.runSqlString(list);
				// mybatis形式执行sql语句
				upgradeLogService.executeSql(list);
				logger.info("=========================执行sql结束==========================");
				// 保存升级记录

				UpgradeLog newLog = new UpgradeLog();
				newLog.setSqlFileName(sqlFile.getFilename());
				newLog.setLastUpdateTime(new Date());
				newLog.setIsSuccess(Constant.YES_INT);
				upgradeLogService.addObj(newLog);
				
			} catch (Exception e) {
				logger.error("sql执行失败", e);

				UpgradeLog errLog = new UpgradeLog();
				errLog.setSqlFileName(sqlFile.getFilename());
				errLog.setIsSuccess(Constant.NO_INT);
				errLog.setFailReason(e.getMessage());
				errLog.setLastUpdateTime(new Date());
				upgradeLogService.addObj(errLog);

//				throw new CommonException(e);
			}
		}

		logger.info("###############执行升级脚本结束#######################");
	}

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.onApplicationEvent(null);
	}
}