package com.els.base.bidding.service.impl;

import com.els.base.auth.entity.User;
import com.els.base.bidding.dao.BiddingOnlineMapper;
import com.els.base.bidding.entity.BiddingHeader;
import com.els.base.bidding.entity.BiddingHeaderExample;
import com.els.base.bidding.entity.BiddingOnline;
import com.els.base.bidding.entity.BiddingOnlineExample;
import com.els.base.bidding.entity.BiddingSupplier;
import com.els.base.bidding.entity.BiddingSupplierExample;
import com.els.base.bidding.enums.TenderStatusEumn;
import com.els.base.bidding.service.BiddingHeaderService;
import com.els.base.bidding.service.BiddingOnlineService;
import com.els.base.bidding.service.BiddingSupplierService;
import com.els.base.company.entity.Company;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.utils.uuid.UUIDGenerator;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import javax.annotation.Resource;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service("defaultBiddingOnlineService")
public class BiddingOnlineServiceImpl implements BiddingOnlineService {
    @Resource
    protected BiddingOnlineMapper biddingOnlineMapper;
    
    @Resource 
    private BiddingHeaderService biddingHeaderService;
    
    @Resource 
    private BiddingSupplierService biddingSupplierService;

    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
    public void addObj(BiddingOnline t) {
        this.biddingOnlineMapper.insertSelective(t);
    }
    
    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
    @Transactional
	public void insert(Company supCompany, User loginUser, BiddingOnline biddingOnline) {
		// 根据招标计划中降价幅度，判断供应商报价是否合理
    	if (0 != biddingOnline.getPriceCutPercentage().compareTo(BigDecimal.ZERO) && null != biddingOnline.getBeforeUntaxedPrice()) {
    		BigDecimal tamp = null ; 
    		if ("%".equals(biddingOnline.getAmplitudeUnit())) {
    			tamp = biddingOnline.getBeforeUntaxedPrice()
    					.multiply(new BigDecimal(100).subtract(biddingOnline.getPriceCutPercentage()))
    					.divide(new BigDecimal(100)).setScale(2, BigDecimal.ROUND_HALF_UP);
    		} else {
    			tamp = biddingOnline.getBeforeUntaxedPrice()
    					.subtract(biddingOnline.getPriceCutPercentage())
    					.setScale(2, BigDecimal.ROUND_HALF_UP);
    		}
    		if (biddingOnline.getUntaxedUnitPrice().doubleValue() > tamp.doubleValue()) {
				throw new CommonException("根据降价幅度，此次报价的不含税单价不能大于："+ tamp);
			}
    	}
    	
    	// 设置供应商、供应商用户信息
    	biddingOnline.setSupCompanyId(supCompany.getId());
    	biddingOnline.setSupCompanyName(supCompany.getCompanyFullName());
    	biddingOnline.setSupCompanySrmCode(supCompany.getCompanyCode());
    	biddingOnline.setSupUserId(loginUser.getId());
    	biddingOnline.setSupUserName(loginUser.getLoginName());
    	// 报价时间
    	biddingOnline.setOfferTime(new Date());
    	// 创建时间
    	biddingOnline.setCreateTime(new Date());
    	// 报价状态（0无效竞价，1有效竞价，2中标），供应商报价一开始都是为有效竞价
    	biddingOnline.setQuotationState("1");
    	// 执行保存操作
    	this.biddingOnlineMapper.insertSelective(biddingOnline);
    	// 更新供应商投标状态
    	BiddingSupplier supplier = new BiddingSupplier();
    	supplier.setSupBidding("1");// 投标状态：0：未投标，1：已投标
    	supplier.setLastUpdateTime(new Date());// 最后更新时间
    	supplier.setLastUpdateUser(loginUser.getNickName());// 供应商的最后更新人
    	BiddingSupplierExample supplierExample = new BiddingSupplierExample();
    	supplierExample.createCriteria().andBiddingNoEqualTo(biddingOnline.getBiddingNo())
    					.andSupCompanyIdEqualTo(supCompany.getId());
    	this.biddingSupplierService.modifyObjSup(supplier, supplierExample);
		
	}
    
    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
	public void batchInvalidationById(List<String> ids) {
    	/**
    	 *  作废前判断定标审批是否在进行，在进行就不能操作，
    	 *  因为是同一招标单，随便取个ID查询在线竞价信息，得到招标单号
    	 */
    	Assert.isNotEmpty(ids, "ID为空，操作失败");
    	BiddingOnline biddingOnline = this.queryObjById(ids.get(0));
    	BiddingHeaderExample headerExample = new BiddingHeaderExample();
    	headerExample.createCriteria().andBiddingNoEqualTo(biddingOnline.getBiddingNo());
    	List<BiddingHeader> biddingHeaderList = this.biddingHeaderService.queryAllObjByExample(headerExample);
    	Assert.isNotEmpty(biddingHeaderList, "根据招标单号查询出的基本信息为空，操作失败");
    	if ("2".equals(biddingHeaderList.get(0).getBiddingAuditstatus())) {
    		throw new CommonException("单据定标审批已通过，不能进行作废操作");
    	}
    	if (TenderStatusEumn.BIDOVER.getValue().equals(biddingHeaderList.get(0).getBiddingStatus())) {
    		throw new CommonException("单据已评标，不能再进行作废操作");
    	}
    	
		BiddingOnlineExample example = new BiddingOnlineExample();
		// 报价状态（0无效竞价，1有效竞价，2中标）
		example.createCriteria().andIdIn(ids).andQuotationStateEqualTo("0");
		if (this.biddingOnlineMapper.countByExample(example) >0) {
			throw new CommonException("所选报价信息包含无效竞价，作废失败");
		} 
		example.clear(); 
		// 将有效竞价，中标改为无效竞价
		BiddingOnline online = new BiddingOnline();
		online.setQuotationState("0");
		example.createCriteria().andIdIn(ids);
		this.biddingOnlineMapper.updateByExampleSelective(online, example);
		
	}
    
    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
	public void winTheBiddingById(String id) {
    	BiddingOnline biddingOnline = this.queryObjById(id);
    	// 判断是否在定标审批，如果是则不能修改
    	BiddingHeaderExample headerExample = new BiddingHeaderExample();
    	headerExample.createCriteria().andBiddingNoEqualTo(biddingOnline.getBiddingNo());
    	List<BiddingHeader> biddingHeaderList = this.biddingHeaderService.queryAllObjByExample(headerExample);
    	Assert.isNotEmpty(biddingHeaderList, "根据招标单号查询出的基本信息为空，操作失败");
    	if ("2".equals(biddingHeaderList.get(0).getBiddingAuditstatus())) {
    		throw new CommonException("单据定标审批已通过，不能再修改中标状态");
    	}
    	if (TenderStatusEumn.BIDOVER.getValue().equals(biddingHeaderList.get(0).getBiddingStatus())) {
    		throw new CommonException("单据已评标，不能再修改中标状态");
    	}
    	// 报价状态（0无效竞价，1有效竞价，2中标）
    	if (!"1".equals(biddingOnline.getQuotationState())) {
    		throw new CommonException("所选报价信息为无效竞价或已中标状态，操作失败");
    	}
    	// 只有一条报价能中标,循环全部报价信息，看是否已有中标报价
    	BiddingOnlineExample example = new BiddingOnlineExample();
    	example.createCriteria().andBiddingNoEqualTo(biddingOnline.getBiddingNo());
    	List<BiddingOnline> onlineList = this.queryAllObjByExample(example);
    	if (CollectionUtils.isNotEmpty(onlineList)) {
    		onlineList.forEach(online ->{
    			if ("2".equals(online.getQuotationState())) {
    				throw new CommonException("已有一条报价中标，不能再选择其他报价");
    			}
    		});
    	}
    	// 如果全部报价中没有中标的报价，将选择报价的有效竞价改为中标状态
    	biddingOnline.setQuotationState("2");
    	this.modifyObj(biddingOnline);
		
	}
    
    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
	public void noWinTheBiddingById(String id) {
    	BiddingOnline biddingOnline = this.queryObjById(id);
    	// 判断是否在定标审批，如果是则不能修改
    	BiddingHeaderExample headerExample = new BiddingHeaderExample();
    	headerExample.createCriteria().andBiddingNoEqualTo(biddingOnline.getBiddingNo());
    	List<BiddingHeader> biddingHeaderList = this.biddingHeaderService.queryAllObjByExample(headerExample);
    	Assert.isNotEmpty(biddingHeaderList, "根据招标单号查询出的基本信息为空，操作失败");
    	if ("2".equals(biddingHeaderList.get(0).getBiddingAuditstatus())) {
    		throw new CommonException("单据定标审批已通过，不能再修改中标状态");
    	}
    	if (TenderStatusEumn.BIDOVER.getValue().equals(biddingHeaderList.get(0).getBiddingStatus())) {
    		throw new CommonException("单据已评标，不能再修改中标状态");
    	}
    	// 报价状态（0无效竞价，1有效竞价，2中标）
    	if (!"2".equals(biddingOnline.getQuotationState())) {
    		throw new CommonException("所选报价不是中标状态，操作失败");
    	}
    	
		// 将中标状态改为有效竞价
    	biddingOnline.setQuotationState("1");
    	this.modifyObj(biddingOnline);
	}

    @Transactional
    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
    public void addAll(List<BiddingOnline> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        list.stream().forEach(item->{
            if (StringUtils.isBlank(item.getId())) {
                item.setId(UUIDGenerator.generateUUID());
            }
        });
        this.biddingOnlineMapper.insertBatch(list);
    }

    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.biddingOnlineMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
    public void modifyObj(BiddingOnline t) {
        Assert.isNotBlank(t.getId(), "id 为空，无法修改");
        this.biddingOnlineMapper.updateByPrimaryKeySelective(t);
    }
    
    @CacheEvict(value={"biddingOnline"}, allEntries=true)
    @Override
	public void submitResult(List<BiddingOnline> list) {
    	BiddingHeaderExample headerExample = new BiddingHeaderExample();
    	headerExample.createCriteria().andBiddingNoEqualTo(list.get(0).getBiddingNo());
    	List<BiddingHeader> headerList = this.biddingHeaderService.queryAllObjByExample(headerExample);
    	Assert.isNotEmpty(headerList, "根据招标单号查询的基本信息为空，操作失败"); 
    	// 提交评标结果时，判断最后一轮投标时间是否结束
    	if (System.currentTimeMillis() < headerList.get(0).getBiddingEnddate().getTime()) {
    		throw new CommonException("最后一轮投标没结束，不能提交评标结果");
    	}
    	// 判断招标单是否在评标中，只有在评标中才能提交评标结果
    	if (TenderStatusEumn.BIDOVER.getValue().equals(headerList.get(0).getBiddingStatus()) 
    			|| TenderStatusEumn.ISFINISH.getValue().equals(headerList.get(0).getBiddingStatus())
    			|| TenderStatusEumn.ISSCALING.getValue().equals(headerList.get(0).getBiddingStatus())) {
			throw new CommonException("评标结果已提交，不能重复执行提交评标结果操作");
		}
		// 判断提交的评标结果中是否有供应商中标
    	Boolean flag = true;
    	for (BiddingOnline biddingOnline : list) {
    		// 0无效竞价，1有效竞价，2中标
    		if ("2".equals(biddingOnline.getQuotationState())) {
    			flag = false;
    			break;
    		}
    	}
    	if (flag) {
    		throw new CommonException("提交的评标结果中，没有设置供应商中标");
    	}
    	// 更新基础信息中的招标状态
    	if (CollectionUtils.isNotEmpty(headerList)) {
    		BiddingHeader biddingHeader = new BiddingHeader();
    		biddingHeader.setId(headerList.get(0).getId());
    		biddingHeader.setBiddingStatus(TenderStatusEumn.BIDOVER.getValue());
    		biddingHeader.setBiddingAuditstatus("0");// 0-尚未审批，1-正在审批，2-审批通过，3-已驳回
            biddingHeader.setBidSubmitDate(new Date());
            this.biddingHeaderService.modifyObj(biddingHeader);
    	}
    	// 更新 已应标 的供应商招标状态为 已评标
        BiddingSupplierExample supplierExample = new BiddingSupplierExample();
        supplierExample.createCriteria().andBiddingNoEqualTo(list.get(0).getBiddingNo())
                .andSupBiddingStatusEqualTo(TenderStatusEumn.ISBID.getValue());
        supplierExample.or().andSupBiddingStatusEqualTo(TenderStatusEumn.INBID.getValue());
        List<BiddingSupplier> biddingSupplierList = this.biddingSupplierService
                .queryAllObjByExample(supplierExample);
        for (int i = 0; CollectionUtils.isNotEmpty(biddingSupplierList)
                && i < biddingSupplierList.size(); i++) {
            BiddingSupplier biddingSupplier = new BiddingSupplier();
            biddingSupplier.setId(biddingSupplierList.get(i).getId());
            biddingSupplier.setSupBiddingStatus(TenderStatusEumn.BIDOVER.getValue());
            this.biddingSupplierService.modifyObj(biddingSupplier);
        }
		
	}

    @Cacheable(value="biddingOnline", keyGenerator="redisKeyGenerator")
    @Override
    public BiddingOnline queryObjById(String id) {
        return this.biddingOnlineMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="biddingOnline", keyGenerator="redisKeyGenerator")
    @Override
    public List<BiddingOnline> queryAllObjByExample(BiddingOnlineExample example) {
        return this.biddingOnlineMapper.selectByExample(example);
    }

    @Cacheable(value="biddingOnline", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<BiddingOnline> queryObjByPage(BiddingOnlineExample example) {
        PageView<BiddingOnline> pageView = example.getPageView();
        pageView.setQueryResult(this.biddingOnlineMapper.selectByExampleByPage(example));
        return pageView;
    }
    
    
	@Override
	public List<BiddingOnline> queryAllObj(BiddingOnlineExample example) {
		return this.biddingOnlineMapper.selectCompanyRankingByExample(example);
	}

}