package com.els.base.bidding.service.impl;

import java.util.Date;
import java.util.List;
import javax.annotation.Resource;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import com.els.base.auth.entity.User;
import com.els.base.bidding.dao.BiddingQuestionMapper;
import com.els.base.bidding.entity.BiddingQuestion;
import com.els.base.bidding.entity.BiddingQuestionExample;
import com.els.base.bidding.entity.BiddingSupplier;
import com.els.base.bidding.entity.BiddingSupplierExample;
import com.els.base.bidding.service.BiddingQuestionService;
import com.els.base.bidding.service.BiddingSupplierService;
import com.els.base.company.entity.Company;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;

@Service("defaultBiddingQuestionService")
public class BiddingQuestionServiceImpl implements BiddingQuestionService {
    @Resource
    protected BiddingQuestionMapper biddingQuestionMapper;
    
    @Resource
    protected BiddingSupplierService biddingSupplierService;

    @CacheEvict(value={"biddingQuestion"}, allEntries=true)
    @Override
    public void addObj(BiddingQuestion t) {
        this.biddingQuestionMapper.insertSelective(t);
    }

    @CacheEvict(value={"biddingQuestion"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.biddingQuestionMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"biddingQuestion"}, allEntries=true)
    @Override
    public void modifyObj(BiddingQuestion t) {
        Assert.isNotBlank(t.getId(), "id 为空，无法修改");
        this.biddingQuestionMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="biddingQuestion", keyGenerator="redisKeyGenerator")
    @Override
    public BiddingQuestion queryObjById(String id) {
        return this.biddingQuestionMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="biddingQuestion", keyGenerator="redisKeyGenerator")
    @Override
    public List<BiddingQuestion> queryAllObjByExample(BiddingQuestionExample example) {
        return this.biddingQuestionMapper.selectByExample(example);
    }

    @Cacheable(value="biddingQuestion", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<BiddingQuestion> queryObjByPage(BiddingQuestionExample example) {
        PageView<BiddingQuestion> pageView = example.getPageView();
        pageView.setQueryResult(this.biddingQuestionMapper.selectByExampleByPage(example));
        return pageView;
    }
    
    @CacheEvict(value={"biddingQuestion"}, allEntries=true)
	@Override
	public void createBiddingQuestion(BiddingQuestion question,Company supCompany,User supUser) {
		//验证前端提交的数据
    	this.isCreateLegalParmas(question);
    	
    	//设置提问内容的基本数据
    	this.initQuestionInfo(question, supCompany, supUser);
    	
    	this.addObj(question);
	}
    
    private void isCreateLegalParmas(BiddingQuestion question){
    	//Assert.isNotBlank(question.getBiddingId(), "招标单ID不能为空!");
    	Assert.isNotBlank(question.getBiddingNo(), "招标单单据号不能空!");
    	Assert.isNotBlank(question.getQuestionContent(), "提问内容不能为空!");
    }
    
    private void initQuestionInfo(BiddingQuestion question,Company supCompany,User supUser){
    	question.setSupCompanyId(supCompany.getId());
    	question.setSupCompanySrmCode(supCompany.getCompanyCode());
    	question.setSupCompanySapCode(supCompany.getCompanySapCode());
    	question.setSupCompanyFullName(supCompany.getCompanyFullName());
    	question.setSupCompanyName(supCompany.getCompanyName());
    	
    	//临时编码的设置
    	BiddingSupplierExample example  = new BiddingSupplierExample();
    	example.createCriteria()
    		   .andBiddingNoEqualTo(question.getBiddingNo())
    		   .andSupCompanyIdEqualTo(supCompany.getId());
    	List<BiddingSupplier> list = this.biddingSupplierService.queryAllObjByExample(example);
    	if(CollectionUtils.isNotEmpty(list)){
    		question.setSupCompanyTempCode(list.get(0).getSupplierTemporaryCode());
    	}
    	
    	//设置其他信息
    	question.setSupUserId(supUser.getId()); //提问者用户ID
    	
    	StringBuffer supUserName = new StringBuffer();
    	if(StringUtils.isNotEmpty(supUser.getLoginName())){
    		supUserName.append(supUser.getLoginName());
    	}
    	if(StringUtils.isNotEmpty(supUser.getNickName())){
    		supUserName.append("+");
    		supUserName.append(supUser.getNickName());
    	}
    	question.setSupUserName(supUserName.toString()); //提问者用户名
    	
    	question.setQuestionType(Constant.NO_INT); //私有问题
    	question.setIsSupCheck(Constant.NO_INT);  //未查看
    	question.setIsPurAnswer(Constant.NO_INT); //未回答
    	question.setIsEnable(Constant.YES_INT);  //未删除
    	question.setCreateTime(new Date());  //提问时间
    	
    }
    
    @CacheEvict(value={"biddingQuestion"}, allEntries=true)
	@Override
	public void deleteQuestion(List<String> ids) {
    	Assert.isNotNull(ids, "传递的ID不能为空");
    	
    	BiddingQuestionExample example = new BiddingQuestionExample();
    	example.createCriteria()
    		   .andIsPurAnswerEqualTo(Constant.YES_INT)
    		   .andIdIn(ids);
    	if(this.biddingQuestionMapper.countByExample(example) > 0){
    		throw new CommonException("您选择的数据中包含采购商已答复的问题,无法删除数据!");
    	}
    	
    	example.clear();
    	example.createCriteria()
    		   .andIdIn(ids);
    	
    	BiddingQuestion question = new BiddingQuestion();
    	question.setIsEnable(Constant.NO_INT);
    	question.setUpdateTime(new Date());
    	
    	this.biddingQuestionMapper.updateByExampleSelective(question, example);
    }
    
    @CacheEvict(value={"biddingQuestion"}, allEntries=true)
	@Override
	public BiddingQuestion findDetailById(String id,Company supCompany) {
    	Assert.isNotBlank(id, "id不能为空!");
    	BiddingQuestion question = this.queryObjById(id);
    	Assert.isNotNull(question, "查询的问题不存在!");
    	
    	//要针对提问题的供应商,只有他自己查看了,才算查看。避免出现当问题公开后,被其他供应商查看,问题就变成已查看的BUG
    	//也就是说：只有这条问题是你建的,到你查看的时候才有权限修改这个问题为已查看
    	if(supCompany.getId().equals(question.getSupCompanyId())){
    		if(question.getIsSupCheck() != null && Constant.NO_INT.equals(question.getIsSupCheck())){
        		if(question.getIsPurAnswer() != null && Constant.YES_INT.equals(question.getIsPurAnswer())){
            		//如果是供应商未查看,采购商已回复的,要修改状态
        			question.setIsSupCheck(Constant.YES_INT);
        			question.setUpdateTime(new Date());
        			
        			this.biddingQuestionMapper.updateByPrimaryKeySelective(question);
            	}
        	}
    	}
    	
		return question;
	}
    
    @CacheEvict(value={"biddingQuestion"}, allEntries=true)
	@Override
	public void editQuestionForSup(BiddingQuestion question) {
    	Assert.isNotNull(question, "问题不能为空!");
    	Assert.isNotBlank(question.getId(), "问题id不能为空!");
    	Assert.isNotBlank(question.getQuestionContent(), "问题内容不能为空!");
    	
    	BiddingQuestion bq = this.queryObjById(question.getId());
    	Assert.isNotNull(bq, "问题不存在!");
    	
    	if(Constant.NO_INT.equals(bq.getIsEnable())){
    		throw new CommonException("已删除问题不能修改!");
    	}
    	
    	if(Constant.YES_INT.equals(bq.getIsPurAnswer())){
    		throw new CommonException("已回复的问题不能修改!");
    	}
    	
    	BiddingQuestion data = new BiddingQuestion();
    	data.setId(question.getId());
    	data.setQuestionContent(question.getQuestionContent());
    	data.setUpdateTime(new Date());
    	
    	this.biddingQuestionMapper.updateByPrimaryKeySelective(data);
	}

	@Override
	public int countSupplierQuestionInfo(String biddingNo,Company supCompany) {
		BiddingQuestionExample example = new BiddingQuestionExample();
		example.createCriteria()
			   .andBiddingNoEqualTo(biddingNo)
			   .andSupCompanyIdEqualTo(supCompany.getId())
			   .andIsEnableEqualTo(Constant.YES_INT)
			   .andIsPurAnswerEqualTo(Constant.NO_INT);
		example.or()
			   .andBiddingNoEqualTo(biddingNo)
			   .andSupCompanyIdEqualTo(supCompany.getId())
			   .andIsEnableEqualTo(Constant.YES_INT)
			   .andIsPurAnswerEqualTo(Constant.YES_INT)
			   .andIsSupCheckEqualTo(Constant.NO_INT);
		int count = this.biddingQuestionMapper.countByExample(example);
		return count;
	}

	@Override
	public int countPurQuestionInfo(String biddingNo) {
		BiddingQuestionExample example = new BiddingQuestionExample();
		example.createCriteria()
			   .andBiddingNoEqualTo(biddingNo)
			   .andIsEnableEqualTo(Constant.YES_INT)
			   .andIsPurAnswerEqualTo(Constant.NO_INT);
		int count = this.biddingQuestionMapper.countByExample(example);
		return count;
	}
	
	@CacheEvict(value={"biddingQuestion"}, allEntries=true)
	@Override
	public void answerSupQuestion(BiddingQuestion question, User purUser) {
		Assert.isNotNull(question, "问题不能为空!");
		Assert.isNotBlank(question.getId(), "问题id不能为空!");
		Assert.isNotBlank(question.getAnswerContent(), "答复内容不能为空!");
		Assert.isNotNull(question.getQuestionType(),"是否公开问题字段不能为空!");
		
		BiddingQuestion ques = this.queryObjById(question.getId());
		if(null == ques){
			throw new CommonException("问题不存在!");
		}
		
		BiddingQuestion q = new BiddingQuestion();
		q.setId(question.getId());
		q.setAnswerContent(question.getAnswerContent()); //答复内容
		q.setAnswerTime(new Date()); //答复时间
		q.setIsPurAnswer(Constant.YES_INT); //设置为已回答
		q.setQuestionType(question.getQuestionType()); //公开、私有
		q.setIsSupCheck(Constant.NO_INT); //设置为未查看
		q.setUpdateTime(new Date());
		
		q.setPurUserId(purUser.getId()); //答复者id
		
		StringBuffer purUserName = new StringBuffer();
    	if(StringUtils.isNotEmpty(purUser.getLoginName())){
    		purUserName.append(purUser.getLoginName());
    	}
    	if(StringUtils.isNotEmpty(purUser.getNickName())){
    		purUserName.append("+");
    		purUserName.append(purUser.getNickName());
    	}
    	q.setPurUserName(purUserName.toString()); //答复者用户名
    	
    	this.biddingQuestionMapper.updateByPrimaryKeySelective(q);
	}
}