package com.els.base.bidding.web.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.entity.User;
import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.bidding.entity.BiddingHeaderExample;
import com.els.base.bidding.entity.BiddingOffer;
import com.els.base.bidding.entity.BiddingOfferExample;
import com.els.base.bidding.entity.vo.BiddingBaseVo;
import com.els.base.bidding.entity.vo.SaveBidInfoVo;
import com.els.base.bidding.entity.vo.TemporaryCodeReducedRate;
import com.els.base.bidding.service.BiddingHeaderService;
import com.els.base.bidding.service.BiddingOfferService;
import com.els.base.company.entity.Company;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.inquiry.IOrderItemDetail;
import com.els.base.inquiry.IOrderItemDetailService;
import com.els.base.inquiry.entity.PropertyValue;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.service.TplMouldDetailService;
import com.els.base.inquiry.service.TplOrderItemDetailService;
import com.els.base.inquiry.utils.json.DetailDeSerializer;
import com.els.base.inquiry.utils.json.DetailSerialzer;

@Api(value = "招投标-供应商报价")
@Controller
@RequestMapping("biddingOffer")
public class BiddingOfferController {
    @Resource
    protected BiddingOfferService biddingOfferService;

    @Resource
    protected BiddingHeaderService biddingHeaderService;
    
    @Resource
    protected TplOrderItemDetailService tplOrderItemDetailService;
    
    @Resource
    protected TplMouldDetailService tplMouldDetailService;

    @ApiOperation(httpMethod = "POST", value = "创建供应商报价")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody BiddingOffer biddingOffer) {
        this.biddingOfferService.addObj(biddingOffer);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "编辑供应商报价")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody BiddingOffer biddingOffer) {
        Assert.isNotBlank(biddingOffer.getId(), "id 为空，保存失败");
        this.biddingOfferService.modifyObj(biddingOffer);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "删除供应商报价")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required = true) String id) {
        Assert.isNotBlank(id, "删除失败,id不能为空");
        this.biddingOfferService.deleteObjById(id);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "查询供应商报价")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 BiddingOffer", paramType = "body", dataType = "QueryParamWapper") })
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BiddingOffer>> findByPage(@RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        BiddingOfferExample example = new BiddingOfferExample();
        example.setPageView(new PageView<BiddingOffer>(pageNo, pageSize));

        // BiddingOfferExample.Criteria criteria = example.createCriteria();
        // 填写具体的查询条件，例如
        // criteria.andIdEqualTo("1");
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }

        PageView<BiddingOffer> pageData = this.biddingOfferService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "评标界面")
    @RequestMapping("service/findBidInfo")
    @ResponseBody
    public ResponseResult<List<BiddingOffer>> findBidInfo(String biddingNo, String type, String isBidding,String currentRound) {
        List<BiddingOffer> biddingOffers = this.biddingOfferService.findBidInfo(biddingNo, type, isBidding,currentRound);
        return ResponseResult.success(biddingOffers);
    }

    @ApiOperation(httpMethod = "POST", value = "评标界面-供应商")
    @RequestMapping("service/findBidInfoSup")
    @ResponseBody
    public ResponseResult<List<BiddingOffer>> findBidInfoSup(String biddingNo, String type, String isBidding,String currentRound) {
    	 // 获取当前登录人公司信息
        Company company = CompanyUtils.currentCompany();
        // 获取当前登录人用户信息
        User user = SpringSecurityUtils.getLoginUser();
        
        List<BiddingOffer> biddingOffers = this.biddingOfferService.findBidInfoSup(biddingNo, type, isBidding,currentRound,user,company);
        return ResponseResult.success(biddingOffers);
    }
    
    @ApiOperation(httpMethod = "POST", value = "供应商报价保存")
    @RequestMapping("service/editUnitPrice")
    @ResponseBody
    public ResponseResult<BiddingBaseVo> editUnitPrice(@RequestBody BiddingBaseVo biddingBaseVo) throws InstantiationException, IllegalAccessException {

    	if (StringUtils.isBlank(biddingBaseVo.getBiddingHeader().getBiddingNo())) {
            throw new CommonException("列表数据为空，保存失败", "base_canot_be_null", "招标单号为空");
        }
        if (biddingBaseVo.getBiddingContent() == null) {
            throw new CommonException("列表数据为空，保存失败", "base_canot_be_null", "内容列表数据为空");
        }
        if (StringUtils.isBlank(biddingBaseVo.getHandleType())) {
        	throw new CommonException("编码不存在不能操作", "finish_status","操作类型为空(保存或投标)");
        }
        biddingBaseVo.setCompany(CompanyUtils.currentCompany());
        biddingBaseVo.setUser(SpringSecurityUtils.getLoginUser());
        // this.biddingHeaderService.saveBiddingInfo(biddingBaseVo);
        this.biddingOfferService.saveUnitPrice(biddingBaseVo);
        return ResponseResult.success(biddingBaseVo);
    }

    @ApiOperation(httpMethod="POST", value="计算不含税价格")
    @RequestMapping("service/calculateItemDetailValue")
    @ResponseBody
    public ResponseResult<List<PropertyValue>>  calculateItemDetailValue(@RequestParam(required=true) String costAnalysisId,  @RequestBody List<PropertyValue> propertyValueList) throws InstantiationException, IllegalAccessException{
    	
    	//TemplateConf templateConf = this.templateConfService.queryObjById(templateId);
    	//DetailDeSerializer<? extends IOrderItemDetail > serializer = new DetailDeSerializer<>(templateConf.getTplOrderItemDetail().getOrderItemDetailClass());
    	
    	//根据成本清单ID获取对应的service
    	//IOrderItemDetailService service = this.tplOrderItemDetailService.queryObjById(costAnalysisId).getOrderItemDetailService();
    	DetailDeSerializer<? extends IOrderItemDetail > serializer = new DetailDeSerializer<>(this.tplOrderItemDetailService.queryObjById(costAnalysisId).getOrderItemDetailClass());
    	
    	IOrderItemDetail itemDetail = serializer.transform(propertyValueList);
    	itemDetail.calculate();
    	//itemDetail.setTemplateId(templateId);
    	itemDetail.setTemplateId(costAnalysisId);
    	DetailSerialzer<IOrderItemDetail> serialzer = new DetailSerialzer<>();
    	return ResponseResult.success(serialzer.getProperyValueClass(itemDetail));
    }
    
    @ApiOperation(httpMethod = "POST", value = "保存(save)，提交(submit)评标结果")
    @RequestMapping("service/save")
    @ResponseBody
    public ResponseResult<String> save(@RequestBody SaveBidInfoVo saveBidInfoVo, String saveOrSubmit) {
        BiddingHeaderExample headerExample = new BiddingHeaderExample();
        headerExample.createCriteria().andBiddingNoEqualTo(
                saveBidInfoVo.getBiddingOfferList().get(0).getBiddingNo());
        // List<BiddingHeader> biddingHeaders =
        // this.biddingHeaderService.queryAllObjByExample(headerExample);
        // if(biddingHeaders.get(0).getBiddingStatus().equals("5")){
        // throw new CommonException("该标单号已评标，不可重复操作！", "is_submit");
        // }
        this.biddingOfferService.save(saveBidInfoVo, saveOrSubmit);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod = "POST", value = "查询中标的供应商和份额")
    @RequestMapping("service/findBidSupplier")
    @ResponseBody
    public ResponseResult<List<TemporaryCodeReducedRate>> findBidSupplier(String biddingNo, String type,String currentRound) {
        List<TemporaryCodeReducedRate> codeReducedRates = this.biddingOfferService.findBidSupplier(biddingNo, type,currentRound);
        return ResponseResult.success(codeReducedRates);
    }

}
