package com.els.base.bidding.web.controller;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.bidding.entity.BiddingHeader;
import com.els.base.bidding.entity.BiddingHeaderExample;
import com.els.base.bidding.entity.BiddingOnline;
import com.els.base.bidding.entity.BiddingOnlineExample;
import com.els.base.bidding.entity.vo.BiddingOfferRankingVO;
import com.els.base.bidding.service.BiddingHeaderService;
import com.els.base.bidding.service.BiddingOnlineService;
import com.els.base.company.entity.Company;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

import java.math.BigDecimal;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

@Api(value="招投标-在线竞价")
@Controller
@RequestMapping("biddingOnline")
public class BiddingOnlineController {
    @Resource
    protected BiddingOnlineService biddingOnlineService;
    
    @Resource
    protected BiddingHeaderService biddingHeaderService;

    @ApiOperation(httpMethod="POST", value="创建招投标-在线竞价")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody BiddingOnline biddingOnline) {
        this.biddingOnlineService.addObj(biddingOnline);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="供应商在线报价")
    @RequestMapping("service/quotation")
    @ResponseBody
    public ResponseResult<String> quotation(@RequestBody BiddingOnline biddingOnline) {
    	Assert.isNotNull(biddingOnline, "在线竞价供应商报价信息并不能为空");
    	Assert.isNotNull(biddingOnline.getUntaxedUnitPrice(), "未税单价不能为空");
    	Assert.isNotBlank(biddingOnline.getBiddingNo(), "招标单单号不能为空");
    	Assert.isNotBlank(biddingOnline.getMaterialCode(), "物料编码不能为空");
    	Assert.isNotBlank(biddingOnline.getCurrency(), "币种不能为空");
    	Assert.isNotBlank(biddingOnline.getTaxCode(), "税码不能为空");
    	Assert.isNotNull(biddingOnline.getPriceUnit(), "价格单位不能为空");
    	Assert.isNotNull(biddingOnline.getPriceEffectiveFrom(), "价格有效期从不能为空");
    	Assert.isNotNull(biddingOnline.getPriceEffectiveTo(), "价格有效期到不能为空");
    	this.biddingOnlineService.insert(CompanyUtils.currentCompany(),SpringSecurityUtils.getLoginUser(),biddingOnline);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="编辑招投标-在线竞价")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody BiddingOnline biddingOnline) {
        Assert.isNotBlank(biddingOnline.getId(), "id 为空，保存失败");
        this.biddingOnlineService.modifyObj(biddingOnline);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="删除招投标-在线竞价")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required=true) String id) {
        Assert.isNotBlank(id, "删除失败,id不能为空");
        this.biddingOnlineService.deleteObjById(id);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="批量作废供应商报价")
    @RequestMapping("service/batchInvalidation")
    @ResponseBody
    public ResponseResult<String> batchInvalidation(@RequestBody(required=true) List<String> ids) {
        Assert.isNotEmpty(ids, "作废失败,id不能为空");
        this.biddingOnlineService.batchInvalidationById(ids);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="中标操作供应商报价")
    @RequestMapping("service/winTheBidding")
    @ResponseBody
    public ResponseResult<String> winTheBidding(@RequestParam(required=true) String id) {
        Assert.isNotBlank(id, "操作失败,id不能为空");
        this.biddingOnlineService.winTheBiddingById(id);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="未中标操作供应商报价")
    @RequestMapping("service/noWinTheBidding")
    @ResponseBody
    public ResponseResult<String> noWinTheBidding(@RequestParam(required=true) String id) {
        Assert.isNotBlank(id, "操作失败,id不能为空");
        this.biddingOnlineService.noWinTheBiddingById(id);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="提交评标结果")
    @RequestMapping("service/submitBiddingResult")
    @ResponseBody
    public ResponseResult<String> submitBiddingResult(@RequestBody(required=true) List<BiddingOnline> list) {
        Assert.isNotEmpty(list, "提交的数据为空，操作失败");
        this.biddingOnlineService.submitResult(list);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="查询招投标-在线竞价（采购商）")
    @RequestMapping("service/findByPageForPur")
    @ResponseBody
    public ResponseResult<List<BiddingOnline>> findByPageForPur(
    		@RequestParam(required=true) String biddingNo,@RequestParam String biddingRoundsNumber) {
        BiddingOnlineExample example = new BiddingOnlineExample();
        example.setOrderByClause("OFFER_TIME DESC");
        BiddingOnlineExample.Criteria criteria = example.createCriteria();
        criteria.andBiddingNoEqualTo(biddingNo);
        if (StringUtils.isNotBlank(biddingRoundsNumber)) {
        	criteria.andBiddingRoundsNumberEqualTo(biddingRoundsNumber);
        }
        
        List<BiddingOnline> pageData = this.biddingOnlineService.queryAllObjByExample(example);
        return ResponseResult.success(pageData);
    }
    
    @ApiOperation(httpMethod="POST", value="查询招投标-在线竞价（供应商）")
    @ApiImplicitParams({ 
    	@ApiImplicitParam( name = "biddingNo",required = true,value = "招标编号", paramType = "query", dataType = "String"),
		@ApiImplicitParam( name = "pageNo",required = false,value = "所在页", paramType = "query", dataType = "String", defaultValue = "0" ),  
		@ApiImplicitParam( name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10" ),  
		@ApiImplicitParam( name = "wapper", required = false, value = "查询条件,属性名请参考 BiddingOnline", paramType = "body", dataType = "QueryParamWapper" )  
	}) 
    @RequestMapping("service/findByPageForSup")
    @ResponseBody
    public ResponseResult<PageView<BiddingOnline>> findByPageForSup( 
    	@RequestParam String biddingNo,
		@RequestParam(defaultValue="0") int pageNo,  
		@RequestParam(defaultValue="10") int pageSize, 
		@RequestBody(required=false) QueryParamWapper wapper) {
        BiddingOnlineExample example = new BiddingOnlineExample();
        example.setPageView(new PageView<BiddingOnline>(pageNo, pageSize));
        example.setOrderByClause("OFFER_TIME DESC");
        
        Company supCompany = CompanyUtils.currentCompany();
        String userId = SpringSecurityUtils.getLoginUserId();
        
        BiddingOnlineExample.Criteria criteria = example.createCriteria();
        criteria.andBiddingNoEqualTo(biddingNo);
        criteria.andSupCompanyIdEqualTo(supCompany.getId());
        criteria.andSupUserIdEqualTo(userId);
        //填写具体的查询条件，例如
        //criteria.andIdEqualTo("1");
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        
        PageView<BiddingOnline> pageData = this.biddingOnlineService.queryObjByPage(example);
        // 根据招标单号查询招标单基本信息
        BiddingHeaderExample headerExample = new BiddingHeaderExample();
        headerExample.createCriteria().andBiddingNoEqualTo(biddingNo);
        List<BiddingHeader> headerList = this.biddingHeaderService.queryAllObjByExample(headerExample);
        if (CollectionUtils.isNotEmpty(headerList)) {
        	// 判断此招标单是否通过定标审核（2是通过审核）,如果没有通过审核供应商看不到中标的信息
        	if (!"2".equals(headerList.get(0).getBiddingAuditstatus())) {
        		pageData.getQueryResult().forEach(online ->{
        			// 将中标状态变为有效竞价
        			if ("2".equals(online.getQuotationState())) {
        				online.setQuotationState("1");
        			}
        		});
        	}
        }
        return ResponseResult.success(pageData);
    }
    
    @ApiOperation(httpMethod="GET", value="采购商查询供应商报价排名")
    @RequestMapping("service/purFindSupCompanyRanking")
    @ResponseBody
    public ResponseResult<List<BiddingOnline>> purFindSupCompanyRanking(@RequestParam(required = true) String biddingNo){
    	// 根据招标单号查出全部供应商有效、中标报价,根据报价升序排序，报价相同时根据报价时间升序排
    	Assert.isNotBlank(biddingNo, "招标单号不能为空");
    	BiddingOnlineExample example = new BiddingOnlineExample();
    	example.setOrderByClause("untaxedUnitPrice ASC,offerTime ASC");
    	example.createCriteria().andBiddingNoEqualTo(biddingNo)
    		.andQuotationStateNotEqualTo("0");
    	List<BiddingOnline> list = this.biddingOnlineService.queryAllObj(example);
    	
    	return ResponseResult.success(list);
    }
    
    @ApiOperation(httpMethod="POST", value="供应商查询供应商报价排名")
    @RequestMapping("service/supFindSupCompanyRanking")
    @ResponseBody
    public ResponseResult<BiddingOfferRankingVO> supFindSupCompanyRanking(@RequestParam(required = true) String biddingNo){
    	// 根据招标单号查出全部供应商有效、中标报价,根据报价升序排序，报价相同时根据报价时间升序排
    	Assert.isNotBlank(biddingNo, "招标单号不能为空");
    	Company supCompany = CompanyUtils.currentCompany();
    	BiddingOnlineExample example = new BiddingOnlineExample();
    	example.setOrderByClause("untaxedUnitPrice ASC,offerTime ASC");
    	example.createCriteria().andBiddingNoEqualTo(biddingNo)
    		.andQuotationStateNotEqualTo("0");
    	List<BiddingOnline> list = this.biddingOnlineService.queryAllObj(example);
    	BiddingOfferRankingVO vo = new BiddingOfferRankingVO();
    	vo.setOnlineList(list);
    	// 获取当前供应商在众多供应商报价中的排名，如果报价相同，排名并列
    	int index = 1;
    	int temp = 0;
    	Set<BigDecimal> set = new HashSet<>();
    	if (CollectionUtils.isNotEmpty(list)) {
    		for (int i = 0; i < list.size(); i++) {
    			if (set.add(list.get(i).getUntaxedUnitPrice())) {
    				if (supCompany.getCompanyCode().equals(list.get(i).getSupCompanySrmCode())) {
    					vo.setRanking(String.valueOf(index));
    					vo.setLowestPrice(list.get(i).getUntaxedUnitPrice());
    					break;
    				}
    				temp = index;
    				index++;
    			} else {
    				if (supCompany.getCompanyCode().equals(list.get(i).getSupCompanySrmCode())) {
    					vo.setRanking(String.valueOf(temp));
    					vo.setLowestPrice(list.get(i).getUntaxedUnitPrice());
    					break;
    				}
    			}
    		}
    	} else {
    		vo.setRanking("");
    		vo.setLowestPrice(BigDecimal.ZERO);
    	}
    	return ResponseResult.success(vo);
    }
}