package com.els.base.bidding.web.controller;

import java.util.List;
import javax.annotation.Resource;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.bidding.entity.BiddingQuestion;
import com.els.base.bidding.entity.BiddingQuestionExample;
import com.els.base.bidding.service.BiddingQuestionService;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

@Api(value="招投标在线答疑")
@Controller
@RequestMapping("biddingQuestion")
public class BiddingQuestionController {
    @Resource
    protected BiddingQuestionService biddingQuestionService;

    @ApiOperation(httpMethod="POST", value="创建招投标在线答疑")
    @RequestMapping("service/sup/createBiddingQuestion")
    @ResponseBody
    public ResponseResult<String> createBiddingQuestion(@RequestBody BiddingQuestion question) {
        Assert.isNotNull(question, "问题不能为空!");
        this.biddingQuestionService.createBiddingQuestion(question, CompanyUtils.currentCompany(), SpringSecurityUtils.getLoginUser());
    	return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="删除在线答疑问题")
    @RequestMapping("service/sup/deleteQuestion")
    @ResponseBody
    public ResponseResult<String> deleteQuestion(@RequestBody List<String> ids) {
        this.biddingQuestionService.deleteQuestion(ids);
    	return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="GET", value="供应商查看问题详情")
    @RequestMapping("service/sup/findDetailById")
    @ResponseBody
    public ResponseResult<BiddingQuestion> findDetailByIdForSup(@RequestParam String id) {
        BiddingQuestion question = this.biddingQuestionService.findDetailById(id,CompanyUtils.currentCompany());
    	return ResponseResult.success(question);
    }
    
    @ApiOperation(httpMethod="GET", value="供应商查看红圈数量")
    @RequestMapping("service/sup/countSupplierQuestionInfo")
    @ResponseBody
    public ResponseResult<Integer> countSupplierQuestionInfo(@RequestParam String biddingNo) {
        int count = this.biddingQuestionService.countSupplierQuestionInfo(biddingNo,CompanyUtils.currentCompany());
    	return ResponseResult.success(count);
    }
    
    
    @ApiOperation(httpMethod="GET", value="采购方查看红圈数量")
    @RequestMapping("service/pur/countPurQuestionInfo")
    @ResponseBody
    public ResponseResult<Integer> countPurQuestionInfo(@RequestParam String biddingNo) {
        int count = this.biddingQuestionService.countPurQuestionInfo(biddingNo);
    	return ResponseResult.success(count);
    }
    
    
    @ApiOperation(httpMethod="GET", value="采购商查看问题详情")
    @RequestMapping("service/pur/findDetailById")
    @ResponseBody
    public ResponseResult<BiddingQuestion> findDetailByIdForPur(@RequestParam String id) {
    	BiddingQuestion question = this.biddingQuestionService.queryObjById(id);
    	return ResponseResult.success(question);
    }
    

    @ApiOperation(httpMethod="POST", value="编辑招投标在线答疑(供应商方)")
    @RequestMapping("service/sup/editQuestionForSup")
    @ResponseBody
    public ResponseResult<String> editQuestionForSup(@RequestBody BiddingQuestion question) {
        this.biddingQuestionService.editQuestionForSup(question);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="采购方答复供应商问题")
    @RequestMapping("service/pur/answerSupQuestion")
    @ResponseBody
    public ResponseResult<String> answerSupQuestion(@RequestBody BiddingQuestion question) {
        this.biddingQuestionService.answerSupQuestion(question,SpringSecurityUtils.getLoginUser());
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="查询招投标在线答疑(供应商方)")
    @RequestMapping("service/sup/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BiddingQuestion>> findByPageForSup( 
		@RequestParam(defaultValue="0") int pageNo,  
		@RequestParam(defaultValue="10") int pageSize, 
		@RequestParam String biddingNo) {
    	
    	Assert.isNotBlank(biddingNo, "招投标单号不能为空!");
    	
        BiddingQuestionExample example = new BiddingQuestionExample();
        example.setPageView(new PageView<BiddingQuestion>(pageNo, pageSize));
        example.setOrderByClause("CREATE_TIME DESC");
        
        BiddingQuestionExample.Criteria criteria = example.createCriteria();
        criteria.andIsEnableEqualTo(Constant.YES_INT)
        		.andSupCompanyIdEqualTo(CompanyUtils.currentCompanyId())
        		.andBiddingNoEqualTo(biddingNo);
        
        example.or()
        	   .andBiddingNoEqualTo(biddingNo)
        	   .andIsEnableEqualTo(Constant.YES_INT)
        	   .andQuestionTypeEqualTo(Constant.YES_INT);
        
        
        PageView<BiddingQuestion> pageData = this.biddingQuestionService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
    
    
    
    @ApiOperation(httpMethod="POST", value="查询招投标在线答疑(采购方)")
    @RequestMapping("service/pur/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BiddingQuestion>> findByPageForPur( 
		@RequestParam(defaultValue="0") int pageNo,  
		@RequestParam(defaultValue="10") int pageSize, 
		@RequestParam String biddingNo) {
    	
    	Assert.isNotBlank(biddingNo, "招投标单号不能为空!");
    	
        BiddingQuestionExample example = new BiddingQuestionExample();
        example.setPageView(new PageView<BiddingQuestion>(pageNo, pageSize));
        example.setOrderByClause("CREATE_TIME DESC");
        
        BiddingQuestionExample.Criteria criteria = example.createCriteria();
        criteria.andIsEnableEqualTo(Constant.YES_INT)
        		.andBiddingNoEqualTo(biddingNo);
        
        PageView<BiddingQuestion> pageData = this.biddingQuestionService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
}