package com.els.base.bill.command;

import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillItem;
import com.els.base.bill.utils.BillUtil;
import com.els.base.bill.utils.CalculateBillUtil;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.voucher.entity.BillVoucher;
import com.els.base.voucher.entity.BillVoucherExample;
import org.apache.commons.collections.CollectionUtils;

import java.util.Date;
import java.util.List;

/**
 *
 * 直接新建-通过开票清单新建
 * @author liuhf
 */
public class CreateCommand extends AbstractBillCommand<Bill> {

    private Bill bill;

    public CreateCommand() {
    }

    public CreateCommand(Bill bill) {
        this.bill = bill;
    }

    @Override
    public Bill execute(BillInvorker billInvorker) {
        this.billInvorker = billInvorker;

        //1、校验
        this.billInvorker.getBillService().prepareFilter(bill);
        this.billInvorker.getBillService().createBillFilter(bill);

        //关键业务：
        //前端传的bill里面包含了折让单和入退库凭证信息  2. 只传不要的，节省流量
        String nextCode = this.billInvorker.getGenerateCodeService().getNextCode("BILL_NO");
        bill.setBillNo(nextCode);

        //初始化数据
        this.init(bill);

        //保存开票清单头数据
        this.billInvorker.getBillService().addObj(bill);

        //查询改sap旗下的全部未对账凭证
        BillVoucherExample billVoucherExample = new BillVoucherExample();
        billVoucherExample.createCriteria()
                .andBillTypeIsNotNull()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andBillFlagEqualTo(Constant.NO_INT)
                .andSupCompanySapCodeEqualTo(bill.getSupCompanySapCode())
                .andPostingTimeBetween(bill.getPostingStartTime(), bill.getPostingEndTime());

        List<BillVoucher> myList = this.billInvorker.getBillVoucherService().queryAllObjByExample(billVoucherExample);
        Assert.isNotEmpty(myList, "未存在需要对账的开票凭证！");

        //前端上传的未勾选数据
        List<BillVoucher> otherList = bill.getBillVoucherList();

        //后台去除入退库凭证的信息==此处需要重写equals方法
        if (CollectionUtils.isNotEmpty(otherList)) {
            //把不要的数据remove掉
            myList.removeAll(otherList);
        }
        //去除之后，还剩多少条数据
        Assert.isNotEmpty(myList, "请选择一条以上的入退库凭证单据，谢谢！");

        //插入行数据
        this.insertBillItem(myList);

        //计算金额
        CalculateBillUtil.calculateAmount(bill, myList);
        //计算金额之后重新更新一下对账单抬头数据
        this.billInvorker.getBillService().modifyObj(bill);

        //更新凭证为已对账
        this.billInvorker.getBillVoucherService().switchBillVoucherFlag(myList, Constant.YES_INT);

        return null;
    }

    private void init(Bill bill) {

        //采购基本信息
        bill=BillUtil.setPurCompanyInfo(this.getProject(), bill, this.billInvorker.getCompanyService());

        //设置供应商基本信息
        bill= BillUtil.setSupCompanyInfo(bill.getSupCompanySapCode(), bill,this.billInvorker.getCompanyService());

        //制单时间
        bill.setCreateTime(new Date());

        //一开始默认为未过账
        bill.setInvoiceSignFlag(Constant.NO_INT);

    }

    private void insertBillItem(List<BillVoucher> myList) {
        //分组插入
        List<List<BillItem>> group = BillUtil.setBillItems(bill,myList);
        for (List<BillItem> list : group) {
            this.billInvorker.getBillItemService().insertBatch(list);
        }
    }

}
