package com.els.base.bill.command;

import com.els.base.billswitch.entity.BillSwitch;
import com.els.base.billswitch.entity.BillSwitchExample;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * 凭证日期统一进行校验
 *
 * @author liuhf
 */
public class PostTimeFilterCommand extends AbstractBillCommand<String> {

    Logger logger = LoggerFactory.getLogger(getClass());

    private String supCompanySapCode;

    /**
     * 用户传输的开始日期
     */
    private Date postingStartTime;

    /**
     * 用户传输的结束日期
     */
    private Date postingEndTime;

    public PostTimeFilterCommand(String supCompanySapCode, Date postingStartTime, Date postingEndTime) {
        this.supCompanySapCode = supCompanySapCode;
        this.postingStartTime = postingStartTime;
        this.postingEndTime = postingEndTime;
    }

    @Override
    public String execute(BillInvorker billInvorker) {

        this.billInvorker = billInvorker;

        this.validData(supCompanySapCode, postingStartTime, postingEndTime);

        return null;
    }

    private void validData(String supCompanySapCode, Date postingStartTime, Date postingEndTime) {
        Assert.isNotBlank(supCompanySapCode, "供应商SAP编码不能为空！");
        Assert.isNotNull(postingStartTime, "凭证开始日不能为空！");
        Assert.isNotNull(postingEndTime, "凭证结束日不能为空！");

        //时间校验
        if (postingStartTime.getTime() > postingEndTime.getTime()) {
            throw new CommonException("凭证开始日不能大于凭证结束日,请重新选择日期，谢谢！");
        }


        //根据供应商查询
        BillSwitchExample billSwitchExample = new BillSwitchExample();
        billSwitchExample.createCriteria()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andSupCompanySapCodeEqualTo(supCompanySapCode);
        List<BillSwitch> billSwitches = this.billInvorker.getBillSwitchService().queryAllObjByExample(billSwitchExample);

        //没有限制，默认为为放行
        if (CollectionUtils.isEmpty(billSwitches)) {
            return;
        }

        for (BillSwitch billSwitch : billSwitches) {

            if (billSwitch == null) {
                continue;
            }

            //有，但是没有开启，也放行
            if (billSwitch.getSwitchFlag().equals(Constant.NO_INT)) {
                continue;
            }

            //凭证开始时间
            Date voucherStartTime = billSwitch.getVoucherStartTime();
            //凭证结束时间
            Date voucherEndTime = billSwitch.getVoucherEndTime();

            //第一种情况：没有限制凭证日期,不做限制
            if (voucherStartTime == null && voucherEndTime == null) {
                continue;
            }

            //第二种情况：只维护了了开始日期,结束日期没有维护
            if (voucherStartTime != null && voucherEndTime == null) {
                continue;
            }

            //第三种情况：维护开始时间和结束时间
            if (voucherStartTime != null && voucherEndTime != null) {

                String str = DateFormatUtils.format(voucherStartTime, "yyyy-MM-dd");
                String str2 = DateFormatUtils.format(voucherEndTime, "yyyy-MM-dd");

                logger.info("用户传的凭证开始日:{}", DateFormatUtils.format(postingStartTime, "yyyy-MM-dd"));
                logger.info("用户传的凭证结束日:{}", DateFormatUtils.format(postingEndTime, "yyyy-MM-dd"));

                logger.info("采购设置的凭证开始日：{}", str);
                logger.info("采购设置的凭证结束日：{}", str2);

                //清除时间，只比较日期
                Date postingStartTime1 = DateUtils.truncate(postingStartTime, Calendar.DATE);
                Date postingEndTime1 = DateUtils.truncate(postingEndTime, Calendar.DATE);

                Date voucherStartTime1 = DateUtils.truncate(voucherStartTime, Calendar.DATE);
                Date voucherEndTime1 = DateUtils.truncate(voucherEndTime, Calendar.DATE);

                //凭证的选择时间是否在限制的范围内
                boolean b = postingStartTime1.getTime() >= voucherStartTime1.getTime();
                boolean b1 = postingEndTime1.getTime() <= voucherEndTime1.getTime();
                if (b && b1) {
                    continue;
                } else {
                    throw new CommonException("你所选的对账凭证时间不在时间规定范围内,允许时间为：" + str + "到" + str2);
                }
            }
        }
    }
}
