package com.els.base.bill.command;

import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillExample;
import com.els.base.bill.utils.BillMessageUtils;
import com.els.base.bill.utils.BillStatusConfirmEnum;
import com.els.base.bill.utils.CreateBillType;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 采购确认单据没有问题
 * @author liuhf
 */
public class PurConfirmCommand extends AbstractBillCommand<String> {

    Logger logger = LoggerFactory.getLogger(this.getClass());

    private List<Bill> billList;

    public PurConfirmCommand() {
    }

    public PurConfirmCommand(List<Bill> billList) {
        this.billList = billList;
    }

    @Override
    public String execute(BillInvorker billInvorker) {
        this.billInvorker = billInvorker;

        this.validate(billList);

        this.processor(billList);

        //发送消息
        this.sendMessage(billList);

        return null;
    }

    /**
     * 发送消息
     * @param billList
     */
    private void sendMessage(List<Bill> billList) {
        logger.info("开始发送消息");
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        billList=this.getBillInvorker().getBillService().queryAllObjByExample(billExample);

        for (Bill bill : billList) {
            Map<String, Object> msg = new HashMap<String, Object>();
            msg.put("billNo", bill.getBillNo());
            List<String> userIdList = this.getBillInvorker().getCompanyUserRefService().queryUserOfCompany(bill.getSupCompanyId());
            if (CollectionUtils.isNotEmpty(userIdList) && StringUtils.isNotBlank(userIdList.get(0))) {
                Message<Map<String, Object>> message = Message.init(msg)
                        .setCompanyCode(bill.getPurCompanySrmCode())// 发送者公司srmcode
                        .setSenderId(getPurUser().getId())// 发送者 userId
                        .setBusinessTypeCode(BillMessageUtils.BILL_PUR_CONFIRM)// 业务类型
                        .addReceiverId(userIdList.get(0))// 接受者 userId
                        .setMsgLevel(MessageLevelEnum.HIGH);// 消息等级
                MessageSendUtils.sendMessage(message);

                logger.info("发送内容，"+message);
            }
        }

        logger.info("结束发送消息");
    }

    private void processor(List<Bill> billList) {
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT);
        Bill bill = new Bill();
        bill.setPurConfirmStatus(BillStatusConfirmEnum.STATUS_CONFIRM.getConfirmStatus());
        this.billInvorker.getBillService().updateByExampleSelective(bill, billExample);
    }


    private void validate(List<Bill> billList) {

        //单据是否有效
        Assert.isNotEmpty(billList, "请勾选需要确认单据！");
        List<String> ids = new ArrayList<>();
        for (Bill bill : billList) {
            Assert.isNotNull(bill.getId(), "确认单据的ID不能为空！");
            ids.add(bill.getId());
        }

        //自己创建的不需要确认
        BillExample example = new BillExample();
        example.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andCreateBillTypeEqualTo(CreateBillType.PUR.getType());
        int count = this.billInvorker.getBillService().countByExample(example);
        if(count>0){
            throw new CommonException("所选单据中含有自己创建的单据，无需再确认单据，请重新选择！");
        }

        //已经确认的单据也不需要再确认
        BillExample billExample = new BillExample();
        billExample.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andPurConfirmStatusEqualTo(BillStatusConfirmEnum.STATUS_UNCONFIRM.getConfirmStatus());
        int count2 = this.billInvorker.getBillService().countByExample(billExample);
        if(billList.size()!=count2){
            throw new CommonException("所选单据中含有已确认的单据，无法进行确认单据，请重新选择！");
        }

    }
}
