package com.els.base.bill.command;

import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillItem;
import com.els.base.bill.utils.*;
import com.els.base.billswitch.entity.BillSwitch;
import com.els.base.billswitch.entity.BillSwitchExample;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.voucher.entity.BillVoucher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.List;

/**
 * 通过指定的可开票清单进行创建开票清单
 * * @author liuhf
 */
public class PurCreateCommand extends AbstractBillCommand<String> {

    Logger logger = LoggerFactory.getLogger(getClass());

    List<BillVoucher> billVoucherList;

    public PurCreateCommand(List<BillVoucher> billVoucherList) {
        this.billVoucherList = billVoucherList;
    }

    @Override
    public String execute(BillInvorker billInvorker) {
        this.billInvorker = billInvorker;
        //1、校验是否可以开票
        this.check(billVoucherList);

        //2、数据处理
        this.process(billVoucherList);

        //3、标记为已对账
        this.getBillInvorker().getBillVoucherService().switchBillVoucherFlag(billVoucherList, Constant.YES_INT);

        return null;
    }

    /**
     * 创建业务
     *
     * @param billVoucherList
     */
    private void process(List<BillVoucher> billVoucherList) {

        Bill bill = new Bill();

        //1、组装开票清单
        this.init(bill);

        //2、设置开票清单行
        this.setBillItem(bill, billVoucherList);
    }


    /**
     * 把凭证转成开票清单行
     *
     * @param bill
     * @param billVoucherList
     */
    private void setBillItem(Bill bill, List<BillVoucher> billVoucherList) {
        logger.info("开始设置对账单行信息");
        List<List<BillItem>> group = BillUtil.setBillItems(bill, billVoucherList);
        for (List<BillItem> list : group) {
            this.billInvorker.getBillItemService().insertBatch(list);
        }
    }

    private void init(Bill bill) {
        logger.info("开始初始化");
        //基础信息
        bill.setId(null);
        bill.setIsEnable(Constant.YES_INT);
        bill.setProjectId(getProject().getId());
        String nextCode = this.billInvorker.getGenerateCodeService().getNextCode("BILL_NO");
        bill.setBillNo(nextCode);

        //一开始默认为未过账
        bill.setInvoiceSignFlag(Constant.NO_INT);

        //状态
        bill.setIsEnable(Constant.YES_INT);
        bill.setSendSapFlag(Constant.NO_INT);

        //采购员新建单状态
        bill.setCreateBillType(CreateBillType.PUR.getType());
        bill.setPurSendStatus(BillSendStatus.NO_INT);
        bill.setPurConfirmBillStatus(Constant.NO_INT);//未确认发票
        bill.setSupConfirmStatus(BillStatusConfirmEnum.STATUS_UNCONFIRM.getConfirmStatus());//供应方为确认单据
        bill.setSupConfirmBillStatus(Constant.NO_INT);
        bill.setApproveStatus(ApproveStatusEnum.NOT_AUDITED.getCode());

        //其他状态无需操作
        bill.setSupSendStatus(BillSendStatus.NO);//供应商无需发送
        bill.setPurConfirmStatus(BillStatusConfirmEnum.NO.getConfirmStatus());//采购方无需确认单据

        //计算开票各个价格
        CalculateBillUtil.calculateAmount(bill, billVoucherList);

        //采购基本信息
        BillUtil.setPurCompanyInfo(this.getProject(), bill, this.billInvorker.getCompanyService());

        //设置供应商基本信息
        BillVoucher billVoucher = billVoucherList.get(0);//注意先初始化sap编码
        BillUtil.setSupCompanyInfo(billVoucher.getSupCompanySapCode(), bill, this.billInvorker.getCompanyService());

        //制单时间
        bill.setCreateTime(new Date());
        //制单人
        bill.setPurUserId(this.getPurUser().getId());
        bill.setPurUserName(this.getPurUser().getNickName());

        this.billInvorker.getBillService().addObj(bill);
    }

    /**
     * 校验是符合开票
     *
     * @param billVoucherList
     */
    private void check(List<BillVoucher> billVoucherList) {
        logger.info("开始检查");
        Assert.isNotEmpty(billVoucherList, "请选择要操作的物料凭证！");
        String supCompanySapCode = billVoucherList.get(0).getSupCompanySapCode();
        //是否在对账日期内
        BillSwitchExample billSwitchExample = new BillSwitchExample();
        billSwitchExample.createCriteria()
                .andSwitchFlagEqualTo(Constant.YES_INT)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andSupCompanySapCodeEqualTo(supCompanySapCode);
        List<BillSwitch> billSwitches = this.billInvorker.getBillSwitchService().queryAllObjByExample(billSwitchExample);

        //校验凭证是否在开票时间内
        for (BillVoucher billVoucher : billVoucherList) {
            BillUtil.checkBillTime(billSwitches,  billVoucher.getPostingTime());
        }
    }
}
