package com.els.base.bill.command;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.result.ExcelImportResult;
import com.els.base.bill.utils.BillUtil;
import com.els.base.billswitch.entity.BillSwitch;
import com.els.base.billswitch.entity.BillSwitchExample;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.company.entity.Company;
import com.els.base.company.entity.CompanyExample;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.voucher.entity.BillVoucher;
import com.els.base.voucher.entity.BillVoucherExample;
import com.els.base.voucher.vo.BillVoucherVo;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 供应商通过Excel导入凭证号和行号数据进行创建开票清单
 * * @author liuhf
 */
public class PurExcelImportCommand extends AbstractBillCommand<String> {

    Logger logger = LoggerFactory.getLogger(PurExcelImportCommand.class);

    private MultipartFile file;

    private Validator validator = Validation.buildDefaultValidatorFactory().getValidator();

    public PurExcelImportCommand(MultipartFile file) {
        this.file = file;
    }

    @Override
    public String execute(BillInvorker billInvorker) {
        this.billInvorker = billInvorker;
        //校验是否可以开票
        List<BillVoucher> billVoucherList = this.check(file);

        if (CollectionUtils.isNotEmpty(billVoucherList)) {
            PurCreateCommand cmd = new PurCreateCommand(billVoucherList);
            cmd.setProject(getProject());
            cmd.setPurCompany(getPurCompany());
            cmd.setPurUser(getPurUser());
            this.billInvorker.invoke(cmd);

        }else {
            throw new CommonException("没有可以操作的凭证，请检查！");
        }
        return null;
    }


    private List<BillVoucher> check(MultipartFile file) {

        Assert.isNotNull(file, "文件不能为空！");

        //数据合法性校验
        ExcelImportResult<BillVoucherVo> result = null;
        try {
            ImportParams importParams = new ImportParams();
            result = ExcelImportUtil.importExcelMore(file.getInputStream(), BillVoucherVo.class, importParams);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new CommonException("文件解析失败：请检查文件是否是加密文件，或者已经是损坏的文件！");
        }

        //校验多少行出现问题
        List<BillVoucherVo> list = result.getList();
        if (CollectionUtils.isEmpty(list)) {
            throw new CommonException("导入数据不能为空，请检查！");
        }

        BillVoucherVo temp = list.get(0);
        if (StringUtils.isBlank(temp.getSupCompanySapCode())) {
            throw new CommonException("供应商SAP编码不能为空！");
        }
        String tempSapCompanyCode = temp.getSupCompanySapCode().trim();
        CompanyExample companyExample = new CompanyExample();
        companyExample.createCriteria().andIsEnableEqualTo(Constant.YES_INT)
                .andCompanySapCodeEqualTo(tempSapCompanyCode);
        List<Company> companies = this.billInvorker.getCompanyService().queryAllObjByExample(companyExample);
        Assert.isNotEmpty(companies,"供应商SAP编码为"+tempSapCompanyCode+"不是SRM的用户，请检查！");
        Company supCompany = companies.get(0);
        Assert.isNotNull(supCompany,"供应商SAP编码为"+tempSapCompanyCode+"不是SRM的用户，请检查！");

        this.setSupCompany(supCompany);

        int index = 1;
        for (BillVoucherVo billVoucherVo : list) {

            //循环比较是否有不符合规范的数据
            index++;
            Set<ConstraintViolation<BillVoucherVo>> set = validator.validate(billVoucherVo);
            if (CollectionUtils.isNotEmpty(set)) {
                for (ConstraintViolation<BillVoucherVo> constraintViolation : set) {
                    throw new CommonException("请检查第" + index + "行的数据：" + constraintViolation.getMessage());
                }
            }

            //循环比较是否有错误的sap编码
            Assert.isNotBlank(billVoucherVo.getSupCompanySapCode(), "供应商SAP编码不能为空!");
            if (!tempSapCompanyCode.equals(billVoucherVo.getSupCompanySapCode().trim())) {
                throw new CommonException("请检查第" + index + "行的数据：" + "供应商SAP编码和第一行不一致");
            }
        }

        List<BillVoucherVo> successList = result.getList();
        List<BillVoucherVo> failList = result.getFailList();

        logger.info("是否存在验证未通过的数据:" + result.isVerfiyFail());
        logger.info("验证通过的数量:" + successList.size());
        logger.info("验证未通过的数量:" + failList.size());

        if (result.isVerfiyFail()) {
            StringBuffer stringBuffer = new StringBuffer();
            for (int i = 0; i < failList.size(); i++) {
                stringBuffer.append(i + ":物料凭证号为：" + failList.get(i).getMaterialVoucherNo() + "行号为：" + failList.get(i).getMaterialVoucherItemNo() + ";");
            }
            throw new CommonException("请检查：" + stringBuffer);
        }

        //是否在对账日期内
        BillSwitchExample billSwitchExample = new BillSwitchExample();
        billSwitchExample.createCriteria()
                .andSwitchFlagEqualTo(Constant.YES_INT)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andSupCompanySapCodeEqualTo(tempSapCompanyCode);
        List<BillSwitch> billSwitches = this.billInvorker.getBillSwitchService().queryAllObjByExample(billSwitchExample);

        //业务逻辑校验
        List<BillVoucher> billVoucherList = new ArrayList<>();
        for (BillVoucherVo billVoucherVo : successList) {
            BillVoucherExample billVoucherExample = new BillVoucherExample();
            billVoucherExample.createCriteria()
                    .andPurCompanyIdEqualTo(getPurCompany().getId())
                    .andSupCompanySapCodeEqualTo(tempSapCompanyCode)
                    .andIsEnableEqualTo(Constant.YES_INT)
                    .andMaterialVoucherNoEqualTo(billVoucherVo.getMaterialVoucherNo())
                    .andMaterialVoucherItemNoEqualTo(billVoucherVo.getMaterialVoucherItemNo());
            List<BillVoucher> billVouchers = this.billInvorker.getBillVoucherService().queryAllObjByExample(billVoucherExample);
            Assert.isNotEmpty(billVouchers, "物料凭证号：" + billVoucherVo.getMaterialVoucherNo() + ",物料凭证行号：" + billVoucherVo.getMaterialVoucherItemNo() + "不存在，请检查");

            BillVoucher billVoucher = billVouchers.get(0);

            //是否已经对过账
            if (Constant.YES_INT.equals(billVoucher.getBillFlag())) {
                throw new CommonException("物料凭证号：" + billVoucherVo.getMaterialVoucherNo() + ",物料凭证行号：" + billVoucherVo.getMaterialVoucherItemNo() + "已经对过账了，请检查");
            }

            //具体比较一下这些凭证是否在日期之内
            BillUtil.checkBillTime(billSwitches,billVoucher.getPostingTime());

            billVoucherList.add(billVoucher);
        }
        return billVoucherList;
    }

}
