package com.els.base.bill.command;

import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillExample;
import com.els.base.bill.utils.BillMessageUtils;
import com.els.base.bill.utils.BillSendStatus;
import com.els.base.bill.utils.CreateBillType;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author 采购发送
 */
public class PurSendCommand extends AbstractBillCommand<String> {

    Logger logger = LoggerFactory.getLogger(this.getClass());

    private List<Bill> billList;

    public PurSendCommand() {
    }

    public PurSendCommand(List<Bill> billList) {
        this.billList = billList;
    }

    @Override
    public String execute(BillInvorker billInvorker) {
        this.billInvorker = billInvorker;

        this.validate(billList);

        this.processor(billList);

        //发送消息
        this.sendMessage(billList);

        return null;
    }

    /**
     * 发送消息
     * @param billList
     */
    private void sendMessage(List<Bill> billList) {
        logger.info("开始发送消息");
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        billList=this.getBillInvorker().getBillService().queryAllObjByExample(billExample);

        for (Bill bill : billList) {
            Map<String, Object> msg = new HashMap<String, Object>();
            msg.put("billNo", bill.getBillNo());
            List<String> userIdList = this.getBillInvorker().getCompanyUserRefService().queryUserOfCompany(bill.getSupCompanyId());
            if (CollectionUtils.isNotEmpty(userIdList) && StringUtils.isNotBlank(userIdList.get(0))) {
                Message<Map<String, Object>> message = Message.init(msg)
                        .setCompanyCode(bill.getPurCompanySrmCode())// 发送者公司srmcode
                        .setSenderId(getPurUser().getId())// 发送者 userId
                        .setBusinessTypeCode(BillMessageUtils.BILL_PUR_SEND)// 业务类型
                        .addReceiverId(userIdList.get(0))// 接受者 userId
                        .setMsgLevel(MessageLevelEnum.HIGH);// 消息等级
                MessageSendUtils.sendMessage(message);

                logger.info("发送内容，"+message);
            }
        }

        logger.info("结束发送消息");
    }

    private void processor(List<Bill> billList) {
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        Bill bill = new Bill();
        bill.setPurSendStatus(BillSendStatus.YES_INT);
        this.billInvorker.getBillService().updateByExampleSelective(bill, billExample);
    }

    private void validate(List<Bill> billList) {

        //校验数据是否合法
        Assert.isNotEmpty(billList, "请勾选需要发送的单据！");
        List<String> ids = new ArrayList<>();
        for (Bill bill : billList) {
            Assert.isNotNull(bill.getId(), "发送的单据ID不能为空！");
            ids.add(bill.getId());
        }

        //业务逻辑

        //1、是否存在供应商创建的数据
        BillExample one = new BillExample();
        one.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andCreateBillTypeEqualTo(CreateBillType.SUP.getType());
        int count = this.billInvorker.getBillService().countByExample(one);
        if (count>0) {
            throw new CommonException("所选单据中含有供应商创建的单据，无法进行发送，请重新选择！");
        }

        //2、只能操作未发送的数据
        BillExample two = new BillExample();
        two.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andPurSendStatusEqualTo(Constant.NO_INT);

        //看看条数是否相等
        int count2 = this.billInvorker.getBillService().countByExample(two);
        if (count2!=billList.size()) {
            throw new CommonException("所选单据中含有已发送的单据，无法进行发送，请重新选择！");
        }
    }
}
