package com.els.base.bill.command;

import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillExample;
import com.els.base.bill.entity.BillItemExample;
import com.els.base.bill.utils.BillMessageUtils;
import com.els.base.bill.utils.BillSendStatus;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 采购商确认发票没有问题
 * @author liuhf
 */
public class PurSignCommand extends AbstractBillCommand<String> {

    Logger logger = LoggerFactory.getLogger(this.getClass());

    private List<Bill> billList;

    public PurSignCommand() {
    }

    public PurSignCommand(List<Bill> billList) {
        this.billList = billList;
    }

    @Override
    public String execute(BillInvorker billInvorker) {
        this.billInvorker = billInvorker;

        this.validate(billList);

        this.processor(billList);

        //发送消息
        this.sendMessage(billList);

        return null;
    }

    /**
     * 发送消息
     * @param billList
     */
    private void sendMessage(List<Bill> billList) {
        logger.info("开始发送消息");
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        billList=this.getBillInvorker().getBillService().queryAllObjByExample(billExample);

        for (Bill bill : billList) {
            Map<String, Object> msg = new HashMap<String, Object>();
            msg.put("billNo", bill.getBillNo());
            List<String> userIdList = this.getBillInvorker().getCompanyUserRefService().queryUserOfCompany(bill.getSupCompanyId());
            if (CollectionUtils.isNotEmpty(userIdList) && StringUtils.isNotBlank(userIdList.get(0))) {
                Message<Map<String, Object>> message = Message.init(msg)
                        .setCompanyCode(bill.getPurCompanySrmCode())// 发送者公司srmcode
                        .setSenderId(getPurUser().getId())// 发送者 userId
                        .setBusinessTypeCode(BillMessageUtils.BILL_PUR_SIGN)// 业务类型
                        .addReceiverId(userIdList.get(0))// 接受者 userId
                        .setMsgLevel(MessageLevelEnum.HIGH);// 消息等级
                MessageSendUtils.sendMessage(message);

                logger.info("发送内容，"+message);
            }
        }

        logger.info("结束发送消息");
    }

    private void processor(List<Bill> billList) {
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        Bill bill = new Bill();
        bill.setPurConfirmBillStatus(BillSendStatus.YES_INT);
        this.billInvorker.getBillService().updateByExampleSelective(bill, billExample);
    }


    private void validate(List<Bill> billList) {
        Assert.isNotEmpty(billList, "请勾选需要回签的单据！");

        List<String> ids = new ArrayList<>();
        for (Bill bill : billList) {
            Assert.isNotNull(bill.getId(), "回签的单据ID不能为空！");
            ids.add(bill.getId());
        }


        //前端传的只有ID，重新查询数据库
        BillExample billExample = new BillExample();
        billExample.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT);
        billList = this.billInvorker.getBillService().queryAllObjByExample(billExample);

        Assert.isNotEmpty(billList, "回签的单据不存在，请检查！");

        //判断一下，如果还有未对账的sap对账明细，那么不允许回签,为了能方便报出哪条没有填写完发票选择循环
        for (Bill bill : billList) {

            //已经确认的单据就不用重新确认了
            if (bill.getPurConfirmBillStatus().equals(Constant.YES_INT)) {
                throw new CommonException("所选单据中，单号为：" + bill.getBillNo() + "的单据中,含有已经回签的单据，无法重新回签，请重新选择！");
            }

            BillItemExample billItemExample = new BillItemExample();
            billItemExample.createCriteria()
                    .andIsEnableEqualTo(Constant.YES_INT)
                    .andBillIdEqualTo(bill.getId())
                    .andBillInvoiceFlagEqualTo(Constant.NO_INT);

            int i = this.billInvorker.getBillItemService().countByExample(billItemExample);
            if (i > 0) {
                throw new CommonException("所选单据中，单号为：" + bill.getBillNo() + "的单据中,含有未填写发票信息的单据，无法进行回签，请重新选择！");
            }
        }

    }
}
