package com.els.base.bill.command;

import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillExample;
import com.els.base.bill.service.GenPreInvoiceService;
import com.els.base.bill.utils.ApproveStatusEnum;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.utils.AbstractResultVO;
import com.els.base.utils.SpringContextHolder;
import org.apache.commons.collections.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 单独回写sap接口
 *
 * @author liuhf
 */
public class SendInvoiceCommand extends AbstractBillCommand<String> {

    private List<Bill> billList;

    public SendInvoiceCommand() {
    }

    public SendInvoiceCommand(List<Bill> billList) {
        this.billList = billList;
    }

    @Override
    public String execute(BillInvorker billInvorker) {

        this.billInvorker = billInvorker;

        billList = this.validate(billList);

        //写入sap
        GenPreInvoiceService genPreInvoiceService = SpringContextHolder.getOneBean(GenPreInvoiceService.class);
        for (Bill bill : billList) {
            AbstractResultVO abstractResultVO = genPreInvoiceService.writeBillInvoiceToSap(bill);
            if (abstractResultVO.getCode() == AbstractResultVO.CodeEnum.ERROR.getValue()) {
                throw new CommonException(abstractResultVO.getMessage());
            }
        }

        //修改sap同步标识符
        this.billInvorker.getBillService().updateSendSapFlag(billList, Constant.YES_INT);

        return null;
    }


    private List<Bill> validate(List<Bill> billList) {
        //检查是否为空
        Assert.isNotEmpty(billList, "请勾选需要发送的单据!");

        List<String> ids = new ArrayList<>();
        for (Bill bill : billList) {
            Assert.isNotBlank(bill.getId(), "开票清单ID不能为空，请检查！");
            ids.add(bill.getId());
        }

        //检验是否存在记录
        BillExample billExample = new BillExample();
        billExample.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT);
        List<Bill> list = this.billInvorker.getBillService().queryAllObjByExample(billExample);
        if (CollectionUtils.isEmpty(list)) {
            throw new CommonException("开票清单不存在,请检查！");
        }

        //没审核通过的不给回写
        for (Bill bill : list) {
            if (!bill.getApproveStatus().equals(ApproveStatusEnum.AUDITED.getCode())) {
                throw new CommonException("所选单据中含有未审核通过的单据，无法进行写入SAP，请重新选择！");
            }

            if (bill.getSendSapFlag().equals(Constant.YES_INT)) {
                throw new CommonException("所选单据中含有已写入的单据，无法进行写入SAP，请重新选择！");
            }
        }
        return list;
    }
}
