package com.els.base.bill.command;

import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillExample;
import com.els.base.bill.utils.BillMessageUtils;
import com.els.base.bill.utils.CreateBillType;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author 供应商发送
 */
public class SupSendCommand extends AbstractBillCommand<String> {

    Logger logger = LoggerFactory.getLogger(this.getClass());

    private List<Bill> billList;

    public SupSendCommand() {
    }

    public SupSendCommand(List<Bill> billList) {
        this.billList = billList;
    }

    @Override
    public String execute(BillInvorker billInvorker) {
        this.billInvorker = billInvorker;

        this.validate(billList);

        this.processor(billList);

        //发送消息
        this.sendMessage(billList);

        return null;
    }

    /**
     * 发送消息
     *
     * @param billList
     */
    private void sendMessage(List<Bill> billList) {
        logger.info("开始发送消息");
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        billList = this.getBillInvorker().getBillService().queryAllObjByExample(billExample);

        for (Bill bill : billList) {
            User user = this.getBillInvorker().getCompanyUserRefService().queryMainUserOfCompany(bill.getPurCompanyId());
            if(user!=null){
                //发送通知
                Message<Bill> message = Message.init(bill) //传入数据
                        .setBusinessTypeCode(BillMessageUtils.BILL_SUP_SEND) //业务编码
                        .setCompanyCode(getSupCompany().getCompanyCode()) //发送的企业srm编码
                        .setMsgLevel(MessageLevelEnum.HIGH) //消息等级
                        .setSenderId(getSupUser().getId()) //发送人的userid
                        .addReceiverId(user.getId()); //添加单个收件人 userid
                //.addReceiverIdList(new ArrayList<String>()); //添加多个收件人 userid
                MessageSendUtils.sendMessage(message);
                logger.info("发送内容，" + message);
            }
         }

        logger.info("结束发送消息");
    }

    private void processor(List<Bill> billList) {
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        Bill bill = new Bill();
        bill.setSupSendStatus(Constant.YES_INT);
        this.billInvorker.getBillService().updateByExampleSelective(bill, billExample);
    }

    private void validate(List<Bill> billList) {

        Assert.isNotEmpty(billList, "请勾选需要发送的单据！");
        List<String> ids = new ArrayList<>();
        for (Bill bill : billList) {
            Assert.isNotNull(bill.getId(), "发送的单据ID不能为空！");
            ids.add(bill.getId());
        }

        //业务逻辑

        //1、是否存在供应商创建的数据
        BillExample one = new BillExample();
        one.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andCreateBillTypeEqualTo(CreateBillType.PUR.getType());
        int count = this.billInvorker.getBillService().countByExample(one);
        if (count>0) {
            throw new CommonException("所选单据中含有采购商创建的单据，无法进行发送，请重新选择！");
        }

        //2、只能操作未发送的数据
        BillExample billExample = new BillExample();
        billExample.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andSupSendStatusEqualTo(Constant.NO_INT);
        int count2 = this.billInvorker.getBillService().countByExample(billExample);
        if (count2 != billList.size()) {
            throw new CommonException("所选单据中含有已发送的单据，无法进行发送，请重新选择！");
        }
    }
}
