package com.els.base.bill.command;

import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillExample;
import com.els.base.bill.entity.BillItemExample;
import com.els.base.bill.utils.BillMessageUtils;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.invoice.entity.BillInvoice;
import com.els.base.invoice.entity.BillInvoiceExample;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 供应商确认发票没有问题
 * @author liuhf
 */
public class SupSignCommand extends AbstractBillCommand<String> {

    Logger logger = LoggerFactory.getLogger(this.getClass());

    private List<Bill> billList;

    public SupSignCommand() {
    }

    public SupSignCommand(List<Bill> billList) {
        this.billList = billList;
    }

    @Override
    public String execute(BillInvorker billInvorker) {

        this.billInvorker = billInvorker;

        this.validate(billList);

        this.processor(billList);

        //发送消息
        this.sendMessage(billList);

        return null;
    }


    /**
     * 发送消息
     * @param billList
     */
    private void sendMessage(List<Bill> billList) {
        logger.info("开始发送消息");
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        billList=this.getBillInvorker().getBillService().queryAllObjByExample(billExample);

        for (Bill bill : billList) {
            User user = this.getBillInvorker().getCompanyUserRefService().queryMainUserOfCompany(bill.getPurCompanyId());
            if(user!=null){
                //发送通知
                Message<Bill> message = Message.init(bill) //传入数据
                        .setBusinessTypeCode(BillMessageUtils.BILL_SUP_SIGN) //业务编码
                        .setCompanyCode(getSupCompany().getCompanyCode()) //发送的企业srm编码
                        .setMsgLevel(MessageLevelEnum.HIGH) //消息等级
                        .setSenderId(getSupUser().getId()) //发送人的userid
                        .addReceiverId(user.getId()); //添加单个收件人 userid
                //.addReceiverIdList(new ArrayList<String>()); //添加多个收件人 userid
                MessageSendUtils.sendMessage(message);
                logger.info("发送内容，" + message);
            }
        }

        logger.info("结束发送消息");
    }

    private void processor(List<Bill> billList) {

        //更新对账单位回签状态
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        Bill bill = new Bill();
        bill.setSupConfirmBillStatus(Constant.YES_INT);
        this.billInvorker.getBillService().updateByExampleSelective(bill, billExample);

        //更新发票为已回签
        BillInvoiceExample billInvoiceExample = new BillInvoiceExample();
        billInvoiceExample.createCriteria().andBillIdIn(ids);
        int i = this.billInvorker.getBillInvoiceService().countByExample(billInvoiceExample);
        if (i > 0) {
            BillInvoice billInvoice = new BillInvoice();
            billInvoice.setInvoiceSignFlag(Constant.YES_INT);
            this.billInvorker.getBillInvoiceService().updateByExampleSelective(billInvoice, billInvoiceExample);
        }
    }


    private void validate(List<Bill> billList) {

        //1、校验是否为空
        Assert.isNotEmpty(billList, "请勾选需要回签的单据！");
        List<String> ids = new ArrayList<>();
        for (Bill bill : billList) {
            Assert.isNotNull(bill.getId(), "回签的单据ID不能为空！");
            ids.add(bill.getId());
        }

        //前端传的只有ID，重新查询数据库
        BillExample billExample = new BillExample();
        billExample.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT);
        billList = this.billInvorker.getBillService().queryAllObjByExample(billExample);

        Assert.isNotEmpty(billList, "回签的单据不存在，请检查！");


        //判断一下，如果还有未对账的sap对账明细，那么不允许回签,为了能方便报出哪条没有填写完发票选择循环
        for (Bill bill : billList) {

            //已经确认的单据就不用重新确认了
            if (bill.getSupConfirmBillStatus().equals(Constant.YES_INT)) {
                throw new CommonException("所选单据中，单号为：" + bill.getBillNo() + "的单据中,含有已经回签的单据，无法重新回签，请重新选择！");
            }

            BillItemExample billItemExample = new BillItemExample();
            billItemExample.createCriteria()
                    .andIsEnableEqualTo(Constant.YES_INT)
                    .andBillIdEqualTo(bill.getId())
                    .andBillInvoiceFlagEqualTo(Constant.NO_INT);

            int i = this.billInvorker.getBillItemService().countByExample(billItemExample);
            if (i > 0) {
                throw new CommonException("所选单据中，单号为：" + bill.getBillNo() + "的单据中,含有未填写发票信息的单据，无法进行回签，请重新选择！");
            }
        }

    }
}
