package com.els.base.bill.controller;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.bill.command.*;
import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillExample;
import com.els.base.bill.service.BillItemService;
import com.els.base.bill.service.BillService;
import com.els.base.common.BillInvorker;
import com.els.base.company.entity.Company;
import com.els.base.company.service.CompanyPurRefService;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.file.entity.FileData;
import com.els.base.invoice.entity.BillInvoice;
import com.els.base.invoice.service.BillInvoiceService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

@Api(value = "采购结算管理-开票清单数据")
@Controller
@RequestMapping("bill")
public class BillController {

    @Resource
    private BillService billService;
    @Resource
    private CompanyPurRefService companyPurRefService;
    @Resource
    private BillInvoiceService billInvoiceService;
    @Resource
    private BillInvorker invoker;

    @ApiOperation(httpMethod = "POST", value = "手动发送预制发票到sap")
    @RequestMapping("service/sendInvoiceToSap")
    @ResponseBody
    public ResponseResult<String> sendInvoiceToSap(@RequestBody(required = true) List<Bill> billList) {
        SendInvoiceCommand cmd = new SendInvoiceCommand(billList);
        cmd.setProject(ProjectUtils.getProject());
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(value = "导出对账清单数据", httpMethod = "POST")
    @RequestMapping(value = {"service/exportBill","service/pur/exportBill"})
    @ResponseBody
    public ResponseResult<FileData> exportBill(
            @RequestBody(required = false) QueryParamWapper wapper) throws IOException {
        ExportCommand cmd = new ExportCommand(wapper);
        cmd.setProject(ProjectUtils.getProject());
        cmd.setPurCompany(CompanyUtils.currentCompany());
        return ResponseResult.success(this.invoker.invoke(cmd));
    }

    @ApiOperation(value = "导出对账清单数据", httpMethod = "POST")
    @RequestMapping(value = "service/sup/exportBill")
    @ResponseBody
    public ResponseResult<FileData> supExportBill(
            @RequestBody(required = false) QueryParamWapper wapper) throws IOException {
        SupExportCommand cmd = new SupExportCommand(wapper);
        cmd.setProject(ProjectUtils.getProject());
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        return ResponseResult.success(this.invoker.invoke(cmd));
    }


    @ApiOperation(httpMethod = "POST", value = "创建采购结算管理-采购员创建开票清单预览数据")
    @RequestMapping("service/pur/prepare")
    @ResponseBody
    public ResponseResult<Bill> prepareForPur(@RequestBody Bill bill) {
        //创建预览数据
        PurPrepareCommand cmd = new PurPrepareCommand(bill);
        cmd.setProject(ProjectUtils.getProject());
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        Bill newBill = this.invoker.invoke(cmd);
        return ResponseResult.success(newBill);
    }

    @ApiOperation(httpMethod = "POST", value = "创建采购结算管理-供应商创建开票清单预览数据")
    @RequestMapping("service/sup/prepare")
    @ResponseBody
    public ResponseResult<Bill> prepareForSup(@RequestBody Bill bill) {
        SupPrepareCommand cmd = new SupPrepareCommand(bill);
        cmd.setProject(ProjectUtils.getProject());
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        Bill newBill = this.invoker.invoke(cmd);
        return ResponseResult.success(newBill);
    }

    @ApiOperation(httpMethod = "POST", value = "创建采购结算管理-创建开票清单数据")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody Bill bill) {
        CreateCommand cmd = new CreateCommand(bill);
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "删除采购结算管理-批量删除开票清单")
    @RequestMapping("service/deleteByIds")
    @ResponseBody
    public ResponseResult<String> deleteByIds(@RequestBody(required = true) List<Bill> billList) {
        DeleteCommand cmd = new DeleteCommand(billList);
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "GET", value = "采购结算管理-查询采购开票清单详情")
    @RequestMapping("service/findDetailById")
    @ResponseBody
    public ResponseResult<Bill> findDetailById(@RequestParam(required = true) String id) {
        Assert.isNotBlank(id, "操作失败,id不能为空");
        Bill bill = this.billService.queryObjById(id);
        return ResponseResult.success(bill);
    }


    @ApiOperation(httpMethod = "POST", value = "采购结算管理-采购员发送开票清单给供应商")
    @RequestMapping("service/pur/send")
    @ResponseBody
    public ResponseResult<String> sendForPur(@RequestBody(required = true) List<Bill> billList) {
        PurSendCommand cmd = new PurSendCommand(billList);
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "采购结算管理-供应商发送开票清单给采购员")
    @RequestMapping("service/sup/send")
    @ResponseBody
    public ResponseResult<String> sendForSup(@RequestBody(required = true) List<Bill> billList) {
        SupSendCommand cmd = new SupSendCommand(billList);
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "采购结算管理-供应商确认单据")
    @RequestMapping("service/sup/confirm")
    @ResponseBody
    public ResponseResult<String> confirmForSup(@RequestBody(required = true) List<Bill> billList) {
        SupConfirmCommand cmd = new SupConfirmCommand(billList);
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "采购结算管理-供应商编辑单据")
    @RequestMapping("service/sup/updateForSup")
    @ResponseBody
    public ResponseResult<String> updateForSup(@RequestBody(required = true) Bill bill) {
        SupUpdateCommand cmd = new SupUpdateCommand(bill);
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "采购结算管理-供应商回签，确认发票没有问题")
    @RequestMapping("service/sup/sign")
    @ResponseBody
    public ResponseResult<String> signForSup(@RequestBody(required = true) List<Bill> billList) {
        SupSignCommand cmd = new SupSignCommand(billList);
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "采购结算管理-采购员确认单据发票没有问题")
    @RequestMapping("service/pur/sign")
    @ResponseBody
    public ResponseResult<String> signForPur(@RequestBody(required = true) List<Bill> billList) {
        PurSignCommand cmd = new PurSignCommand(billList);
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "采购结算管理-采购员确认单据")
    @RequestMapping("service/pur/confirm")
    @ResponseBody
    public ResponseResult<String> confirmForPur(@RequestBody(required = true) List<Bill> billList) {
        PurConfirmCommand cmd = new PurConfirmCommand(billList);
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "采购结算管理-采购员拒绝单据")
    @RequestMapping("service/pur/reject")
    @ResponseBody
    public ResponseResult<String> rejectForPur(@RequestBody(required = true) List<Bill> billList) {

        PurRejectCommand cmd = new PurRejectCommand(billList);
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);

        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "采购结算管理-供应商拒绝单据")
    @RequestMapping("service/sup/reject")
    @ResponseBody
    public ResponseResult<String> rejectForSup(@RequestBody(required = true) List<Bill> billList) {
        SupRejectCommand cmd = new SupRejectCommand(billList);
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "采购结算管理-采购员作废单据")
    @RequestMapping("service/pur/abolish")
    @ResponseBody
    public ResponseResult<String> abolishForPur(@RequestBody(required = true) List<Bill> billList) {
        PurAbolishCommand cmd = new PurAbolishCommand(billList);
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "采购结算管理-供应商作废单据")
    @RequestMapping("service/sup/abolish")
    @ResponseBody
    public ResponseResult<String> abolishForSup(@RequestBody(required = true) List<Bill> billList) {
        SupAbolishCommand cmd = new SupAbolishCommand(billList);
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "采购结算管理-采购员提交审批")
    @RequestMapping("service/sendApprove")
    @ResponseBody
    public ResponseResult<String> sendApprove(@RequestBody(required = true) List<Bill> billList) {
        ApproveCommand cmd = new ApproveCommand(billList);
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        cmd.setProject(ProjectUtils.getProject());
        this.invoker.invoke(cmd);
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "供应商分页查询查询采购结算管理")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 Bill", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/sup/findByPage")
    @ResponseBody
    public ResponseResult<PageView<Bill>> findForSupByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        BillExample example = new BillExample();
        example.setPageView(new PageView<Bill>(pageNo, pageSize));
        example.setOrderByClause("CREATE_TIME DESC");

        BillExample.Criteria criteria = example.createCriteria();

        //供应商看不到采购未发送的单据
        criteria.andIsEnableEqualTo(Constant.YES_INT)
                .andProjectIdEqualTo(ProjectUtils.getProjectId())
                .andPurCompanyIdEqualTo(ProjectUtils.getCompanyId())
                .andPurSendStatusNotEqualTo(Constant.NO_INT)
                .andSupCompanySapCodeEqualTo(CompanyUtils.currentCompany().getCompanySapCode());

        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        PageView<Bill> pageData = this.billService.queryObjByPage(example);

        //TODO 待优化，涉及前端判断回签状态
        List<Bill> queryResult = this.getBillList(pageData.getQueryResult());
        pageData = pageData.setQueryResult(queryResult);

        return ResponseResult.success(pageData);
    }

    /**
     * 获取开票清单以及发票信息
     *
     * @param billList
     */
    private List<Bill> getBillList(List<Bill> billList) {
        List<Bill> billTempList = new ArrayList<>();
        if (CollectionUtils.isEmpty(billList)) {
            return billTempList;
        }
        for (Bill bill : billList) {
            List<BillInvoice> billInvoiceList = billInvoiceService.queryByBillId(bill.getId());
            bill.setInvoiceList(billInvoiceList);
            billTempList.add(bill);
        }
        return billTempList;
    }

    @ApiOperation(httpMethod = "POST", value = "采购员分页查询开票清单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 Bill", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/pur/findByPage")
    @ResponseBody
    public ResponseResult<PageView<Bill>> findForPurByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {

        //查询当前采购员下管理的所有的供应商的srm账号
        List<Company> companyList = this.companyPurRefService.queryAllCompanyByUserId(SpringSecurityUtils.getLoginUserId(), null);
        List<String> sapCompanyList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(companyList)) {
            for (Company c : companyList) {
                sapCompanyList.add(c.getCompanySapCode());
            }
        } else {
            PageView<Bill> billPageView = new PageView<>();
            billPageView.setQueryResult(new ArrayList<Bill>());
            return ResponseResult.success(billPageView);
        }

        BillExample example = new BillExample();
        example.setPageView(new PageView<Bill>(pageNo, pageSize));
        example.setOrderByClause("CREATE_TIME DESC");

        BillExample.Criteria criteria = example.createCriteria();

        //采购不能看到供应商未发送的数据
        criteria.andIsEnableEqualTo(Constant.YES_INT)
                .andProjectIdEqualTo(ProjectUtils.getProjectId())
                .andPurCompanyIdEqualTo(ProjectUtils.getCompanyId())
                .andSupSendStatusNotEqualTo(Constant.NO_INT)
                .andSupCompanySapCodeIn(sapCompanyList);

        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        PageView<Bill> pageData = this.billService.queryObjByPage(example);

        //TODO 待优化，涉及前端判断回签状态
        List<Bill> queryResult = this.getBillList(pageData.getQueryResult());
        pageData = pageData.setQueryResult(queryResult);

        return ResponseResult.success(pageData);
    }


    @ApiOperation(httpMethod = "POST", value = "供应商通过Excel导入凭证，新建开票清单")
    @PostMapping("service/sup/import")
    @ResponseBody
    public ResponseResult<String> importExcelForSup(@RequestParam(value = "file", required = true) MultipartFile file) {
        SupExcelImportCommand cmd = new SupExcelImportCommand(file);
        cmd.setProject(ProjectUtils.getProject());
        cmd.setSupCompany(CompanyUtils.currentCompany());
        cmd.setSupUser(SpringSecurityUtils.getLoginUser());
        this.invoker.invoke(cmd);
        return ResponseResult.success("创建成功，请到开票清单列表查看！");
    }


    @ApiOperation(httpMethod = "POST", value = "采购方通过Excel导入凭证，新建开票清单")
    @PostMapping("service/pur/import")
    @ResponseBody
    public ResponseResult<String> importExcelForPur(@RequestParam(value = "file", required = true) MultipartFile file) {
        PurExcelImportCommand cmd = new PurExcelImportCommand(file);
        cmd.setProject(ProjectUtils.getProject());
        cmd.setPurCompany(CompanyUtils.currentCompany());
        cmd.setPurUser(SpringSecurityUtils.getLoginUser());
        this.invoker.invoke(cmd);
        return ResponseResult.success("创建成功，请到开票清单列表查看！");
    }
}