package com.els.base.bill.controller;

import com.els.base.bill.command.ExportReportCommand;
import com.els.base.bill.entity.BillItem;
import com.els.base.bill.entity.BillItemExample;
import com.els.base.bill.entity.BillItemReport;
import com.els.base.bill.entity.BillItemReportExample;
import com.els.base.bill.service.BillItemService;
import com.els.base.common.BillInvorker;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.file.entity.FileData;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Api(value = "采购结算管理-开票清单行入退库凭证,折让单")
@Controller
@RequestMapping("billItem")
public class BillItemController {
    @Resource
    protected BillItemService billItemService;
    @Resource
    private BillInvorker invoker;
    @ApiOperation(httpMethod = "POST", value = "根据开票清单ID，查询对应的所有的折让数据")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 BillItem", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/rebate/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BillItem>> findForRebateByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        BillItemExample example = new BillItemExample();
        example.setPageView(new PageView<BillItem>(pageNo, pageSize));

        //添加默认排序：按照创建时间倒序排
        example.setOrderByClause("CREATE_TIME DESC");

        BillItemExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件，例如
        criteria.andIsEnableEqualTo(Constant.YES_INT)
                .andBillTypeIsNotNull()
                .andIsRebateEqualTo(Constant.YES_INT);
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        PageView<BillItem> pageData = this.billItemService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }


    @ApiOperation(httpMethod = "POST", value = "根据开票清单ID，查询对应的所有的sap对账明细数据")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 BillItem", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/voucher/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BillItem>> findForVoucherByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        BillItemExample example = new BillItemExample();
        example.setPageView(new PageView<BillItem>(pageNo, pageSize));

        //添加默认排序：按照创建时间倒序排
        example.setOrderByClause("CREATE_TIME DESC");

        BillItemExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件，例如
        criteria.andIsEnableEqualTo(Constant.YES_INT)
                //.andBillFlagEqualTo(Constant.NO_INT)
                .andBillTypeIsNotNull();

        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        PageView<BillItem> pageData = this.billItemService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }


    @ApiOperation(httpMethod = "POST", value = "开发票的查询接口，根据取消的sap对账凭证，查询其剩下所有未开发票的的折让单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "billId", required = true, paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "billInvoiceFlag", required = false, paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
    })
    @RequestMapping("service/rebate/reverse/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BillItem>> findReverseRebateByPage(
            @RequestParam(required = true) String billId,
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(required = false) String billInvoiceFlag,
            @RequestBody(required = false) List<BillItem> billItemList) {

        Assert.isNotBlank(billId, "对账ID不能为空！");

        //查询全部的折让数据出来，并且排除掉前面没有勾选的折让ID数据
        if (CollectionUtils.isNotEmpty(billItemList)) {

            List<String> ids = billItemList.stream().map(BillItem::getId).collect(Collectors.toList());

            BillItemExample example = new BillItemExample();
            example.setPageView(new PageView<BillItem>(pageNo, pageSize));

            //添加默认排序：按照创建时间倒序排
            example.setOrderByClause("CREATE_TIME DESC");

            BillItemExample.Criteria criteria = example.createCriteria();
            criteria.andIsEnableEqualTo(Constant.YES_INT)
                    .andBillIdEqualTo(billId)
                    .andIsRebateEqualTo(Constant.YES_INT)
                    .andBillTypeIsNotNull()
                    .andIdNotIn(ids);

            if (StringUtils.isNotBlank(billInvoiceFlag)) {
                if (!StringUtils.isNumeric(billInvoiceFlag)) {
                    throw new CommonException("传输参数billInvoiceFlag=" + billInvoiceFlag + "有误，请核对后再传！");
                }
                criteria.andBillInvoiceFlagEqualTo(Integer.valueOf(billInvoiceFlag));
            }
            PageView<BillItem> pageData = this.billItemService.queryObjByPage(example);
            return ResponseResult.success(pageData);
        } else {
            //如果没有传条件，说明了全部选中数据
            BillItemExample example = new BillItemExample();
            example.setPageView(new PageView<BillItem>(pageNo, pageSize));

            //添加默认排序：按照创建时间倒序排
            example.setOrderByClause("CREATE_TIME DESC");

            BillItemExample.Criteria criteria = example.createCriteria();
            criteria.andIsEnableEqualTo(Constant.YES_INT)
                    .andBillIdEqualTo(billId)
                    .andBillTypeIsNotNull().andIsRebateEqualTo(Constant.YES_INT);

            if (StringUtils.isNotBlank(billInvoiceFlag)) {
                if (!StringUtils.isNumeric(billInvoiceFlag)) {
                    throw new CommonException("传输参数billInvoiceFlag=" + billInvoiceFlag + "有误，请核对后再传！");
                }
                criteria.andBillInvoiceFlagEqualTo(Integer.valueOf(billInvoiceFlag));
            }

            PageView<BillItem> pageData = this.billItemService.queryObjByPage(example);
            return ResponseResult.success(pageData);
        }
    }


    @ApiOperation(httpMethod = "POST", value = "开发票的查询接口，根据选择的sap对账凭证，查询其对应的未开发票的的折让单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "billId", required = true, paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "billInvoiceFlag", required = false, paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
    })
    @RequestMapping("service/rebate/positive/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BillItem>> findPositiveRebateByPage(
            @RequestParam(required = true) String billId,
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(required = false) String billInvoiceFlag,
            @RequestBody(required = false) List<BillItem> billItemList) {

        //billItemList 正向，代表是前端传的是，选中的入退库凭证
        Assert.isNotBlank(billId, "对账ID不能为空！");

        if (CollectionUtils.isNotEmpty(billItemList)) {
            List<String> ids = billItemList.stream().map(BillItem::getId).collect(Collectors.toList());
            //查询条件
            BillItemExample example = new BillItemExample();
            example.setPageView(new PageView<BillItem>(pageNo, pageSize));

            //添加默认排序：按照创建时间倒序排
            example.setOrderByClause("CREATE_TIME DESC");

            BillItemExample.Criteria criteria = example.createCriteria();
            criteria.andIsEnableEqualTo(Constant.YES_INT)
                    //.andBillFlagEqualTo(Constant.NO_INT)
                    .andBillIdEqualTo(billId)
                    .andIdIn(ids)
                    .andBillTypeIsNotNull()
                    .andIsRebateEqualTo(Constant.YES_INT);

            if (StringUtils.isNotBlank(billInvoiceFlag)) {
                if (!StringUtils.isNumeric(billInvoiceFlag)) {
                    throw new CommonException("传输参数billInvoiceFlag=" + billInvoiceFlag + "有误，请核对后再传！");
                }
                criteria.andBillInvoiceFlagEqualTo(Integer.valueOf(billInvoiceFlag));
            }

            PageView<BillItem> pageData = this.billItemService.queryObjByPage(example);
            return ResponseResult.success(pageData);
        } else {
            PageView<BillItem> pageData = new PageView<BillItem>();
            pageData.setQueryResult(new ArrayList<BillItem>());
            return ResponseResult.success(pageData);
        }
    }



    @ApiOperation(httpMethod = "POST", value = "采购结算核票清单报表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 BillItem", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/findBillItemReportByPage")
    @ResponseBody
    public ResponseResult<PageView<BillItemReport>> findBillItemReportByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        BillItemReportExample example = new BillItemReportExample();
        example.setPageView(new PageView<BillItemReport>(pageNo, pageSize));

        //添加默认排序：按照创建时间倒序排
        example.setOrderByClause("CREATE_TIME DESC");

        BillItemReportExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件，例如
        criteria.andIsEnableEqualTo(Constant.YES_INT)
                //.andBillFlagEqualTo(Constant.NO_INT)
                .andBillTypeIsNotNull();

        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        PageView<BillItemReport> pageData = this.billItemService.queryBillItemReportByPage(example);
        return ResponseResult.success(pageData);
    }


    @ApiOperation(value = "导出对账清单数据", httpMethod = "POST")
    @RequestMapping(value = {"service/exportBillItemReport","service/pur/exportBillItemReport"})
    @ResponseBody
    public ResponseResult<FileData> exportBillItemReport(
            @RequestBody(required = false) QueryParamWapper wapper) throws IOException {
        ExportReportCommand cmd = new ExportReportCommand(wapper);
        cmd.setProject(ProjectUtils.getProject());
        cmd.setPurCompany(CompanyUtils.currentCompany());
        return ResponseResult.success(this.invoker.invoke(cmd));
    }

}