package com.els.base.bill.service.impl;

import com.els.base.bill.dao.BillMapper;
import com.els.base.bill.entity.Bill;
import com.els.base.bill.entity.BillExample;
import com.els.base.bill.entity.BillItem;
import com.els.base.bill.entity.BillItemExample;
import com.els.base.bill.service.BillItemService;
import com.els.base.bill.service.BillService;
import com.els.base.bill.service.GenPreInvoiceService;
import com.els.base.bill.utils.ApproveStatusEnum;
import com.els.base.billswitch.entity.BillSwitch;
import com.els.base.billswitch.entity.BillSwitchExample;
import com.els.base.billswitch.service.BillSwitchService;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.invoice.service.BillInvoiceService;
import com.els.base.utils.AbstractResultVO;
import com.els.base.utils.SpringContextHolder;
import com.els.base.voucher.entity.BillVoucher;
import com.els.base.voucher.entity.BillVoucherExample;
import com.els.base.voucher.service.BillVoucherService;
import com.els.base.workflow.common.event.TaskOperateEvent;
import com.els.base.workflow.common.service.ITaskListener;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

@Service("defaultBillService")
public class BillServiceImpl implements BillService, ITaskListener {

    Logger logger = LoggerFactory.getLogger(BillServiceImpl.class);
    @Resource
    protected BillMapper billMapper;
    @Resource
    protected BillVoucherService billVoucherService;
    @Resource
    protected BillItemService billItemService;
    @Resource
    private BillSwitchService billSwitchService;
    @Resource
    private BillInvoiceService billInvoiceService;

    @CacheEvict(value = {"bill"}, allEntries = true)
    @Transactional
    @Override
    public void deleteByExample(BillExample example) {
        Assert.isNotEmpty(example.getOredCriteria(),"不能进行全表删除，请检查！");
        this.billMapper.deleteByExample(example);
    }

    @CacheEvict(value = {"bill"}, allEntries = true)
    @Override
    public int updateByExampleSelective(Bill record, BillExample example) {
        Assert.isNotNull(record);
        return this.billMapper.updateByExampleSelective(record, example);
    }

    @Cacheable(value = "bill", keyGenerator = "redisKeyGenerator")
    @Override
    public int countByExample(BillExample example) {
        return this.billMapper.countByExample(example);
    }

    @CacheEvict(value = {"bill"}, allEntries = true)
    @Transactional
    @Override
    public int updateSendSapFlag(List<Bill> billList, Integer flag) {
        Assert.isNotEmpty(billList, "对账单不能为空！");
        Assert.isNotNull(flag, "标识符不能为空！");
        List<String> ids = billList.stream().map(Bill::getId).collect(Collectors.toList());
        BillExample billExample = new BillExample();
        billExample.createCriteria().andIdIn(ids);
        Bill bill = new Bill();
        bill.setSendSapFlag(flag);
        int i = 0;
        if (this.billMapper.countByExample(billExample) > 0) {
            i = this.billMapper.updateByExampleSelective(bill, billExample);
        }
        return i;
    }

    /**
     * 主要是控制供应商什么时间可以对账
     */
    @Override
    public void createBillFilter(Bill bill) {

        BillSwitchExample billSwitchExample = new BillSwitchExample();
        billSwitchExample.createCriteria()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andSupCompanySapCodeEqualTo(bill.getSupCompanySapCode())
                .andSwitchFlagEqualTo(Constant.YES_INT);

        List<BillSwitch> billSwitches = this.billSwitchService.queryAllObjByExample(billSwitchExample);

        if (CollectionUtils.isNotEmpty(billSwitches)) {
            for (BillSwitch billSwitch : billSwitches) {
                //第一步，该供应商是否有做对账限制，如果没有则放行
                if (CollectionUtils.isEmpty(billSwitches)) {
                    continue;
                }

                //第二步，如果有做对账限制，那么是否在允许的时间范围之内
                Assert.isNotNull(billSwitch);
                Integer switchFlag = billSwitch.getSwitchFlag();
                Assert.isNotNull(switchFlag, "开账时间标记不能为空！");

                //没有开启对账限制
                if (switchFlag.equals(Constant.NO_INT)) {
                    continue;
                }

                Integer startTime = billSwitch.getStartTime();
                Integer endTime = billSwitch.getEndTime();

                Assert.isNotNull(startTime, "开票开始时间不能为空！");
                Assert.isNotNull(endTime, "开票结束时间不能为空！");

                logger.info("采购员设置的开票开始时间为：{}", startTime);
                logger.info("采购员设置的开票结束时间为：{}", endTime);

                //开始和结束日期  1号 - 31 号
                Calendar calendar = Calendar.getInstance();
                int date = calendar.get(Calendar.DATE);
                logger.info("系统当前天数为：{}", date);

                //如果在开票范围内，则放行
                if (date >= startTime && date <= endTime) {

                } else {
                    throw new CommonException("不在对账时间内,请联系采购员设置对账时间，谢谢！");
                }
            }
        }
    }


    /**
     * 校验对账凭证时间
     *
     * @param bill
     */
    @Override
    public void prepareFilter(Bill bill) {

        //数据的基本校验
        Assert.isNotNull(bill, "参数不能为空！");

        String supCompanySapCode = bill.getSupCompanySapCode();
        Date postingStartTime = bill.getPostingStartTime();
        Date postingEndTime = bill.getPostingEndTime();

        Assert.isNotBlank(supCompanySapCode, "供应商SAP编码不能为空！");
        Assert.isNotNull(postingStartTime, "凭证开始日不能为空！");
        Assert.isNotNull(postingEndTime, "凭证结束日不能为空！");


        //时间校验
        if (bill.getPostingStartTime().getTime() > bill.getPostingEndTime().getTime()) {
            throw new CommonException("凭证开始日不能大于凭证结束日,请重新选择日期，谢谢！");
        }

        //根据供应商查询
        this.checkBillSwitch(supCompanySapCode, postingStartTime, postingEndTime);

    }

    /**
     * 根据供应商和开始时间和结束时间
     *
     * @param supCompanySapCode
     * @param postingStartTime
     * @param postingEndTime
     */
    private void checkBillSwitch(String supCompanySapCode, Date postingStartTime, Date postingEndTime) {
        BillSwitchExample billSwitchExample = new BillSwitchExample();
        billSwitchExample.createCriteria()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andSupCompanySapCodeEqualTo(supCompanySapCode);
        List<BillSwitch> billSwitches = this.billSwitchService.queryAllObjByExample(billSwitchExample);

        if (CollectionUtils.isNotEmpty(billSwitches)) {
            for (BillSwitch billSwitch : billSwitches) {

                //没有限制，默认为为放行
                if (CollectionUtils.isEmpty(billSwitches)) {
                    continue;
                }

                if (billSwitch == null) {
                    continue;
                }

                //有，但是没有开启，也放行
                if (billSwitch.getSwitchFlag().equals(Constant.NO_INT)) {
                    continue;
                }

                //凭证开始时间
                Date voucherStartTime = billSwitch.getVoucherStartTime();
                //凭证结束时间
                Date voucherEndTime = billSwitch.getVoucherEndTime();

                //第一种情况：没有限制凭证日期,不做限制
                if (voucherStartTime == null && voucherEndTime == null) {
                    continue;
                }

                //第二种情况：只维护了了开始日期,结束日期没有维护
                if (voucherStartTime != null && voucherEndTime == null) {
                    //return;
                    continue;
                }

                //第三种情况：维护开始时间和结束时间
                if (voucherStartTime != null && voucherEndTime != null) {

                    String str = DateFormatUtils.format(voucherStartTime, "yyyy-MM-dd");
                    String str2 = DateFormatUtils.format(voucherEndTime, "yyyy-MM-dd");

                    logger.info("用户传的凭证开始日:{}", DateFormatUtils.format(postingStartTime, "yyyy-MM-dd"));
                    logger.info("用户传的凭证结束日:{}", DateFormatUtils.format(postingEndTime, "yyyy-MM-dd"));

                    logger.info("采购设置的凭证开始日：{}", str);
                    logger.info("采购设置的凭证结束日：{}", str2);

                    //清除时间，只比较日期
                    Date postingStartTime1 = DateUtils.truncate(postingStartTime, Calendar.DATE);
                    Date postingEndTime1 = DateUtils.truncate(postingEndTime, Calendar.DATE);

                    Date voucherStartTime1 = DateUtils.truncate(voucherStartTime, Calendar.DATE);
                    Date voucherEndTime1 = DateUtils.truncate(voucherEndTime, Calendar.DATE);

                    //凭证的选择时间是否在限制的范围内
                    boolean b = postingStartTime1.getTime() >= voucherStartTime1.getTime();
                    boolean b1 = postingEndTime1.getTime() <= voucherEndTime1.getTime();
                    if (b && b1) {

                    } else {
                        throw new CommonException("你所选的对账凭证时间不在时间规定范围内,允许时间为：" + str + "到" + str2);
                    }
                }
            }
        }
    }

    @CacheEvict(value = {"bill"}, allEntries = true)
    @Transactional
    @Override
    public void listen(TaskOperateEvent event) {

        if (event == null) {
            return;
        }

        //获取业务ID
        String businessId = event.getBusinessId();
        if (StringUtils.isBlank(businessId)) {
            return;
        }
        Bill bill = this.billMapper.selectByPrimaryKey(businessId);
        if (bill == null) {
            return;
        }

        //1、审批流完成而且通过
        if (event.isFinished() && event.isPass()) {

            try {
                //做修改业务操作
                bill.setApproveTime(new Date());
                bill.setApproveStatus(ApproveStatusEnum.AUDITED.getCode());

                //更新sap同步状态
                bill.setSendSapFlag(Constant.YES_INT);
                bill.setInvoiceSignFlag(Constant.YES_INT);
                this.billMapper.updateByPrimaryKeySelective(bill);

                //对账需要更新状态
                this.updateBillVoucher(bill, Constant.YES_INT);

                //传预制发票到SAP中
                GenPreInvoiceService genPreInvoiceService = SpringContextHolder.getOneBean(GenPreInvoiceService.class);
                AbstractResultVO abstractResultVO = genPreInvoiceService.writeBillInvoiceToSap(bill);
                if (abstractResultVO.getCode() == AbstractResultVO.CodeEnum.ERROR.getValue()) {
                    logger.info("往SAP写入发票失败，失败原因为:{}", abstractResultVO.getMessage());
                    throw new CommonException(abstractResultVO.getMessage());
                }

            } catch (Exception e) {
                e.printStackTrace();
                logger.info("往SAP写入发票失败，失败原因为:{}", e.getMessage());
                throw new CommonException(e.getMessage());
            }

            //2、审批流完成但被驳回
        } else if (event.isFinished() && !event.isPass()) {
            bill.setApproveTime(new Date());
            bill.setApproveStatus(ApproveStatusEnum.DISMISSAL.getCode());
            this.billMapper.updateByPrimaryKeySelective(bill);

            //对账需要更新状态
            this.updateBillVoucher(bill, Constant.NO_INT);

            //释放对账发票
            List<String> billIdList = new ArrayList<>();
            billIdList.add(businessId);
            this.billInvoiceService.updateBillInvoiceFlag(billIdList, Constant.NO_INT);

            //部分审核的时候
        } else {

        }


    }

    /**
     * 更新对账凭证行信息
     *
     * @param bill
     * @param billFlag
     */
    private void updateBillVoucher(Bill bill, Integer billFlag) {
        BillItemExample billItemExample = new BillItemExample();
        billItemExample.createCriteria()
                .andBillIdEqualTo(bill.getId())
                .andIsEnableEqualTo(Constant.YES_INT);
        List<BillItem> billItems = this.billItemService.queryAllObjByExample(billItemExample);

        Assert.isNotEmpty(billItems, "操作失败，对账凭证行不存在！");

        for (BillItem billItem : billItems) {
            BillVoucher voucher = new BillVoucher();
            voucher.setId(billItem.getVoucherId());
            voucher.setBillFlag(billFlag);
            this.billVoucherService.modifyObj(voucher);
        }
    }


    /**
     * 根据对账id更新对账凭证为未对账状态，已对账状态
     *
     * @param ids 对账id
     */
    @Override
    public void updateVoucherByBillIds(List<String> ids, int billFlag) {

        //查找行
        BillItemExample billItemExample = new BillItemExample();
        billItemExample.createCriteria()
                .andBillIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT);
        List<BillItem> billItems = this.billItemService.queryAllObjByExample(billItemExample);
        if (CollectionUtils.isEmpty(billItems)) {
            return;
        }

        //拿到对账清单凭证IDs
        List<String> voucherIds = new ArrayList<>();
        for (BillItem billItem : billItems) {
            voucherIds.add(billItem.getVoucherId());
        }
        if (CollectionUtils.isEmpty(voucherIds)) {
            return;
        }

        //更新对账凭证状态
        BillVoucherExample billVoucherExample = new BillVoucherExample();
        billVoucherExample.createCriteria().andIdIn(voucherIds);
        int i = this.billVoucherService.countByExample(billVoucherExample);
        if (i > 0) {
            BillVoucher billVoucher = new BillVoucher();
            billVoucher.setBillFlag(billFlag);
            this.billVoucherService.updateByExampleSelective(billVoucher, billVoucherExample);
        }

    }

    @CacheEvict(value = {"bill"}, allEntries = true)
    @Override
    public void addObj(Bill t) {
        this.billMapper.insertSelective(t);
    }

    @CacheEvict(value = {"bill"}, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.billMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = {"bill"}, allEntries = true)
    @Override
    public void modifyObj(Bill t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.billMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "bill", keyGenerator = "redisKeyGenerator")
    @Override
    public Bill queryObjById(String id) {
        return this.billMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "bill", keyGenerator = "redisKeyGenerator")
    @Override
    public List<Bill> queryAllObjByExample(BillExample example) {
        return this.billMapper.selectByExample(example);
    }

    @Cacheable(value = "bill", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<Bill> queryObjByPage(BillExample example) {
        PageView<Bill> pageView = example.getPageView();
        pageView.setQueryResult(this.billMapper.selectByExampleByPage(example));
        return pageView;
    }
}