package com.els.base.categoryref.command;

import com.els.base.billswitch.entity.BillSwitch;
import com.els.base.billswitch.entity.BillSwitchExample;
import com.els.base.category.entity.CompanyCategory;
import com.els.base.category.entity.CompanyCategoryExample;
import com.els.base.categoryref.entity.CompanyCategoryRef;
import com.els.base.categoryref.entity.CompanyCategoryRefExample;
import com.els.base.categoryswitch.entity.CategoryBillSwitch;
import com.els.base.categoryswitch.entity.CategoryBillSwitchExample;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.company.entity.Company;
import com.els.base.company.entity.CompanyExample;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.utils.uuid.UUIDGenerator;
import org.apache.commons.collections.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

public class AppointCommand extends AbstractBillCommand<String> {

    /**
     * 分组号
     */
    String companyCategoryNo;

    /**
     * 供应商列表
     */
    List<Company> companyList;


    public AppointCommand(String companyCategoryNo, List<Company> companyList) {
        this.companyCategoryNo = companyCategoryNo;
        this.companyList = companyList;
    }

    @Override
    public String execute(BillInvorker billInvorker) {

        this.billInvorker = billInvorker;

        //1、检查是否合法
        this.check(companyCategoryNo, companyList);

        //2、处理业务数据
        this.process(companyCategoryNo, companyList);

        return null;
    }

    /**
     * 检验数据是否合法
     *
     * @param companyCategoryNo
     * @param companyList
     */
    private void check(String companyCategoryNo, List<Company> companyList) {
        Assert.isNotBlank(companyCategoryNo, "分组编码不能为空！");
        CompanyCategoryExample companyCategoryExample = new CompanyCategoryExample();
        companyCategoryExample.createCriteria()
                .andCategoryNoEqualTo(companyCategoryNo)
                .andIsEnableEqualTo(Constant.YES_INT);
        int i = this.billInvorker.getCompanyCategoryService().countByExample(companyCategoryExample);
        if (i==0) {
           throw new CommonException("分组编码不存在，请检查！");
        }

        Assert.isNotEmpty(companyList, "分配的公司不能为空！");
        for (Company company : companyList) {
            Assert.isNotBlank(company.getCompanySapCode(),"供应商SAP编码不能为空");
        }
    }


    /**
     * 处理业务
     *
     * @param companyCategoryNo
     * @param companyList
     */
    private void process(String companyCategoryNo, List<Company> companyList) {

        //重新查询,避免前端有些值没有传上来
        List<String> companySapCode = companyList.stream().map(Company::getCompanySapCode).collect(Collectors.toList());
        CompanyExample example = new CompanyExample();
        example.createCriteria().andCompanySapCodeIn(companySapCode);
        companyList = this.billInvorker.getCompanyService().queryAllObjByExample(example);
        Assert.isNotEmpty(companyList,"没有对应的供应商，请检查！");

        //1、把指定的供应商列表关联关系删除，然后重新插入
        this.updateCompanyCategoryRef(companyCategoryNo, companyList);

        //2、获取改组对应的对账组限制信息，然后赋值到新的供应商
        this.updateCategoryBillSwitch(companyCategoryNo, companyList);

    }

    /**
     * 处理对账分组和对账具体限制信息
     *
     * @param companyCategoryNo
     * @param companyList
     */
    private void updateCategoryBillSwitch(String companyCategoryNo, List<Company> companyList) {

        //1、如果该供应商分组没有建立对应的开票限制组，那么不用进行操作
        CategoryBillSwitchExample categoryBillSwitchExample = new CategoryBillSwitchExample();
        categoryBillSwitchExample.createCriteria()
                .andCategoryNoEqualTo(companyCategoryNo)
                .andIsEnableEqualTo(Constant.YES_INT);
        List<CategoryBillSwitch> categoryBillSwitches = this.billInvorker.getCategoryBillSwitchService().queryAllObjByExample(categoryBillSwitchExample);
        if (CollectionUtils.isEmpty(categoryBillSwitches)) {
            return;
        }

        //2、更新具体的对账限制
        this.updateBillSwitch(categoryBillSwitches, companyList);

    }

    private void updateBillSwitch(List<CategoryBillSwitch> categoryBillSwitches, List<Company> companyList) {

        CategoryBillSwitch categoryBillSwitch = categoryBillSwitches.get(0);

        BillSwitchExample billSwitchExample = new BillSwitchExample();
        billSwitchExample.createCriteria().andCategoryNoEqualTo(categoryBillSwitch.getCategoryNo());
        this.billInvorker.getBillSwitchService().deleteByExample(billSwitchExample);

        List<BillSwitch> billSwitches = new ArrayList<>();
        for (Company company : companyList) {

            BillSwitch billSwitch = new BillSwitch();
            //基本信息
            billSwitch.setId(UUIDGenerator.generateUUID());
            billSwitch.setIsEnable(Constant.YES_INT);
            billSwitch.setCreateTime(new Date());
            billSwitch.setUpdateTime(new Date());

            billSwitch.setSwitchFlag(categoryBillSwitch.getSwitchFlag());
            billSwitch.setProjectId(categoryBillSwitch.getProjectId());

            //采购基本信息
            billSwitch.setPurCompanyFullName(categoryBillSwitch.getPurCompanyFullName());
            billSwitch.setPurCompanyId(categoryBillSwitch.getPurCompanyId());
            billSwitch.setPurCompanyName(categoryBillSwitch.getPurCompanyName());
            billSwitch.setPurCompanySapCode(categoryBillSwitch.getPurCompanySapCode());
            billSwitch.setPurCompanySrmCode(categoryBillSwitch.getPurCompanySrmCode());
            billSwitch.setPurUserId(categoryBillSwitch.getPurUserId());
            billSwitch.setPurUserName(categoryBillSwitch.getPurUserName());

            //供应商基本信息
            billSwitch.setSupCompanyFullName(company.getCompanyFullName());
            billSwitch.setSupCompanyId(company.getId());
            billSwitch.setSupCompanyName(company.getCompanyName());
            billSwitch.setSupCompanySapCode(company.getCompanySapCode());
            billSwitch.setSupCompanySrmCode(company.getCompanyCode());

            //分组关联信息
            billSwitch.setCategoryNo(categoryBillSwitch.getCategoryNo());
            billSwitch.setCategoryName(categoryBillSwitch.getCategoryName());

            //对账时间信息
            billSwitch.setStartTime(categoryBillSwitch.getStartTime());
            billSwitch.setEndTime(categoryBillSwitch.getEndTime());
            billSwitch.setVoucherStartTime(categoryBillSwitch.getVoucherStartTime());
            billSwitch.setVoucherEndTime(categoryBillSwitch.getVoucherEndTime());

            billSwitches.add(billSwitch);
        }

        this.billInvorker.getBillSwitchService().addAll(billSwitches);
    }

    private void updateCompanyCategoryRef(String companyCategoryNo, List<Company> companyList) {
        //删除该组已经存在的信息
        this.deleteCompanyCategoryRef(companyCategoryNo);

        //插入新的数据
        this.addCompanyCategoryRef(companyCategoryNo, companyList);
    }

    private void addCompanyCategoryRef(String companyCategoryNo, List<Company> companyList) {
        List<CompanyCategoryRef> companyCategoryRefs = new ArrayList<>();
        for (Company supCompany : companyList) {
            CompanyCategoryRef categoryRef = new CompanyCategoryRef();

            categoryRef.setId(UUIDGenerator.generateUUID());
            categoryRef.setIsEnable(Constant.YES_INT);
            categoryRef.setProjectId(getProject().getId());

            //采购商
            categoryRef.setPurCompanyId(getProject().getCompanyId());
            categoryRef.setPurCompanyFullName(getPurCompany().getCompanyFullName());
            categoryRef.setPurCompanySapCode(getPurCompany().getCompanySapCode());
            categoryRef.setPurCompanyName(getPurCompany().getCompanyName());
            categoryRef.setPurCompanySrmCode(getPurCompany().getCompanyCode());

            //用户
            categoryRef.setPurUserId(getPurUser().getId());
            categoryRef.setPurUserName(getPurUser().getNickName());

            //供应商
            categoryRef.setSupCompanyId(supCompany.getId());
            categoryRef.setSupCompanySrmCode(supCompany.getCompanyCode());
            categoryRef.setSupCompanySapCode(supCompany.getCompanySapCode());
            categoryRef.setSupCompanyName(supCompany.getCompanyName());
            categoryRef.setSupCompanyFullName(supCompany.getCompanyFullName());

            //关联关系
            categoryRef.setCategoryNo(companyCategoryNo);

            CompanyCategory companyCategory = this.billInvorker.getCompanyCategoryService().queryCompanyCategoryByCategory(companyCategoryNo);
            if (companyCategory != null) {
                categoryRef.setCategoryName(companyCategory.getCategoryName());
            }
            companyCategoryRefs.add(categoryRef);
        }
        this.billInvorker.getCompanyCategoryRefService().addAll(companyCategoryRefs);
    }

    private void deleteCompanyCategoryRef(String companyCategoryNo) {
        CompanyCategoryRefExample companyCategoryRefExample = new CompanyCategoryRefExample();
        companyCategoryRefExample.createCriteria()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andCategoryNoEqualTo(companyCategoryNo);
        this.billInvorker.getCompanyCategoryRefService().deleteByExample(companyCategoryRefExample);
    }
}
