package com.els.base.invoice.command;

import com.els.base.bill.entity.BillItem;
import com.els.base.bill.entity.BillItemExample;
import com.els.base.bill.utils.CalculateBillUtil;
import com.els.base.common.AbstractBillCommand;
import com.els.base.common.BillInvorker;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.invoice.entity.BillInvoice;
import com.els.base.invoice.entity.BillInvoiceExample;
import com.els.base.rebate.entity.BillInvoiceRebate;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;

/**
 * 创建发票
 * @author liuhf
 */
public class CreateCommand extends AbstractBillCommand<BillInvoice> {

    Logger logger = LoggerFactory.getLogger(this.getClass());

    /**
     * 发票行数默认不能超过200行
     */
    private static final int BILL_INVOICE_SIZE=200;


    private BillInvoice billInvoice;

    public CreateCommand() {
    }

    public CreateCommand(BillInvoice billInvoice) {
        this.billInvoice = billInvoice;
    }

    @Override
    public BillInvoice execute(BillInvorker billInvorker) {
        this.billInvorker = billInvorker;

        //创建之前需要检查
        this.validate(billInvoice);

        this.processor(billInvoice);

        return null;
    }

    private void processor(BillInvoice billInvoice) {

        //入退库凭证
        List<BillItem> billItemList = billInvoice.getBillVoucherList();
        Assert.isNotEmpty(billItemList,"入退库凭证不能为空！");

        BillItem billItem = billItemList.get(0);
        billInvoice.setInvoiceFlag(Constant.YES_INT);//默认开始是可以使用的
        billInvoice.setTaxRate(billItem.getTaxRate());
        billInvoice.setTaxCode(billItem.getTaxCode());
        billInvoice.setCurrencyType(billItem.getCurrencyType());
        billInvoice.setCreateTime(new Date());

        //默认是未回签的发票，采购这边暂时不可见
        billInvoice.setInvoiceSignFlag(Constant.NO_INT);

        CalculateBillUtil.calculateInvoiceAmount(billInvoice, billItemList);
        this.billInvorker.getBillInvoiceService().addObj(billInvoice);

        List<String> ids = billItemList.stream().map(BillItem::getId).collect(Collectors.toList());
        BillItem temp =new BillItem();
        temp.setBillInvoiceId(billInvoice.getId());
        temp.setBillInvoiceFlag(Constant.YES_INT);
        BillItemExample  billItemExample =new BillItemExample();
        billItemExample.createCriteria().andIdIn(ids);
        this.billInvorker.getBillItemService().updateByExampleSelective(temp,billItemExample);

        //查出折让单
        BillItemExample  rebateExample =new BillItemExample();
        rebateExample.createCriteria()
                .andIdIn(ids)
                .andIsRebateEqualTo(Constant.YES_INT);
        List<BillItem> list = this.billInvorker.getBillItemService().queryAllObjByExample(rebateExample);

        //插入折让汇总数据
        List<BillItem> billItemsByJava8 = this.getBillItemsByJava8(list);
        if(CollectionUtils.isEmpty(billItemsByJava8)){
            return;
        }
        logger.info("汇总后的折让单行：{}",billItemsByJava8.toString());
        List<BillInvoiceRebate> billInvoiceRebates = new ArrayList<>();
        for (BillItem tempBillItem : billItemsByJava8) {
            BillInvoiceRebate billInvoiceRebate =new BillInvoiceRebate();
            BeanUtils.copyProperties(tempBillItem,billInvoiceRebate);
            billInvoiceRebate.setId(null);
            billInvoiceRebate.setDbCrInd("H");
            billInvoiceRebates.add(billInvoiceRebate);
        }
        this.billInvorker.getBillInvoiceRebateService().addAll(billInvoiceRebates);
    }



    /**
     * 校验数据
     * @param billInvoice
     */
    private void validate(BillInvoice billInvoice) {

        Assert.isNotNull(billInvoice,"发票信息不能为空！");
        Assert.isNotNull(billInvoice.getBillId(),"开票清单ID不能为空！");
        Assert.isNotBlank(billInvoice.getInvoiceNo(),"发票编码不能为空！");
        Assert.isNotNull(billInvoice.getInvoiceTime(),"发票日期不能为空！");

        List<BillItem> billItemList = billInvoice.getBillVoucherList();
        Assert.isNotEmpty(billItemList,"请选择要开发票的数据！");

        if(billItemList.size()>BILL_INVOICE_SIZE){
            throw  new CommonException("发票行数不能超过200行，请检查！");
        }

        //发票编码校验
        int length = StringUtils.trim(billInvoice.getInvoiceNo()).length();
        if(length>18){
            throw  new CommonException("发票编码不能超过18个字符，请检查！");
        }

        //发票号的验证
        BillInvoiceExample billInvoiceExample =new BillInvoiceExample();
        billInvoiceExample.createCriteria()
                .andInvoiceFlagEqualTo(Constant.YES_INT)//作废，拒绝后需要释放这个标识符
                .andInvoiceNoEqualTo(billInvoice.getInvoiceNo());
        int i = this.billInvorker.getBillInvoiceService().countByExample(billInvoiceExample);
        if(i>0){
            throw  new CommonException("发票号已经存在了，请检查是否存在重开！");
        }

        //检查是否已经开票的验证
        List<String> ids = billItemList.stream().map(BillItem::getId).collect(Collectors.toList());
        BillItemExample billItemExample =new BillItemExample();
        billItemExample.createCriteria()
                .andIdIn(ids)
                .andIsEnableEqualTo(Constant.YES_INT)
                .andBillInvoiceFlagEqualTo(Constant.YES_INT);
        int i2 = this.billInvorker.getBillItemService().countByExample(billItemExample);
        if(i2>0){
            throw  new CommonException("存在已开票的单据，请检查！");
        }

    }

    /**
     * 通过java来实现
     * @param billItems
     * @return
     */
    private List<BillItem> getBillItemsByJava8(List<BillItem> billItems) {

        List<BillItem> newBillList =new ArrayList<>();

        //分组
        Map<String, List<BillItem>> listMap = billItems.stream().collect(groupingBy(p -> p.getMaterialNo()+p.getFactory(), toList()));
        logger.info("根据物料号和工厂分组信息为：{}",listMap);

        //每个组进行统计
        Iterator<Map.Entry<String, List<BillItem>>> it = listMap.entrySet().iterator();
        while (it.hasNext()){
            Map.Entry<String,  List<BillItem>> entry = it.next();
            List<BillItem> value = entry.getValue();
            //汇总
            BillItem billItem = value.get(0);

            //数量
            BigDecimal materialQuantity = value.stream().map(BillItem::getMaterialQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
            billItem.setMaterialQuantity(materialQuantity);

            //统计金额
            BigDecimal orderAmount = value.stream().map(BillItem::getOrderAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            billItem.setOrderAmount(orderAmount);

            //开票未税金额
            BigDecimal notIncludeTaxAmount = value.stream().map(BillItem::getNotIncludeTaxAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            billItem.setNotIncludeTaxAmount(notIncludeTaxAmount);
            //开票税额
            BigDecimal taxAmount = value.stream().map(BillItem::getTaxAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            billItem.setTaxAmount(taxAmount);
            //开票含税金额
            BigDecimal includeTaxAmount = value.stream().map(BillItem::getIncludeTaxAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            billItem.setIncludeTaxAmount(includeTaxAmount);

            //折让未税金额
            BigDecimal rebateNotIncludeTaxAmount = value.stream().map(BillItem::getRebateNotIncludeTaxAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            billItem.setRebateNotIncludeTaxAmount(rebateNotIncludeTaxAmount);
            //折让税额
            BigDecimal rebateTaxAmount = value.stream().map(BillItem::getRebateTaxAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            billItem.setRebateTaxAmount(rebateTaxAmount);
            //折让含税金额
            BigDecimal rebateIncludeTaxAmount = value.stream().map(BillItem::getRebateIncludeTaxAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            billItem.setRebateIncludeTaxAmount(rebateIncludeTaxAmount);

            newBillList.add(billItem);
        }
        return newBillList;
    }

}
