package com.els.base.payment.controller;

import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.common.BillInvorker;
import com.els.base.company.entity.Company;
import com.els.base.company.service.CompanyPurRefService;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.payment.command.ImportCommand;
import com.els.base.payment.entity.BillPayment;
import com.els.base.payment.entity.BillPaymentExample;
import com.els.base.payment.service.BillPaymentService;
import com.els.base.payment.vo.BillPaymentVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Api(value = "采购结算管理-付款明细")
@Controller
@RequestMapping("billPayment")
public class BillPaymentController {

    @Resource
    protected BillPaymentService billPaymentService;
    @Resource
    protected CompanyPurRefService companyPurRefService;
    @Resource
    private BillInvorker invoker;

    @ApiOperation(httpMethod="GET", value="从SAP导入付款明细")
    @ApiImplicitParams({
            @ApiImplicitParam( name = "type",required = true,value = "付款类型", paramType = "query", dataType = "String", defaultValue = "1" ),
            @ApiImplicitParam( name = "startYear", required = true, value = "会计年度开始时间", paramType = "query", dataType = "int", defaultValue = "2011" ),
            @ApiImplicitParam( name = "endYear", required = true, value = "会计年度结束时间", paramType = "query", dataType = "int",defaultValue = "2018"),
            @ApiImplicitParam( name = "paymentStartTime", required = false, value = "凭证开始时间"),
            @ApiImplicitParam( name = "paymentEndTime", required = true, value = "凭证开始结束时间"),
            @ApiImplicitParam( name = "companySapCode", required = true, value = "供应商sap编码",paramType = "query", dataType = "String")
    })
    @RequestMapping("service/importBillPaymentFromSap")
    @ResponseBody
    public ResponseResult<String> importBillPaymentFromSap(
            @RequestParam(required = true,defaultValue = "1") String type,
            @RequestParam(required = true,defaultValue = "2011") int startYear,
            @RequestParam(required = true,defaultValue = "2018") int endYear,
            @RequestParam(required = false) Date paymentStartTime,
            @RequestParam(required = false) Date paymentEndTime,
            @RequestParam(required = true) String companySapCode) {

        BillPaymentVo billPaymentVo =new BillPaymentVo();

        billPaymentVo.setCompanySapCode(companySapCode);
        billPaymentVo.setEndYear(endYear);
        billPaymentVo.setStartYear(startYear);
        billPaymentVo.setPaymentStartTime(paymentStartTime);
        billPaymentVo.setPaymentEndTime(paymentEndTime);
        billPaymentVo.setType(type);

        ImportCommand cmd = new ImportCommand(billPaymentVo);
        cmd.setProject(ProjectUtils.getProject());
        invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "GET", value = "从SAP导入付款明细")
    @RequestMapping("service/sap/import")
    @ResponseBody
    public ResponseResult<String> sapImport(@RequestBody(required = true) BillPaymentVo billPayment) {
        ImportCommand cmd = new ImportCommand(billPayment);
        cmd.setProject(ProjectUtils.getProject());
        invoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "查询采购结算管理-付款明细-供应方")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 BillPayment", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/sup/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BillPayment>> findForSupByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        BillPaymentExample example = new BillPaymentExample();
        example.setPageView(new PageView<BillPayment>(pageNo, pageSize));

        BillPaymentExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件，例如
        Company company = CompanyUtils.currentCompany();
        criteria.andSupCompanySapCodeEqualTo(company.getCompanySapCode());
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }

        PageView<BillPayment> pageData = this.billPaymentService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "查询采购结算管理-付款明细-采购方")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 BillPayment", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/pur/findByPage")
    @ResponseBody
    public ResponseResult<PageView<BillPayment>> findForPurByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {

        //查询当前采购员下管理的所有的供应商的srm账号
        List<Company> companyList = this.companyPurRefService.queryAllCompanyByUserId(SpringSecurityUtils.getLoginUserId(), null);

        if (!CollectionUtils.isEmpty(companyList)) {

            List<String> sapCompanyList = new ArrayList<>();
            for (Company c : companyList) {
                Assert.isNotBlank(c.getCompanySapCode(), "供应商SAP编码不能为空！");
                sapCompanyList.add(c.getCompanySapCode());
            }

            BillPaymentExample example = new BillPaymentExample();
            example.setPageView(new PageView<BillPayment>(pageNo, pageSize));

            BillPaymentExample.Criteria criteria = example.createCriteria();
            criteria.andPurCompanyIdEqualTo(ProjectUtils.getCompanyId());
            //填写具体的查询条件，例如
            criteria.andSupCompanySapCodeIn(sapCompanyList);
            if (wapper != null) {
                CriteriaUtils.addExample(example, wapper);
            }

            PageView<BillPayment> pageData = this.billPaymentService.queryObjByPage(example);
            return ResponseResult.success(pageData);

        } else {
            PageView<BillPayment> pageData = new PageView<BillPayment>();
            pageData.setQueryResult(new ArrayList<BillPayment>());
            return ResponseResult.success(pageData);
        }
    }
}