package com.els.base.company.service.impl;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.company.dao.IndustryMapper;
import com.els.base.company.entity.Industry;
import com.els.base.company.entity.IndustryExample;
import com.els.base.company.service.IndustryService;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;

@Service("defaultIndustryService")
public class IndustryServiceImpl implements IndustryService {
	
	private static final Integer YES_INABLED = 1;
	
    @Resource
    protected IndustryMapper industryMapper;

    @CacheEvict(value={"industry"}, allEntries=true)
    @Override
    public void addObj(Industry t) {
    	IndustryExample example = new IndustryExample();
    	example.createCriteria()
    		   .andCodeEqualTo(t.getCode())
    		   .andIsEnableEqualTo(YES_INABLED);
    	if(this.industryMapper.countByExample(example) > 0)
    		throw new CommonException("行业编码已经存在!");
    	
        this.industryMapper.insertSelective(t);
    }

    @CacheEvict(value={"industry"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.industryMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"industry"}, allEntries=true)
    @Override
    public void modifyObj(Industry t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.industryMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="industry", keyGenerator="redisKeyGenerator")
    @Override
    public Industry queryObjById(String id) {
        return this.industryMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="industry", keyGenerator="redisKeyGenerator")
    @Override
    public List<Industry> queryAllObjByExample(IndustryExample example) {
        return this.industryMapper.selectByExample(example);
    }

    @Cacheable(value="industry", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<Industry> queryObjByPage(IndustryExample example) {
        PageView<Industry> pageView = example.getPageView();
        pageView.setQueryResult(this.industryMapper.selectByExampleByPage(example));
        return pageView;
    }
    
    /**
     * 查询所有分类信息
     */
    @Cacheable(value="industry", keyGenerator="redisKeyGenerator")
    @Override
    public List<Industry> queryAllObj() {
    	IndustryExample example = new IndustryExample();
    	example.createCriteria().andIsEnableEqualTo(Constant.YES_INT);
        List<Industry> allList = this.industryMapper.selectByExample(example);
        List<Industry> list = eachParent(allList);
        return list;
    }

    /**
     * 分类层级整合,仅限于两级分类
     */
	private List<Industry> eachParent(List<Industry> allList) {
		List<Industry> parentList = new ArrayList<Industry>();
        List<Industry> childList = new ArrayList<Industry>();
		 // 将一级分类和下级分类区分开
        for(Industry industry : allList){
        	String parentId = industry.getParentId();
        	    // 父级id为空则为一级分类
        	if(StringUtils.isBlank(parentId)){
        		parentList.add(industry);
        	}else{
        		// 下级分类
        		childList.add(industry);
        	}
        }
		for(int i=0;i<parentList.size();i++){
        	Industry industryParent = parentList.get(i);
        	List<Industry> newList = new ArrayList<Industry>();
        	for(int j=0;j<childList.size();j++){
            	Industry industryChild = childList.get(j);
            	// 校验parent_id
        		if(industryParent.getId().equals(industryChild.getParentId())){
        			newList.add(industryChild);
        		}
        	}
        	industryParent.setIndustryList(newList);
        }
		return parentList;
	}
	
	@Cacheable(value="industry", keyGenerator="redisKeyGenerator")
	@Override
	public List<Industry> queryByParentId(String parentId) {
		if(StringUtils.isBlank(parentId)){
			return null;
		}
		IndustryExample example = new IndustryExample();
		example.createCriteria()
			   .andParentIdEqualTo(parentId)
			   .andIsEnableEqualTo(YES_INABLED);
		List<Industry> list = this.industryMapper.selectByExample(example);
		if(CollectionUtils.isEmpty(list)){
			return null;
		}
		
		for (Industry industry : list) {
			industry.setIndustryList(this.queryByParentId(industry.getId()));
		}
		
		return list;
	}
	
	@Cacheable(value="industry", keyGenerator="redisKeyGenerator")
	@Override
	public List<Industry> findTheNextCascadedIndustryInfo(String parentId) {
		if(StringUtils.isBlank(parentId)){
			return null;
		}
		IndustryExample example = new IndustryExample();
		example.createCriteria()
			   .andParentIdEqualTo(parentId)
			   .andIsEnableEqualTo(YES_INABLED);
		List<Industry> list = this.industryMapper.selectByExample(example);
		return list;
	}
	
	@CacheEvict(value={"industry"}, allEntries=true)
	@Override
	public void deleteCascadedIndustryInfo(String parentId) {
		if(StringUtils.isBlank(parentId)){
			throw new CommonException("上一级传的id为空!");
		}
		//先删除
		this.industryMapper.deleteByPrimaryKey(parentId);
		
		IndustryExample example = new IndustryExample();
		example.createCriteria()
			   .andParentIdEqualTo(parentId)
			   .andIsEnableEqualTo(YES_INABLED);
		List<Industry> list = this.industryMapper.selectByExample(example);
		if(CollectionUtils.isEmpty(list)){
			return ;
		}
		
		for (Industry industry : list) {
			this.deleteCascadedIndustryInfo(industry.getId());
		}
		return ;
	}

	@CacheEvict(value={"industry"}, allEntries=true)
	@Transactional
	@Override
	public void addAll(List<Industry> arg0) {

		if(CollectionUtils.isEmpty(arg0)){
			return;
		}
		for (Industry record : arg0) {
			this.industryMapper.insertSelective(record);
		}
			
	}

	@CacheEvict(value={"industry"}, allEntries=true)
	@Override
	public void deleteByExample(IndustryExample arg0) {
		Assert.isNotEmpty(arg0.getOredCriteria(), "删除的条件不能为空");
		this.industryMapper.deleteByExample(arg0);
	}
}