package com.els.base.company.web.controller;

import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import com.els.base.company.entity.CompanySapRelation;
import com.els.base.company.entity.CompanySapRelationExample;
import com.els.base.company.service.CompanySapRelationService;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.utils.excel.BigDecimalConverter;
import com.els.base.utils.excel.ExcelUtils;
import com.els.base.utils.excel.TitleAndModelKey;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import jxl.write.WritableWorkbook;

@Api(value = "账号与供应商关联关系表")
@Controller
@RequestMapping("companySapRelation")
public class CompanySapRelationController {
	@Resource
	protected CompanySapRelationService companySapRelationService;

	@ApiOperation(httpMethod = "POST", value = "创建账号与供应商关联关系表")
	@RequestMapping("service/create")
	@ResponseBody
	public ResponseResult<String> create(@RequestBody CompanySapRelation companySapRelation) {
		if (companySapRelation == null) {
			throw new CommonException("参数不能为空");
		}
		this.companySapRelationService.addObj(companySapRelation);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod = "POST", value = "编辑账号与供应商关联关系表")
	@RequestMapping("service/edit")
	@ResponseBody
	public ResponseResult<String> edit(@RequestBody CompanySapRelation companySapRelation) {
		if (companySapRelation == null) {
			throw new CommonException("参数不能为空");
		}
		if (StringUtils.isBlank(companySapRelation.getId())) {
			throw new CommonException("id 为空，保存失败");
		}
		this.companySapRelationService.modifyObj(companySapRelation);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod = "POST", value = "删除账号与供应商关联关系表")
	@RequestMapping("service/deleteById")
	@ResponseBody
	public ResponseResult<String> deleteById(@RequestParam(required = true) String id) {
		if (StringUtils.isBlank(id)) {
			throw new CommonException("删除失败,id不能为空");
		}
		this.companySapRelationService.deleteObjById(id);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod = "POST", value = "查询账号与供应商关联关系表")
	@ApiImplicitParams({
			@ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
			@ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
			@ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 CompanySapRelation", paramType = "body", dataType = "QueryParamWapper") })
	@RequestMapping("service/findByPage")
	@ResponseBody
	public ResponseResult<PageView<CompanySapRelation>> findByPage(@RequestParam(defaultValue = "0") int pageNo,
			@RequestParam(defaultValue = "10") int pageSize, @RequestBody(required = false) QueryParamWapper wapper) {
		CompanySapRelationExample example = new CompanySapRelationExample();
		example.setPageView(new PageView<CompanySapRelation>(pageNo, pageSize));

		if (wapper != null) {
			CompanySapRelationExample.Criteria criteria = example.createCriteria();
			CriteriaUtils.addCriterion(criteria, wapper);
		}

		PageView<CompanySapRelation> pageData = this.companySapRelationService.queryObjByPage(example);
		return ResponseResult.success(pageData);
	}
	
	@ApiOperation(httpMethod = "POST", value = "检查SAP供应商编码在供应商关联关系表是否存在")
	@RequestMapping("service/checkSapSuppliterCodeIfExist/{code}")
	@ResponseBody
	public ResponseResult<Integer> checkSapSuppliterCodeIfExist(@PathVariable String code) {
		boolean flag = this.companySapRelationService.checkSapSupplierCodeIfExist(code, null);
		if (flag) {
			return ResponseResult.success(1);
		} else {
			return ResponseResult.success(0);
		}
	}
	
	@ApiOperation(httpMethod = "POST", value = "获取当前用户关联的供应商SAP编码")
	@RequestMapping("service/currentSapCode")
	@ResponseBody
	public ResponseResult<List<String>> currentSapCode() {
		String  companyCode = CompanyUtils.currentCompany().getCompanyCode();
		List<String> sapCodes = companySapRelationService.findSapCodeByCompanyCode(companyCode);
		return ResponseResult.success(sapCodes);
	}
	
	@ApiOperation(httpMethod = "GET", value = "导出EXCEL")
	@RequestMapping("service/exportToExcel")
	@ResponseBody
	public void exportToExcel(HttpServletResponse response) {
		
//		List<CompanySapRelation> list = this.companySapRelationService.queryAllObjByExample(null);
		
		try {
            response.reset();
            StringBuffer header = new StringBuffer("attachment;");
            header.append("filename=\"" + URLEncoder.encode("SAP供应商编码关系数据.xls", "UTF-8") + "\";");
            response.setHeader("Content-Disposition", header.toString());
            response.setCharacterEncoding("UTF-8");
            response.setContentType("application/vnd.ms-excel");
            OutputStream outputStream = response.getOutputStream();
            
            List<TitleAndModelKey> titleAndModelKeys = this.createExcelHeaderColumn();
            
            WritableWorkbook writableWorkbook = ExcelUtils.exportDataToExcel(outputStream, titleAndModelKeys,
            		null, "SAP供应商编码关系数据", null, 0);
            writableWorkbook.write();
            outputStream.flush();
            writableWorkbook.close();
            outputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
            response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
        }
		return;
	}
	
	@ApiOperation(value = "导入Excel", httpMethod = "POST")
	@RequestMapping(value = "service/importFromExcel")
	@ResponseBody
	public ResponseResult<String> importFromExcel(MultipartHttpServletRequest request) {

		Map<String, MultipartFile> fileMap = request.getFileMap();
		MultipartFile file = this.vaildFile(fileMap);

		List<TitleAndModelKey> titleAndModelKeys = this.createExcelHeaderColumn();
		List<CompanySapRelation> list;
		try {
			list = ExcelUtils.importExcelDataToMap(file.getInputStream(), 0, 1, 0, titleAndModelKeys, CompanySapRelation.class);
			
			companySapRelationService.importFromExcel(list);
		} catch (Exception e) {
			e.printStackTrace();
			throw new CommonException("导入异常：" + e.getMessage());
		}

		return ResponseResult.success();
	}
	
	@ApiOperation(httpMethod = "POST", value = "删除账号与供应商关联关系表")
	@RequestMapping("service/deleteByIds")
	@ResponseBody
	public ResponseResult<String> deleteByIds(@RequestBody Set<String> ids) {
		this.companySapRelationService.deleteByIds(ids);
		return ResponseResult.success();
	}
	
	private MultipartFile vaildFile(Map<String, MultipartFile> fileMap) {
		if (MapUtils.isEmpty(fileMap)) {
			throw new CommonException("上传文件为空", "file_isNull");
		}
		if (fileMap.size() > 1) {
			throw new CommonException("只接受单个文件导入");
		}

		Set<String> fileKeySet = fileMap.keySet();// 获取所有的key集合

		Iterator<String> keyIterator = fileKeySet.iterator();
		MultipartFile file = null;
		while (keyIterator.hasNext()) {
			file = fileMap.get(keyIterator.next());
		}
		
		if (!file.getOriginalFilename().endsWith(".xls")) {
			throw new CommonException("导入文件的格式不正确，目前只支持xls");
		}
		
		return file;
	}
	
	private List<TitleAndModelKey> createExcelHeaderColumn() {
		List<TitleAndModelKey> titleAndModelKeys = new ArrayList<>();
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商SRM编码", "companyCode"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商SAP编码", "sapSupplierCode"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商全称", "companyFullName"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商英文全称", "companyFullEnName"));
        TitleAndModelKey taxRate = ExcelUtils.createTitleAndModelKey("税率", "taxRate");
        taxRate.setToObjConverter(new BigDecimalConverter());
        taxRate.setToStrConverter(new BigDecimalConverter());
        titleAndModelKeys.add(taxRate);
        
        
        return titleAndModelKeys;
	}
	
}