package com.els.base.company.web.controller;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.company.entity.Department;
import com.els.base.company.entity.DepartmentExample;
import com.els.base.company.service.DepartmentService;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.project.ProjectUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@Api(value="部门信息")
@Controller
@RequestMapping("department")
public class DepartmentController {
    @Resource
    protected DepartmentService departmentService;

    @ApiOperation(httpMethod="POST", value="创建部门信息")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody Department department) {
    	
    	department.setCompanyId(CompanyUtils.currentCompanyId());
    	department.setProjectId(ProjectUtils.getProjectId());
    	
        this.departmentService.addObj(department);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="编辑部门信息")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody Department department) {
        if (StringUtils.isBlank(department.getId())) {
            throw new CommonException("Id为空","id_is_blank");
        }
        this.departmentService.modifyObj(department);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="删除部门信息")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(String id) {
    	Department department = new Department();
    	department.setId(id);
    	department.setIsEnable(Constant.NO_INT);
    	this.departmentService.modifyObj(department);
//        this.departmentService..deleteObjById(id);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="GET", value="查询当前公司的部门信息")
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<Department>> findByPage(
    		@ApiParam(value="所在页",defaultValue="1") @RequestParam(defaultValue="1") int pageNo, 
    		@ApiParam(value="每页数量",defaultValue="10") @RequestParam(defaultValue="10") int pageSize,
    		@ApiParam(value="上级id,不传则查询顶级部门") String parentId) {
        DepartmentExample example = new DepartmentExample();
        example.setPageView(new PageView<Department>(pageNo, pageSize));
       
        DepartmentExample.Criteria cri = example.createCriteria();
        cri.andCompanyIdEqualTo(CompanyUtils.currentCompanyId())
           .andParentIdIsNull()
           .andIsEnableEqualTo(Constant.YES_INT);
        
        PageView<Department> pageData = this.departmentService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
    
    @ApiOperation(httpMethod="GET", value="查询当前公司的部门信息")
    @RequestMapping("service/findDepartmentTree")
    @ResponseBody
    public ResponseResult<List<Department>> findDepartmentTree(){
    	DepartmentExample example = new DepartmentExample();
    	example.createCriteria()
    	       .andIsEnableEqualTo(Constant.YES_INT)
    	       .andParentIdIsNull()
    	       .andCompanyIdEqualTo(CompanyUtils.currentCompanyId());
    	
    	List<Department> departmentList = this.departmentService.queryAllObjByExample(example);
    	for(Department department : departmentList){
    		department.setSubDepList(this.departmentService.queryByParentId(department.getId()));;
    	}
    	
    	return ResponseResult.success(departmentList);
    }
    
    @ApiOperation(httpMethod="GET", value="查询部门内的帐号")
    @RequestMapping("service/findUsersInDepartment")
    @ResponseBody
    public ResponseResult<PageView<User>> findUsersInDepartment(
    		@ApiParam(value="所在页",defaultValue="1") @RequestParam(defaultValue="1") int pageNo, 
    		@ApiParam(value="每页数量",defaultValue="10") @RequestParam(defaultValue="10") int pageSize,
    		String departmentId){
    	if (StringUtils.isBlank(departmentId)) {
			throw new CommonException("Id为空","id_is_blank");
		}
    	
    	return ResponseResult.success(this.departmentService.queryUsersInDepartmentByPage(pageNo, pageSize, departmentId));
    }
    
    @ApiOperation(httpMethod="GET", value="查询部门内的帐号")
    @RequestMapping("service/addUserIntoDepartment")
    @ResponseBody
    public ResponseResult<String> addUserIntoDepartment(String userId, String departmentId){
    	Department department = this.departmentService.queryObjById(departmentId);
    	if (department == null) {
    		throw new CommonException("找不到部门","not_found_depart");
		}
    	
    	if (!department.getCompanyId().equals(CompanyUtils.currentCompanyId())) {
			throw new CommonException("找不到部门","not_found_depart");
		}
    	
    	this.departmentService.addUserIntoDepartment(userId, departmentId);
    	return ResponseResult.success();
    }
    
}