package com.els.base.company.web.controller;

import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.company.dao.IndustryMapper;
import com.els.base.company.entity.Industry;
import com.els.base.company.entity.IndustryExample;
import com.els.base.company.service.IndustryService;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@Api(value="行业信息")
@Controller
@RequestMapping("industry")
public class IndustryController {
	
	private static final Integer YES_INABLED = 1;
	
    @Resource
    protected IndustryService industryService;
    
    @Resource
    protected IndustryMapper industryMapper;

    @ApiOperation(httpMethod="POST", value="创建行业信息")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody Industry industry) {
    	if(StringUtils.isEmpty(industry.getName())){
    		throw new CommonException("行业名称不能为空!");
    	}
    	if(StringUtils.isEmpty(industry.getCode())){
    		throw new CommonException("编码不能为空");
    	}
    	industry.setId(null);
    	industry.setIsEnable(YES_INABLED);
        this.industryService.addObj(industry);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="编辑行业信息")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody Industry industry) {
        if (StringUtils.isBlank(industry.getId())) {
            throw new CommonException("Id为空","id_is_blank");
        }
        if(StringUtils.isEmpty(industry.getName())){
    		throw new CommonException("行业名称不能为空!");
    	}
    	if(StringUtils.isEmpty(industry.getCode())){
    		throw new CommonException("编码不能为空");
    	}
    	//判断修改时是否提交了新的code
    	int flag = 0;
    	
    	Industry obj = industryService.queryObjById(industry.getId());
    	if(null == obj){
    		throw new CommonException("该id查不到数据");
    	}else{
    		String code = obj.getCode();
    		if(StringUtils.isNotBlank(code) && (!code.equals(industry.getCode()))  ){
    			//说明修改时提交了新的code
    			flag = 1;
    		}
    	}
    	
    	if(flag == 1){
    		//提交了新的code,需要判断code值是否存在
    		IndustryExample example = new IndustryExample();
    		example.createCriteria()
    			   .andCodeEqualTo(industry.getCode())
    			   .andIsEnableEqualTo(YES_INABLED);
    		if(this.industryMapper.countByExample(example) > 0){
    			throw new CommonException("该行业编码已经存在,无法修改!");
    		}
    	}
    	industry.setUpdateTime(new Date());
        this.industryService.modifyObj(industry);
        return ResponseResult.success();
    }
    
    @Transactional
    @ApiOperation(httpMethod="POST", value="删除行业信息")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required=true) String id) {
    	if(StringUtils.isEmpty(id)){
    		throw new CommonException("Id不能为空","id_is_blank");
    	}
    	this.industryService.deleteObjById(id);
    	
    	IndustryExample example = new IndustryExample();
    	example.createCriteria()
    		   .andParentIdEqualTo(id);
    	if(this.industryMapper.countByExample(example) > 0){
    		//需要把级联的所有数据也删除
    		List<Industry> list = this.industryService.queryAllObjByExample(example);
    		for (Industry industry : list) {
				this.industryService.deleteCascadedIndustryInfo(industry.getId());
			}
    	}
    	
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="GET", value="查询行业信息")
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<Industry>> findByPage(
    		@ApiParam(value="所在页码",defaultValue="1") @RequestParam(defaultValue="1") int pageNo, 
    		@ApiParam(value="每页的数量",defaultValue="10") @RequestParam(defaultValue="10") int pageSize, 
    		@ApiParam(value="查询条件") String sSearch) {
        IndustryExample example = new IndustryExample();
        example.setPageView(new PageView<Industry>(pageNo, pageSize));
        example.setOrderByClause("CREATE_TIME DESC");
        if (StringUtils.isNotBlank(sSearch)) {
            // TODO Auto-generated method stub
        }
        example.createCriteria().andIsEnableEqualTo(Constant.YES_INT);
        PageView<Industry> pageData = this.industryService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
    
    @ApiOperation(httpMethod="POST", value="通过id查询行业信息")
    @RequestMapping("service/findIndustryById")
    @ResponseBody
    public ResponseResult<Industry> findIndustryById(@RequestParam(required=true) String id){
    	if(StringUtils.isEmpty(id))
    		throw new CommonException("Id不能为空","id_is_blank");
    	Industry i = industryService.queryObjById(id);
    	return ResponseResult.success(i);
    }
    
    @ApiOperation(httpMethod="POST", value="查询所有的行业信息及关联的子行业信息")
    @RequestMapping("service/findAllIndustryInfo")
    @ResponseBody
    public ResponseResult<List<Industry>> findAllIndustryInfo(){
    	IndustryExample example = new IndustryExample();
    	example.setOrderByClause("CREATE_TIME DESC");
		example.createCriteria()
			   .andIsEnableEqualTo(YES_INABLED)
			   .andParentIdIsNull();
		List<Industry> industryList = this.industryService.queryAllObjByExample(example);
		for (Industry industry : industryList) {
			industry.setIndustryList(this.industryService.queryByParentId(industry.getId()));
		}
		
		return ResponseResult.success(industryList);
    }
    
    @ApiOperation(httpMethod="POST", value="根据id查找下一级联的行业信息")
    @RequestMapping("service/findTheNextCascadedIndustryInfo")
    @ResponseBody
    public ResponseResult<List<Industry>> findTheNextCascadedIndustryInfo(@RequestParam(required=true) String id){
    	if(StringUtils.isEmpty(id)){
    		throw new CommonException("Id不能为空","id_is_blank");
    	}
    	List<Industry> list = this.industryService.findTheNextCascadedIndustryInfo(id);
    	return ResponseResult.success(list);
    }
}