package com.els.base.company.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import com.els.base.company.dao.CompanySapRelationMapper;
import com.els.base.company.entity.Company;
import com.els.base.company.entity.CompanyExample;
import com.els.base.company.entity.CompanySapRelation;
import com.els.base.company.entity.CompanySapRelationExample;
import com.els.base.company.service.CompanySapRelationService;
import com.els.base.company.service.CompanyService;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;

@Service("defaultCompanySapRelationService")
public class CompanySapRelationServiceImpl implements CompanySapRelationService {
	@Resource
	protected CompanySapRelationMapper companySapRelationMapper;
	
	@Resource
    protected CompanyService companyService;
//	
//	@Autowired
//	private PurchaseOrderService orderHeaderService;

	@Cacheable(value = "companySapRelation", keyGenerator = "redisKeyGenerator")
	@Override
	public CompanySapRelation findBySrmCodeAndSapCode(String srmCode, String sapCode) {
		Assert.isNotBlank(srmCode,"SRM编码不能为空！");
		Assert.isNotBlank(sapCode,"SAP编码不能为空！");
		CompanySapRelationExample companySapRelationExample =new CompanySapRelationExample();
		companySapRelationExample.createCriteria()
				.andCompanyCodeEqualTo(srmCode)
				.andSapSupplierCodeEqualTo(sapCode);
		List<CompanySapRelation> companySapRelations = this.companySapRelationMapper.selectByExample(companySapRelationExample);
		Assert.isNotEmpty(companySapRelations,"采购员没有在供应商编码管理里面维护信息，请联系相关负责人，谢谢！");
		return companySapRelations.get(0);
	}

	@CacheEvict(value = { "companySapRelation" }, allEntries = true)
	@Override
	public void addObj(CompanySapRelation t) {
		
		if (org.springframework.util.StringUtils.isEmpty(t.getSapSupplierCode())) {
			throw new CommonException("该供应商SAP编码为空");
		}
		
		if (org.springframework.util.StringUtils.isEmpty(t.getCompanyCode())) {
			throw new CommonException("该供应商SAP编码为空");
		}
		
		if (t.getSapSupplierCode().length() > 32) {
			throw new CommonException("供应商SAP编码长度不能大于32");
		}
		
		if (t.getCompanyCode().length() > 32) {
			throw new CommonException("供应商SRM编码长度不能大于32");
		}
		
		Company company = checkCompanyCodeIfExist(t.getCompanyCode());
		if (company == null) {
			throw new CommonException("该供应商SRM编码[" + t.getCompanyCode() + "]不存在系统");
		}
		
		boolean flag = checkSapSupplierCodeIfExist(t.getSapSupplierCode(), null);
		if (flag == true) {
			throw new CommonException("该供应商SAP编码[" + t.getSapSupplierCode() + "]已经存在系统");
		}
		
		t.setCompanyName(company.getCompanyName());
		t.setCompanyFullName(company.getCompanyFullName());
		
		this.companySapRelationMapper.insertSelective(t);
	}

	@CacheEvict(value = { "companySapRelation" }, allEntries = true)
	@Override
	public void deleteObjById(String id) {
		//先去检查以前是否产生了数据
		CompanySapRelation companySapRelation = queryObjById(id);
		boolean flag2 = checkSapSuppliterCodeIfExistData(companySapRelation.getCompanyCode());
		if (flag2 == true) {
			throw new CommonException("该供应商SAP编码[" + companySapRelation.getSapSupplierCode() + 
					"]已经在系统生产了业务数据，不能删除。");
		}
		this.companySapRelationMapper.deleteByPrimaryKey(id);
	}

	@CacheEvict(value = { "companySapRelation" }, allEntries = true)
	@Override
	public void modifyObj(CompanySapRelation t) {
		Assert.isNotBlank(t.getId(), "id 为空，无法更新");
		Assert.isNotBlank(t.getSapSupplierCode(), "该供应商SAP编码为空");
		Assert.isNotBlank(t.getCompanyCode(), "该供应商SRM编码为空");
		
		if (t.getSapSupplierCode().length() > 32) {
			throw new CommonException("供应商SAP编码长度不能大于32");
		}
		if (t.getCompanyCode().length() > 32) {
			throw new CommonException("供应商SRM编码长度不能大于32");
		}
		
		CompanySapRelation oldCompanySapRelation = this.queryObjById(t.getId());
		if (!oldCompanySapRelation.getCompanyCode().equals(t.getCompanyCode())) {
			throw new CommonException("已创建的SRM编码不能修改了");
		}
		if (!oldCompanySapRelation.getSapSupplierCode().equals(t.getSapSupplierCode())) {
			throw new CommonException("已创建的SAP编码不能修改了");
		}
		
		//如果sap编码，现在的跟以前的一样
		if ((t.getCompanyFullName() == null || t.getCompanyFullName().equals(oldCompanySapRelation.getCompanyFullName()))
				&& (t.getCompanyFullEnName() == null || t.getCompanyFullEnName().equals(oldCompanySapRelation.getCompanyFullEnName()))
				&& (t.getTaxRate() == null || t.getTaxRate().equals(oldCompanySapRelation.getTaxRate()))){
			return;
		}
		
		Company company = checkCompanyCodeIfExist(t.getCompanyCode());
		if (company == null) {
			throw new CommonException("该供应商SRM编码[" + t.getCompanyCode() + "]不存在系统");
		}
		
//		boolean isExist = checkSapSupplierCodeIfExist(t.getSapSupplierCode(), t.getId());
//		if (isExist) {
//			throw new CommonException("该供应商SAP编码[" + t.getSapSupplierCode() + "]已经存在系统");
//		}
		
		//先去检查以前是否产生了数据
//		CompanySapRelation companySapRelation = queryObjById(t.getId());
//		if (t.getSapSupplierCode() != null
//				&& !t.getSapSupplierCode().equals(companySapRelation.getSapSupplierCode())) {
//			//如果是修改sap编码，检查是否已经关联业务
//			
//			boolean isRelateBusiness = checkSapSuppliterCodeIfExistData(companySapRelation.getCompanyCode());
//			if (isRelateBusiness) {
//				throw new CommonException("该供应商SAP编码[" + companySapRelation.getSapSupplierCode() + "]已经在系统生产了业务数据，不能再修改");
//			}
//		}
		
		this.companySapRelationMapper.updateByPrimaryKeySelective(t);
	}

	@Cacheable(value = "companySapRelation", keyGenerator = "redisKeyGenerator")
	@Override
	public CompanySapRelation queryObjById(String id) {
		return this.companySapRelationMapper.selectByPrimaryKey(id);
	}

	@Cacheable(value = "companySapRelation", key = "'CompanySapRelationService_' + #root.methodName + '_'+ T(com.els.base.utils.encryption.Md5Utils).md5Object(#example)")
	@Override
	public List<CompanySapRelation> queryAllObjByExample(CompanySapRelationExample example) {
		return this.companySapRelationMapper.selectByExample(example);
	}

	@Cacheable(value = "companySapRelation", key = "'CompanySapRelationService_' + #root.methodName + '_'+ T(com.els.base.utils.encryption.Md5Utils).md5Object(#example)")
	@Override
	public PageView<CompanySapRelation> queryObjByPage(CompanySapRelationExample example) {
		PageView<CompanySapRelation> pageView = example.getPageView();
		pageView.setQueryResult(this.companySapRelationMapper.selectByExampleByPage(example));
		return pageView;
	}

	/**
	 * 检查sap供应商编码是否存在
	 * @author zhoulongpeng
	 * @param sapSupplierCode
	 */
	@Override
	public boolean checkSapSupplierCodeIfExist(String sapSupplierCode, String id) {
		CompanySapRelationExample example = new CompanySapRelationExample();
		CompanySapRelationExample.Criteria criteria = example.createCriteria();
		criteria.andSapSupplierCodeEqualTo(sapSupplierCode);
		if (!org.springframework.util.StringUtils.isEmpty(id)) {
			criteria.andIdNotEqualTo(id);
		}

		List<CompanySapRelation> list = this.queryAllObjByExample(example);

		if (list != null && !list.isEmpty()) {
			return true;
		}
		return false;
	}

	@Override
	public String findCompanyCodeBySapSupplierCode(String sapSupplierCode) {
		CompanySapRelation companySapRelation = findBySapSupplierCode(sapSupplierCode);
		if (companySapRelation == null) {
			return null;
		}
		return companySapRelation.getCompanyCode();
	}
	
	@CacheEvict(value = { "companySapRelation" }, allEntries = true)
	@Override
	public void importFromExcel(List<CompanySapRelation> list) {
		if (list == null || list.isEmpty()) {
			throw new CommonException("导入的数据为空");
		}
		int index = 2;
		for(CompanySapRelation newEntity : list) {
			if (StringUtils.isBlank(newEntity.getSapSupplierCode())) {
				throw new CommonException("第"+index+"行供应商SAP编码数据为空");
			}
			if (StringUtils.isBlank(newEntity.getCompanyCode())) {
				throw new CommonException("第"+index+"行供应商SRM编码数据为空");
			}
			
			newEntity.setCompanyCode(newEntity.getCompanyCode().replace("�", ""));
			newEntity.setSapSupplierCode(newEntity.getSapSupplierCode().replace("�", ""));
			
			CompanySapRelation oldCompanySapRelation = findBySapSupplierCode(newEntity.getSapSupplierCode());
			//如果以前没有存在该SRM供应商编码记录，直接新增
			if (oldCompanySapRelation == null) {
				this.addObj(newEntity);
				continue;
			}
			
			newEntity.setId(oldCompanySapRelation.getId());
			if (newEntity.getCompanyFullEnName()!=null 
					&& newEntity.getCompanyFullEnName().trim().length() == 0) {
				newEntity.setCompanyFullEnName(null);
			}
			if (newEntity.getCompanyFullName()!=null 
					&& newEntity.getCompanyFullName().trim().length() == 0) {
				newEntity.setCompanyFullName(null);
			}
			if (newEntity.getCompanyName()!=null 
					&& newEntity.getCompanyName().trim().length() == 0) {
				newEntity.setCompanyName(null);
			}
			this.modifyObj(newEntity);
			index++;
		}
		
	}
	
	@CacheEvict(value = { "companySapRelation" }, allEntries = true)
	@Override
	public void deleteByIds(Set<String> ids) {
		if (ids == null || ids.isEmpty()) {
			throw new CommonException("参入的ID参数为空");
		}
		for(String id : ids) {
			//先去检查以前是否产生了数据
			CompanySapRelation companySapRelation = queryObjById(id);
			boolean flag2 = checkSapSuppliterCodeIfExistData(companySapRelation.getCompanyCode());
			if (flag2 == true) {
				throw new CommonException("该供应商SAP编码[" + companySapRelation.getSapSupplierCode() + 
						"]已经在系统生产了业务数据，不能删除。");
			}
			companySapRelationMapper.deleteByPrimaryKey(id);
		}
	}
	
	@Override
	public List<String> findSapCodeByCompanyCode(String companyCode) {
		if (org.springframework.util.StringUtils.isEmpty(companyCode)) {
			throw new CommonException("参数不能为空");
		}
		
		List<String> sapCodes = new ArrayList<>();
		
		CompanySapRelationExample example = new CompanySapRelationExample();
		CompanySapRelationExample.Criteria criteria = example.createCriteria();
		criteria.andCompanyCodeEqualTo(companyCode);
		
		List<CompanySapRelation> list = this.queryAllObjByExample(example);
		if (list == null || list.isEmpty()) {
			return null;
		}
		for (CompanySapRelation e : list) {
			sapCodes.add(e.getSapSupplierCode());
		}
		return sapCodes;
	}

	private CompanySapRelation findBySapSupplierCode(String sapSupplierCode) {
		CompanySapRelationExample example = new CompanySapRelationExample();
		CompanySapRelationExample.Criteria criteria = example.createCriteria();
		criteria.andSapSupplierCodeEqualTo(sapSupplierCode);

		List<CompanySapRelation> list = this.queryAllObjByExample(example);

		if (list != null && !list.isEmpty()) {
			return list.get(0);
		}
		return null;
	}
	
	/**
	 * 检查供应商编码是否存在系统
	 * 
	 * @param code
	 * @return  
	 */
	private Company checkCompanyCodeIfExist(String code) {
		CompanyExample example = new CompanyExample();
		CompanyExample.Criteria criteria = example.createCriteria();
		criteria.andCompanyCodeEqualTo(code);
		List<Company> list = companyService.queryAllObjByExample(example);
		if (list == null || list.isEmpty()) {
			return null;
		}
		return list.get(0);
	}
	
	/**
	 * 检查该srm供应商编码的关联srm供应商编码是否产生了数据
	 * 
	 * @param sapSuppliterCode
	 * @return true表示产生了数据，false表示没有产生数据
	 */
	private boolean checkSapSuppliterCodeIfExistData(String companyCode) {
		return true;
		
//		if (StringUtils.isBlank(companyCode)) {
//			return false;
//		}
//		
//		CompanyExample companyExample = new CompanyExample();
//		CompanyExample.Criteria companyCriteria = companyExample.createCriteria();
//		companyCriteria.andCompanyCodeEqualTo(companyCode);
//		List<Company> sapCompanyList = companyService.queryAllObjByExample(companyExample);
//		if(sapCompanyList == null || sapCompanyList.isEmpty()){
//			return false;
//		}
//		Company c = sapCompanyList.get(0);
//		String supCompanyId = c.getId();
//		
//		PurchaseOrderExample orderExample = new PurchaseOrderExample();
//		PurchaseOrderExample.Criteria orderCriteria = orderExample.createCriteria();
//		orderCriteria.andSupCompanyIdEqualTo(supCompanyId);
//		
//		List<PurchaseOrder> listOrder = orderHeaderService.queryAllObjByExample(orderExample);
//		if(listOrder == null || listOrder.isEmpty()){
//			return false;
//		}
//		
//		return true;
	}

	@CacheEvict(value = { "companySapRelation" }, allEntries = true)
	@Override
	public void addAll(List<CompanySapRelation> arg0) {
		if(CollectionUtils.isEmpty(arg0)){
			return;
		}
		this.companySapRelationMapper.insertBatch(arg0);
	}

	@CacheEvict(value = { "companySapRelation" }, allEntries = true)
	@Override
	public void deleteByExample(CompanySapRelationExample arg0) {
		Assert.isNotEmpty(arg0.getOredCriteria(), "删除条件不能为空");
		
		this.companySapRelationMapper.deleteByExample(arg0);
	}
	
	
}