package com.els.base.company.service.impl;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.company.dao.DepartmentMapper;
import com.els.base.company.entity.CompanyUserRef;
import com.els.base.company.entity.CompanyUserRefExample;
import com.els.base.company.entity.Department;
import com.els.base.company.entity.DepartmentExample;
import com.els.base.company.service.CompanyUserRefService;
import com.els.base.company.service.DepartmentService;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.user.User;
import com.els.base.core.entity.user.UserExample;
import com.els.base.core.exception.CommonException;
import com.els.base.core.service.user.UserService;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.utils.SpringContextHolder;

@Service("defaultDepartmentService")
public class DepartmentServiceImpl implements DepartmentService {
    @Resource
    protected DepartmentMapper departmentMapper;
    
    @Resource
    protected CompanyUserRefService companyUserRefService;
    
    @Resource
    protected UserService userService;

    @CacheEvict(value={"department"}, allEntries=true)
    @Override
    public void addObj(Department t) {
    	t.setIsEnable(Constant.YES_INT);
        this.departmentMapper.insertSelective(t);
    }
    
    @CacheEvict(value={"department"}, allEntries=true)
    @Override
	public void insertObj(Department t) {
    	this.departmentMapper.insertSelective(t);
	}

    @CacheEvict(value={"department"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.departmentMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"department"}, allEntries=true)
    @Override
    public void modifyObj(Department t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.departmentMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="department", keyGenerator="redisKeyGenerator")
    @Override
    public Department queryObjById(String id) {
        return this.departmentMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="department", keyGenerator="redisKeyGenerator")
    @Override
    public List<Department> queryAllObjByExample(DepartmentExample example) {
        return this.departmentMapper.selectByExample(example);
    }

    @Cacheable(value="department", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<Department> queryObjByPage(DepartmentExample example) {
        PageView<Department> pageView = example.getPageView();
        pageView.setQueryResult(this.departmentMapper.selectByExampleByPage(example));
        return pageView;
    }

    
    @Cacheable(value="department", keyGenerator="redisKeyGenerator")
	@Override
	public PageView<User> queryUsersInDepartmentByPage(int pageNo, int pageSize, String departmentId) {
		CompanyUserRefExample example = new CompanyUserRefExample();
		example.createCriteria().andDepartmentIdEqualTo(departmentId);
		
		List<CompanyUserRef> allUser = this.companyUserRefService.queryAllObjByExample(example);
		if (CollectionUtils.isEmpty(allUser)) {
			return new PageView<>(pageNo, pageSize);
		}
		
		List<String> userIdList = new ArrayList<>();
		for(CompanyUserRef ref : allUser){
			userIdList.add(ref.getUserId());
		}
		
		UserExample userExample = new UserExample();
		userExample.createCriteria()
		           .andIdIn(userIdList)
		           .andIsEnableEqualTo(Constant.YES_INT);
		
		userExample.setPageView(new PageView<User>(pageNo, pageSize));
		
		return this.userService.queryObjByPage(userExample);
	}
    
    @Cacheable(value="department", keyGenerator="redisKeyGenerator")
    @Override
    public List<Department> queryByParentId(String parentId){
    	if (StringUtils.isBlank(parentId)) {
			return null;
		}
    	
    	DepartmentExample example = new DepartmentExample();
    	example.createCriteria().andParentIdEqualTo(parentId).andIsEnableEqualTo(Constant.YES_INT);
    	
    	List<Department> departmentList = this.departmentMapper.selectByExample(example);
    	if (CollectionUtils.isEmpty(departmentList)) {
			return null;
		}
    	for(Department department: departmentList){
    		department.setSubDepList(this.queryByParentId(department.getId()));
    	}
    	
    	return departmentList;
    }

    @CacheEvict(value={"department","userService"}, allEntries=true)
	@Override
	public void addUserIntoDepartment(String userId, String departmentId) {
		Department department  = SpringContextHolder.getOneBean(DepartmentService.class).queryObjById(departmentId);
		if (department == null) {
			throw new CommonException("找不到部门","not_found_depart");
		}
		
		CompanyUserRefExample userRefExample = new CompanyUserRefExample();
		userRefExample.createCriteria().andUserIdEqualTo(userId);
		
		List<CompanyUserRef> refList = this.companyUserRefService.queryAllObjByExample(userRefExample);
		if (CollectionUtils.isEmpty(refList)) {
			CompanyUserRef ref = new CompanyUserRef();
			ref.setUserId(userId);
			ref.setCompanyId(department.getCompanyId());
			ref.setDepartmentId(departmentId);
			ref.setProjectId(department.getProjectId());
			this.companyUserRefService.addObj(ref);
		}else{
			if (refList.size() > 1) {
				throw new CommonException("数据已存在","base_is_exists","用户");
			}
			
			CompanyUserRef ref = refList.get(0);
			if (!department.getCompanyId().equals(ref.getCompanyId())) {
				throw new CommonException("用户已分配","user_assigned");
			}
			
			CompanyUserRef tmp = new CompanyUserRef();
			tmp.setId(ref.getId());
			tmp.setDepartmentId(departmentId);
			this.companyUserRefService.modifyObj(tmp);
		}
	}

    @Cacheable(value="department", keyGenerator="redisKeyGenerator")
	@Override
	public Department queryDepartmentOfUser(String userId) {
		CompanyUserRefExample example = new CompanyUserRefExample();
		example.createCriteria().andUserIdEqualTo(userId);
		
		List<CompanyUserRef> refList = this.companyUserRefService.queryAllObjByExample(example);
		if (CollectionUtils.isEmpty(refList)) {
			return null;
		}
		
		String departmentId = refList.get(0).getDepartmentId();
		if (StringUtils.isBlank(departmentId)) {
			return null;
		}
		
		return this.queryObjById(departmentId);
	}

    @CacheEvict(value={"department","userService"}, allEntries=true)
    @Transactional
	@Override
	public void addAll(List<Department> arg0) {

		if(CollectionUtils.isEmpty(arg0)){
			return;
		}
		for (Department record : arg0) {
			this.departmentMapper.insertSelective(record);
		}
			
	}

    @CacheEvict(value={"department","userService"}, allEntries=true)
	@Override
	public void deleteByExample(DepartmentExample arg0) {
		Assert.isNotEmpty(arg0.getOredCriteria(), "删除的条件不能为空");
		this.departmentMapper.deleteByExample(arg0);
	}

}