package com.els.base.company.web.controller;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.company.entity.CompanyAddress;
import com.els.base.company.entity.CompanyAddressExample;
import com.els.base.company.service.CompanyAddressService;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@Api(value="公司个性地址")
@Controller
@RequestMapping("companyAddress")
public class CompanyAddressController {
	
    @Resource
    protected CompanyAddressService companyAddressService;
    
    @ApiOperation(httpMethod="POST", value="创建公司个性地址")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody CompanyAddress companyAddress) {
    	companyAddress.setProjectId(ProjectUtils.getProjectId());
    	companyAddress.setCompanyId(CompanyUtils.currentCompanyId());
    	if(companyAddress.getIsEnable()==null){
    		companyAddress.setIsEnable(Constant.YES_INT);
    	}
        this.companyAddressService.addObj(companyAddress);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="编辑公司个性地址")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody CompanyAddress companyAddress) {
        if (StringUtils.isBlank(companyAddress.getId())) {
            throw new CommonException("id 为空，保存失败");
        }
        if(companyAddress.getIsEnable()==null){
    		companyAddress.setIsEnable(Constant.YES_INT);
    	}
        companyAddress.setProjectId(ProjectUtils.getProjectId());
    	companyAddress.setCompanyId(CompanyUtils.currentCompanyId());
        this.companyAddressService.modifyObj(companyAddress);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="删除公司个性地址")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required=true) String id) {
        if (StringUtils.isBlank(id)) {
            throw new CommonException("删除失败,id不能为空");
        }
        CompanyAddress companyAddress = this.companyAddressService.queryObjById(id);
        if (companyAddress==null) {
            throw new CommonException("删除失败,公司不存在");
        }
        //companyAddress.setIsEnable(Constant.NO_INT);
        //this.companyAddressService.modifyObj(companyAddress);
        this.companyAddressService.deleteObjById(id);
        return ResponseResult.success();
    }

    
    @ApiOperation(httpMethod="POST", value="查询公司个性地址")
    @ApiImplicitParams({
		@ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
		@ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
		@ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 CompanyAddress", paramType = "body", dataType = "QueryParamWapper") 
	})
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<CompanyAddress>> findByPage(
    		@RequestParam(defaultValue="0") int pageNo, 
    		@RequestParam(defaultValue="10") int pageSize, 
    		@RequestBody(required=false) QueryParamWapper wapper) {
    	
        CompanyAddressExample example = new CompanyAddressExample();
        example.setPageView(new PageView<CompanyAddress>(pageNo, pageSize));
        example.setOrderByClause(" ADDRESS_CODE ASC ");
        
        CompanyAddressExample.Criteria criteria = example.createCriteria();
        criteria.andProjectIdEqualTo(ProjectUtils.getProjectId());
        criteria.andCompanyIdEqualTo(CompanyUtils.currentCompanyId());
        //criteria.andIsEnableEqualTo(Constant.YES_INT);
        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }
        PageView<CompanyAddress> pageData = this.companyAddressService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
    
    
    @ApiOperation(httpMethod = "GET", value = "启用/禁止企业地址信息")
	@RequestMapping("service/isEnable")
	@ResponseBody
	public ResponseResult<Integer> isEnable(@ApiParam(value = "企业地址ID") @RequestParam(required = true) String id,
			@ApiParam(value = "启用传1,禁用传0") Integer isEnable) {
		if (StringUtils.isBlank(id)) {
			throw new CommonException("Id为，空不能禁用或启用", "id_is_blank");
		}
		int result = this.companyAddressService.isEnable(id, isEnable);
		return ResponseResult.success(result);
	}
    
    
    @ApiOperation(httpMethod = "GET", value = "根据公司Id,公司sap编码，语言编码返回地址")
    @ApiImplicitParams({
		@ApiImplicitParam(name = "companyId", required = true, value = "公司ID", paramType = "query", dataType = "String"),
		@ApiImplicitParam(name = "addressCode", required = false, value = "子公司编码", paramType = "query", dataType = "String"),
		@ApiImplicitParam(name = "languageCode", required = false, value = "语言编码", paramType = "query", dataType = "String") 
	})
   	@RequestMapping("service/queryByCompanyIdAndAddressCodeAndLanguageCode")
   	@ResponseBody
   	public ResponseResult<List<CompanyAddress>> queryByCompanyIdAndAddressCodeAndLanguageCode(
   			@RequestParam(required = true) String companyId,
   			@RequestParam String addressCode,@RequestParam String languageCode) {
   		return ResponseResult.success(this.companyAddressService.queryByCompanyIdAndAddressCodeAndLanguageCode(companyId, addressCode, languageCode));
   	}
    
    
    @ApiOperation(httpMethod = "GET", value = "设置默认地址")
    @ApiImplicitParams({
		@ApiImplicitParam(name = "companyId", required = true, value = "公司ID", paramType = "query", dataType = "String"),
		@ApiImplicitParam(name = "addressCode", required = false, value = "子公司编码", paramType = "query", dataType = "String"),
		@ApiImplicitParam(name = "languageCode", required = false, value = "语言编码", paramType = "query", dataType = "String"),
		@ApiImplicitParam(name = "isDefaultAddress", required = false, value = "默认地址", paramType = "query", dataType = "Integer") 
	})
   	@RequestMapping("service/updateDefaultAddress")
   	@ResponseBody
   	public ResponseResult<String> updateDefaultAddress(
   			@RequestParam(required = true) String companyId,
   			@RequestParam String addressCode,
   			@RequestParam String languageCode,
   			Integer isDefaultAddress) {
    	this.companyAddressService.updateDefaultAddress(companyId, addressCode, languageCode, isDefaultAddress);
   		return ResponseResult.success();
   	}
    
}