package com.els.base.file.service.impl;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.Properties;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import com.els.base.file.entity.FileData;
import com.els.base.file.service.FileDataService;
import com.els.base.file.service.IFileManager;
import com.els.base.file.utils.FileSaveTypeEnum;
import com.els.base.file.web.controller.FileDataController;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.uuid.UUIDGenerator;

/**
 * 本地文件管理
 * @author hzy
 *
 */
public class LocalFileManager implements IFileManager {

	private static String resourcePath = null;
	private static FileDataService fileDataService = SpringContextHolder.getOneBean(FileDataService.class);
	private static String SEPARATOR = "/";

	@Override
	public FileData write(InputStream in, FileData fileData) throws IOException {

		String relateFolderPath = this.createRelateFolderPath(fileData.getProjectId(), fileData.getCompanyId());

		File saveTarget = this.createSaveTarget(fileData.getFileSuffix(), relateFolderPath);
		Long fileSize = 0l;
		try {
			FileOutputStream output = new FileOutputStream(saveTarget);
			try {
				Integer fileSizeInt = IOUtils.copy(in, output);
				fileSize = fileSizeInt.longValue();
			} finally {
				IOUtils.closeQuietly(output);
			}
		} finally {
			IOUtils.closeQuietly(in);
		}

		fileData.setId(UUIDGenerator.generateUUID());
		fileData.setFileRename(saveTarget.getName());
		fileData.setLocalRelatPath(relateFolderPath.replaceAll("\\" + File.separator, SEPARATOR ));
		fileData.setFileSize(fileSize.intValue());
		fileData.setSaveType(FileSaveTypeEnum.LOCAL.getSaveType());
		
		String url = FileDataController.DISPLAY_URL_PREFIX + fileData.getId();
		fileData.setFileUrl(url);
		fileData.setCreateTime(new Date());
		fileDataService.addObj(fileData);

		return fileData;
	}

	@Override
	public FileData write(File file, FileData fileData) throws IOException {
		String relateFolderPath = this.createRelateFolderPath(fileData.getProjectId(), fileData.getCompanyId());
		File saveTarget = this.createSaveTarget(fileData.getFileSuffix(), relateFolderPath);
		FileUtils.copyFile(file, saveTarget);
		
		fileData.setId(UUIDGenerator.generateUUID());
		fileData.setFileName(file.getName());
		fileData.setFileRename(saveTarget.getName());
		fileData.setLocalRelatPath(relateFolderPath.replaceAll("\\" + File.separator, SEPARATOR ));
		fileData.setFileSize(Long.valueOf(FileUtils.readFileToByteArray(file).length).intValue());
		fileData.setSaveType(FileSaveTypeEnum.LOCAL.getSaveType());
		
		String url = FileDataController.DISPLAY_URL_PREFIX + fileData.getId();
		fileData.setFileUrl(url);
		fileData.setCreateTime(new Date());
		fileDataService.addObj(fileData);

		return fileData;
	}

	@Override
	public FileData write(MultipartFile file, FileData fileData) throws IOException {
		// 根据公司名称和项目名称来命名文件保存的路径
		String relateFolderPath = this.createRelateFolderPath(fileData.getProjectId(), fileData.getCompanyId());
		// 本机创建文件用来存储目标文件信息
		File saveTarget = this.createSaveTarget(fileData.getFileSuffix(), relateFolderPath);
		
		// 将目标文件转存
		file.transferTo(saveTarget);

		fileData.setId(UUIDGenerator.generateUUID());
		fileData.setFileName(file.getOriginalFilename());
		fileData.setFileRename(saveTarget.getName());
		fileData.setLocalRelatPath(relateFolderPath.replaceAll("\\" + File.separator, SEPARATOR ));
		fileData.setFileSize(Long.valueOf(file.getSize()).intValue());
		fileData.setSaveType(FileSaveTypeEnum.LOCAL.getSaveType());
		
		String url = FileDataController.DISPLAY_URL_PREFIX + fileData.getId();
		fileData.setFileUrl(url);

		// 文件信息保存到表里
		fileData.setCreateTime(new Date());
		fileDataService.addObj(fileData);
		
		return fileData;
	}

	@Override
	public InputStream read(FileData fileData) throws IOException {
		if (!this.isExist(fileData)) {
			throw new FileNotFoundException("文件不存在");
		}
		
		String fileAbsulutePath = resourcePath + File.separator
				+ fileData.getLocalRelatPath().replaceAll(SEPARATOR, "\\" + File.separator)
				+ fileData.getFileRename();
		File file = new File(fileAbsulutePath);
		
		return FileUtils.openInputStream(file);
	}

	@Transactional
	@Override
	public void remove(FileData fileData) throws IOException {
		fileDataService.deleteObjById(fileData.getId());		
		String absoluteFolderRootPath = resourcePath + File.separator;
		String filePath = absoluteFolderRootPath + fileData.getLocalRelatPath() + fileData.getFileRename();
		
		File file = new File(filePath);
		if (!file.exists()) {
			return;
		}
		
		FileUtils.forceDelete(file);
		
		String fileFolder = absoluteFolderRootPath + fileData.getLocalRelatPath();
		File folder = new File(fileFolder);
		if (folder.isDirectory() && CollectionUtils.isEmpty(FileUtils.listFiles(folder, null, true))) {
			FileUtils.forceDelete(folder);
		}
		
	}

	@Override
	public boolean isExist(FileData fileData) {
		String absoluteFolderRootPath = resourcePath + File.separator;
		String filePath = absoluteFolderRootPath + fileData.getLocalRelatPath() + fileData.getFileRename();
		
		File file = new File(filePath);
		return file.exists();
	}

	/**
	 * 创建 保存的目标文件
	 * 
	 * @param fileSuffix
	 * @return
	 */
	private File createSaveTarget(String fileSuffix, String relateFolderPath) {
		// 如果配置为空，就默认用户目录
		String absoluteFolderRootPath = resourcePath + File.separator;

		String folderPathStr = absoluteFolderRootPath + relateFolderPath;
		File folderPath = new File(folderPathStr);
		if (!folderPath.exists()) {
			folderPath.mkdirs();
		}

		String fileName = String.valueOf(System.currentTimeMillis()) + "_" + (int) (Math.random() * 1000000000) + "."
				+ fileSuffix;
		File tmp = new File(folderPathStr, fileName);
		return tmp;
	}

	/**
	 * 创建保存路径的相对路径，如果找到store，就加storeId为顶层目录
	 * 
	 * @return
	 */
	private String createRelateFolderPath(String projectId, String companyId) {
		String relateFolderPath = "project_" + projectId + File.separator + "company_" + companyId + File.separator
				+ DateFormatUtils.format(new Date(),
						"yyyy" + File.separator + "MM" + File.separator + "dd" + File.separator);

		return relateFolderPath;

	}

	@Override
	public void init() {
		Properties sysConfig = SpringContextHolder.getBean("sysConfig");
		
		String configResourcePath = sysConfig.getProperty("resource.location");
		if (StringUtils.isBlank(configResourcePath)) {
			configResourcePath = System.getProperty("user.home");
		}
		
		resourcePath = configResourcePath.replaceAll("(\\\\| \\/)$", "");
	}

	@Override
	public OutputStream createOutputStream(FileData fileData) throws IOException {
		String relateFolderPath = this.createRelateFolderPath(fileData.getProjectId(), fileData.getCompanyId());

		File saveTarget = this.createSaveTarget(fileData.getFileSuffix(), relateFolderPath);
		fileData.setId(UUIDGenerator.generateUUID());
		fileData.setFileRename(saveTarget.getName());
		fileData.setLocalRelatPath(relateFolderPath.replaceAll("\\" + File.separator, SEPARATOR ));
//		fileData.setFileSize(fileSize.doubleValue());
		fileData.setSaveType(FileSaveTypeEnum.LOCAL.getSaveType());
		
		String url = FileDataController.DISPLAY_URL_PREFIX + fileData.getId();
		fileData.setFileUrl(url);
		fileData.setCreateTime(new Date());
		fileDataService.addObj(fileData);
		
		FileOutputStream outputStream = new FileOutputStream(saveTarget);
		return outputStream;
	}

	@Override
	public File createEmptyFile(FileData fileData) {
		String relateFolderPath = this.createRelateFolderPath(fileData.getProjectId(), fileData.getCompanyId());

		File saveTarget = this.createSaveTarget(fileData.getFileSuffix(), relateFolderPath);
		fileData.setId(UUIDGenerator.generateUUID());
		fileData.setFileRename(saveTarget.getName());
		fileData.setLocalRelatPath(relateFolderPath.replaceAll("\\" + File.separator, SEPARATOR ));
//		fileData.setFileSize(fileSize.doubleValue());
		fileData.setSaveType(FileSaveTypeEnum.LOCAL.getSaveType());
		
		String url = FileDataController.DISPLAY_URL_PREFIX + fileData.getId();
		fileData.setFileUrl(url);
		fileData.setCreateTime(new Date());
		fileDataService.addObj(fileData);
		
		return saveTarget;
	}

}
