package com.els.base.file.web.controller;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.servlet.ModelAndView;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParam;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.file.entity.FileData;
import com.els.base.file.entity.FileDataExample;
import com.els.base.file.service.FileDataService;
import com.els.base.file.service.FileManagerFactory;
import com.els.base.file.service.IFileManager;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.reflect.ReflectUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@Api(value = "文件表")
@Controller
@RequestMapping("fileData")
public class FileDataController {
	
	private static Logger logger = LoggerFactory.getLogger(FileDataController.class);
	
	public static final Long CACHE_EXPIRED_TIME = 30 * 24 * 60 * 60 * 1000l;
	public static final String DISPLAY_URL_PREFIX = "fileData/service/display/";

	@Resource
	protected FileDataService fileDataService;

	@ApiOperation(value = "上传多个附件", httpMethod = "POST")
	@ApiImplicitParams({ 
		@ApiImplicitParam( name = "isEncrypt",required = false,value = "是否加密，1是，0否", paramType = "query", dataType = "String", defaultValue = "0" ),  
		@ApiImplicitParam( name = "sourceType", required = false, value = "文件的来源", paramType = "query", dataType = "String" )
	}) 
	@ResponseBody
	@RequestMapping(value = { "service/multiUpload" })
	public ResponseResult<List<FileData>> multiUpload(MultipartHttpServletRequest request) throws Exception {
		Map<String, MultipartFile> fileMap = request.getFileMap();
		// 标识上传文件是否需要加密
		String isEncrypt = request.getParameter("isEncrypt");
		String sourceType = request.getParameter("sourceType");

		if (MapUtils.isEmpty(fileMap)) {
			throw new CommonException("上传文件为空", "file_isNull");
		}

		Set<String> fileKeySet = fileMap.keySet();// 获取所有的key集合

		List<MultipartFile> fileList = new ArrayList<>();

		Iterator<String> keyIterator = fileKeySet.iterator();
		while (keyIterator.hasNext()) {
			fileList.add(fileMap.get(keyIterator.next()));
		}

		List<FileData> resultList = new ArrayList<>();

		for (MultipartFile file : fileList) {
			String fileSuffix = FilenameUtils.getExtension(file.getOriginalFilename());
			if (StringUtils.isBlank(fileSuffix)) {
				throw new CommonException("不接受后缀名为空的文件", "file_upload_not_accepted");
			}

			FileData fileUpload = new FileData();
			fileUpload.setProjectId(ProjectUtils.getProjectId());
			fileUpload.setCompanyId(currentCompanyId());
			fileUpload.setUserId(SpringSecurityUtils.getLoginUserId());
			fileUpload.setFileName(file.getOriginalFilename());
			fileUpload.setFileSuffix(fileSuffix);
			fileUpload.setIsEncrypt(isEncrypt);
			String validDay = request.getParameter("validDay");
			
			if (StringUtils.isNotBlank(validDay) && StringUtils.isNumeric(validDay)) {
				long validityTime = Long.valueOf(validDay) * 24 * 60 * 60 * 1000L;
				long time = validityTime + System.currentTimeMillis();
				Date expiryDay = new Date(time);
				fileUpload.setExpiryDay(expiryDay);
			}
			
			if (StringUtils.isNotBlank(sourceType)) {
				fileUpload.setSourceType(sourceType);
			}

			FileData result = FileManagerFactory.getFileManager().write(file, fileUpload);
			resultList.add(result);
		}

		return ResponseResult.success(resultList);
	}

	@ApiOperation(value = "上传单个附件", httpMethod = "POST")
	@ApiImplicitParams({ 
		@ApiImplicitParam( name = "isEncrypt",required = false,value = "是否加密，1是，0否", paramType = "query", dataType = "String", defaultValue = "0" ),  
		@ApiImplicitParam( name = "sourceType", required = false, value = "文件的来源", paramType = "query", dataType = "String" )
	}) 
	@ResponseBody
	@RequestMapping(value = { "service/upload" })
	public ResponseResult<FileData> upload(MultipartHttpServletRequest request) throws Exception {
		Map<String, MultipartFile> fileMap = request.getFileMap();
		// 标识上传文件是否需要加密
		String isEncrypt = request.getParameter("isEncrypt");
		String sourceType = request.getParameter("sourceType");

		if (MapUtils.isEmpty(fileMap)) {
			throw new CommonException("上传文件为空", "file_isNull");
		}

		Set<String> fileKeySet = fileMap.keySet();
		if (fileKeySet.size() > 1) {
			throw new CommonException("不接受多个文件上传", "file_upload_not_accepted");
		}

		MultipartFile file = null;

		Iterator<String> keyIterator = fileKeySet.iterator();
		while (keyIterator.hasNext()) {
			file = fileMap.get(keyIterator.next());
		}

		if (file == null || file.isEmpty()) {
			throw new CommonException("不接受多个文件上传", "file_upload_not_accepted");
		}

		String fileSuffix = FilenameUtils.getExtension(file.getOriginalFilename());
		if (StringUtils.isBlank(fileSuffix)) {
			throw new CommonException("不接受后缀名为空的文件", "file_upload_not_accepted");
		}

		FileData fileUpload = new FileData();
		fileUpload.setProjectId(ProjectUtils.getProjectId());
		fileUpload.setCompanyId(currentCompanyId());
		fileUpload.setUserId(SpringSecurityUtils.getLoginUserId());
		fileUpload.setFileName(file.getOriginalFilename());
		fileUpload.setFileSuffix(fileSuffix.toLowerCase());
		fileUpload.setIsEncrypt(isEncrypt);
		// 临时文件的有效期
		String validDay = request.getParameter("validDay");
		if (StringUtils.isNotBlank(validDay) && StringUtils.isNumeric(validDay)) {
			long validityTime = Long.valueOf(validDay) * 24 * 60 * 60 * 1000L;
			long time = validityTime + System.currentTimeMillis();
			Date expiryDay = new Date(time);
			fileUpload.setExpiryDay(expiryDay);
		}
		
		if (StringUtils.isNotBlank(sourceType)) {
			fileUpload.setSourceType(sourceType);
		}

		FileData result = FileManagerFactory.getFileManager().write(file, fileUpload);

		return ResponseResult.success(result);
	}

	@ApiOperation(value = "在线查看文件", httpMethod = "GET")
	@RequestMapping(value = {"service/display/{fileId}", "front/display/{fileId}"})
	public ModelAndView display(@PathVariable String fileId, HttpServletRequest request, HttpServletResponse response) {
		try {
			if (!this.isExpired(request, response)) {
				response.setStatus(HttpServletResponse.SC_NOT_MODIFIED);
				return null;
			}

			FileData fileData = this.fileDataService.queryObjById(fileId);
			if (fileData == null) {
				response.setStatus(HttpServletResponse.SC_NOT_FOUND); // 返回404状态码
				return null;
			}
			IFileManager fileManager = FileManagerFactory.getFileManager(fileData.getSaveType());
			if (!fileManager.isExist(fileData)) {
				response.setStatus(HttpServletResponse.SC_NOT_FOUND); // 返回404状态码
				return null;
			}

			response.addHeader("Content-Disposition", "inline;filename=\"" + URLEncoder.encode(fileData.getFileName(), "UTF-8") + "\";");
			// 设置下载文件名, 设置expire
			if ("png,jpg,jpeg,gif,ico".contains(fileData.getFileSuffix())) {
				response.setContentType("image/" + fileData.getFileSuffix());

			} else if ("pdf".contains(fileData.getFileSuffix())) {
				response.setContentType("application/pdf");

			} else if ("html".contains(fileData.getFileSuffix())) {
				response.setContentType("text/html");
			}
			this.setRespHeaderCache(response);

			InputStream inputStream = fileManager.read(fileData);
			OutputStream outputStream = response.getOutputStream();

			byte[] tmp = new byte[1024];
			while (inputStream.read(tmp) != -1) {
				outputStream.write(tmp);
			}

			outputStream.flush();
			outputStream.close();
			inputStream.close();
		} catch (Exception e) {
			e.printStackTrace();
			response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
		}

		return null;
	}

	/**
	 * 下载附件
	 * 
	 * @param fileId
	 * @param request
	 * @param response
	 * @return
	 */
	@ApiOperation(value = "下载文件", httpMethod = "GET")
	@RequestMapping(value = {"service/download/{fileId}", "front/download/{fileId}"})
	@ResponseBody
	public ModelAndView download(@PathVariable String fileId, HttpServletRequest request, HttpServletResponse response) {
		try {

			FileData fileData = this.fileDataService.queryObjById(fileId);
			if (fileData == null) {
				response.setStatus(HttpServletResponse.SC_NOT_FOUND); // 返回404状态码
				return null;
			}

			IFileManager fileManager = FileManagerFactory.getFileManager(fileData.getSaveType());
			if (!fileManager.isExist(fileData)) {
				response.setStatus(HttpServletResponse.SC_NOT_FOUND); // 返回404状态码
				return null;
			}
			// System.out.println(fileData.getFileName() + "te");
			response.reset();

			StringBuffer header = new StringBuffer("attachment;");
			// header.append("filename=\""+ URLEncoder.encode("通过邮件链接自动登录打开对应链接方法整理.txt", "UTF-8") + "\";");
			header.append("filename=\"" + URLEncoder.encode(fileData.getFileName(), "UTF-8") + "\";");
			response.setHeader("Content-Disposition", header.toString());

			response.setCharacterEncoding("UTF-8");
			response.setContentType("application/octet-stream;charset=UTF-8");

			InputStream inputStream = fileManager.read(fileData);
			OutputStream outputStream = response.getOutputStream();

			byte[] tmp = new byte[1024];
			while (inputStream.read(tmp) != -1) {
				outputStream.write(tmp);
			}

			outputStream.flush();
			outputStream.close();
			inputStream.close();
		} catch (Exception e) {
			e.printStackTrace();
			response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
		}

		return null;
	}

	/**
	 * 删除附件
	 * 
	 * @param fileId
	 * @param request
	 * @param response
	 * @return
	 */
	@ApiOperation(value = "删除文件", httpMethod = "GET")
	@RequestMapping(value = "service/deletedFile/{fileId}")
	@ResponseBody
	public ModelAndView deletedFile(@PathVariable String fileId, HttpServletRequest request, HttpServletResponse response) {
		try {

			FileData fileData = this.fileDataService.queryObjById(fileId);
			if (fileData == null) {
				response.setStatus(HttpServletResponse.SC_NOT_FOUND); // 返回404状态码
				return null;
			}

			IFileManager fileManager = FileManagerFactory.getFileManager(fileData.getSaveType());
			if (!fileManager.isExist(fileData)) {
				response.setStatus(HttpServletResponse.SC_NOT_FOUND); // 返回404状态码
				return null;
			}
			FileManagerFactory.getFileManager().remove(fileData);
		} catch (Exception e) {
			e.printStackTrace();
			response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
		}

		return null;
	}

	@ApiOperation(httpMethod = "POST", value = "查询文件表")
	@RequestMapping("service/findByPage")
	@ResponseBody
	public ResponseResult<PageView<FileData>> findByPage(@ApiParam(value = "所在页", defaultValue = "0") @RequestParam(defaultValue = "0") int pageNo, @ApiParam(value = "每页数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize, @ApiParam(value = "查询条件,属性名请参考 FileData") @RequestBody(required = false) List<QueryParam> params) {
		FileDataExample example = new FileDataExample();
		example.setPageView(new PageView<FileData>(pageNo, pageSize));

		if (CollectionUtils.isNotEmpty(params)) {
			FileDataExample.Criteria criteria = example.createCriteria();
			CriteriaUtils.addCriterion(criteria, params);
		}

		PageView<FileData> pageData = this.fileDataService.queryObjByPage(example);
		return ResponseResult.success(pageData);
	}
	
	@ApiOperation(httpMethod = "POST", value = "批量删除文件")
	@RequestMapping("service/deleteByExample")
	@ResponseBody
	public ResponseResult<Integer> deleteByExample(@ApiParam(value = "查询条件,属性名请参考 FileData") @RequestBody(required = false) QueryParamWapper queryParamWapper) throws IOException {
		if (queryParamWapper == null) {
			throw new CommonException("筛选条件为空");
		}
		
		FileDataExample example = new FileDataExample();

		if (queryParamWapper != null) {
			FileDataExample.Criteria criteria = example.createCriteria();
			CriteriaUtils.addCriterion(criteria, queryParamWapper);
		}
		
		List<FileData> list = this.fileDataService.queryAllObjByExample(example);
		if (CollectionUtils.isEmpty(list)) {
			return ResponseResult.success(0);
		}
		
		for(FileData fileData: list){
			IFileManager fileManager = FileManagerFactory.getFileManager(fileData.getSaveType());
			fileManager.remove(fileData);
		}

		return ResponseResult.success(list.size());
	}
	

	/**
	 * 浏览器缓存是否已过期
	 * 
	 * @param request
	 * @param response
	 * @return
	 */
	private boolean isExpired(HttpServletRequest request, HttpServletResponse response) {
		long header = request.getDateHeader("If-Modified-Since");
		long now = System.currentTimeMillis();

		long expiredTime = 0l;
		if (header > 0) {
			expiredTime = header + CACHE_EXPIRED_TIME;
		}

		if (now > expiredTime) {
			// 如果超出缓存时间，过期
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 缓存时间，单位 毫秒
	 * 
	 * @param response
	 * @return
	 */
	private boolean setRespHeaderCache(HttpServletResponse response) {

		String maxAgeDirective = "max-age=" + (CACHE_EXPIRED_TIME / 1000);
		response.setHeader("Cache-Control", maxAgeDirective);

		response.addDateHeader("Last-Modified", System.currentTimeMillis());
		response.addDateHeader("Expires", System.currentTimeMillis() + CACHE_EXPIRED_TIME);

		response.setStatus(HttpServletResponse.SC_OK);
		return true;
	}
	
	public static String currentCompanyId(){
		
		String userId = SpringSecurityUtils.getLoginUserId();
		if(StringUtils.isBlank(userId)){
			return null;
		}
		
		String companyId = null;
		try {
			Object companyUserRefService = SpringContextHolder.getOneBean(Class.forName("com.els.base.company.service.CompanyUserRefService"));
			companyId = (String) ReflectUtils.invokeMethod(companyUserRefService, "queryCompanyIdOfUser", userId);
		} catch (ClassNotFoundException | NoSuchMethodException | InvocationTargetException | IllegalAccessException e) {
			logger.warn("获取当前companyid失败", e);
		}
		return companyId;
	}
}