package com.mcoding.base.generator.plugins;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;
import org.mybatis.generator.api.GeneratedJavaFile;
import org.mybatis.generator.api.IntrospectedTable;
import org.mybatis.generator.api.PluginAdapter;
import org.mybatis.generator.api.dom.java.CompilationUnit;
import org.mybatis.generator.api.dom.java.FullyQualifiedJavaType;
import org.mybatis.generator.api.dom.java.JavaVisibility;
import org.mybatis.generator.api.dom.java.Method;
import org.mybatis.generator.api.dom.java.Parameter;
import org.mybatis.generator.api.dom.java.TopLevelClass;

import com.mcoding.base.generator.utils.ServiceGenerateDataStorage;
import com.mcoding.base.generator.utils.TableCommentStorage;


public class GenerateTestCasePlugin extends PluginAdapter {

	private String targetPackage;
	private String targetProject;
	private String serviceTargetPackage;
	private String baseUrlPath;
	private String moduleName;

	@Override
	public boolean validate(List<String> warnings) {
		String targetPackage = this.properties.getProperty("targetPackage");
		if (StringUtils.isBlank(targetPackage)) {
			warnings.add("Controller 生成失败， targetPackage 配置失败");
			return false;
		}

		if (!targetPackage.matches("^(\\w+)(\\.\\w*)*\\w$")) {
			warnings.add("TaseCase 生成失败， targetPackage[" + targetPackage + "] 格式错误");
			return false;
		}

		String targetProject = this.properties.getProperty("targetProject");
		if (StringUtils.isBlank(targetProject)) {
			if (StringUtils.isBlank(targetPackage)) {
				warnings.add("TaseCase 生成失败， targetProject 配置失败");
				return false;
			}
		}
		
		String baseUrlPath = this.properties.getProperty("baseUrlPath");
		
		String moduleName = this.properties.getProperty("moduleName");
		if (StringUtils.isBlank(moduleName)) {
			moduleName = "";
		}else {
			moduleName = moduleName + "/";
		}

		this.targetPackage = targetPackage;
		this.targetProject = targetProject;
		this.baseUrlPath = baseUrlPath;
		this.moduleName = moduleName;
		return true;
	}

	@Override
	public List<GeneratedJavaFile> contextGenerateAdditionalJavaFiles(IntrospectedTable introspectedTable) {
		String modelName = introspectedTable.getTableConfiguration().getDomainObjectName();
		String tableName = introspectedTable.getTableConfiguration().getTableName();

		this.serviceTargetPackage = ServiceGenerateDataStorage.getInstance().getServicePackage(tableName);
		if (StringUtils.isBlank(serviceTargetPackage)) {
			throw new NullPointerException("TaseCase 生成失败，因为关联的service未生成，请检查插件[GenerateServicePlugin]配置的顺序。");
		}

		GeneratedJavaFile testCase = null;
		try {
			testCase = new GeneratedJavaFile(this.createTestCase(introspectedTable, modelName), this.targetProject,
					this.context.getJavaFormatter());
		} catch (SQLException e) {
			e.printStackTrace();
		}

		List<GeneratedJavaFile> list = new ArrayList<>(2);
		list.add(testCase);
		return list;
	}

	private CompilationUnit createTestCase(IntrospectedTable introspectedTable, String modelClassName)
			throws SQLException {
		String testCaseFullName = this.targetPackage + "." + modelClassName + "ControllerTest";
		String tableComment = TableCommentStorage.getInstance().get(introspectedTable);
		if (StringUtils.isBlank(tableComment)) {
			tableComment = modelClassName;
		}

		FullyQualifiedJavaType controllerType = new FullyQualifiedJavaType(testCaseFullName);
		TopLevelClass testCase = new TopLevelClass(controllerType);
		testCase.setVisibility(JavaVisibility.PUBLIC);
		
		FullyQualifiedJavaType superClass = new FullyQualifiedJavaType("BaseTest<"+ modelClassName+">");
		testCase.setSuperClass(superClass);

		testCase.addAnnotation("@SpringBootTest(classes=BaseStarterApplication.class, webEnvironment=SpringBootTest.WebEnvironment.DEFINED_PORT)");
		testCase.addAnnotation("@FixMethodOrder(value = MethodSorters.NAME_ASCENDING)");
		
		if(StringUtils.isBlank(this.baseUrlPath)){
			this.baseUrlPath = StringUtils.uncapitalize(modelClassName);
		}

		List<FullyQualifiedJavaType> importList = this.getImportList(introspectedTable, modelClassName);
		for (FullyQualifiedJavaType importItem : importList) {
			testCase.addImportedType(importItem);
		}

		this.addFieldsAndMethod(testCase, introspectedTable, modelClassName);

		return testCase;
	}

	private void addFieldsAndMethod(TopLevelClass testCase, IntrospectedTable introspectedTable,
			String modelClassName) throws SQLException {
		
		testCase.addMethod(this.methodTestCreate(introspectedTable, modelClassName));
		testCase.addMethod(this.methodTestEdit(introspectedTable, modelClassName));
		testCase.addMethod(this.methodTestDeleteByIds(introspectedTable, modelClassName));
//		testCase.addMethod(this.methodFindByPage(introspectedTable, modelClassName));

	}

	private Method methodToAddView(String modelClassName) {
		/*
		 * @ApiIgnore
		 * 
		 * @RequestMapping("service/toAddView") public ModelAndView toAddView()
		 * { ModelAndView view = new ModelAndView();
		 * view.setViewName("dictionary/dicGroup/toAddView"); return view; }
		 */

		Method method = new Method();
		method.setName("toAddView");
		method.setVisibility(JavaVisibility.PUBLIC);
		method.setReturnType(new FullyQualifiedJavaType("ModelAndView"));
		
		String path = this.moduleName + StringUtils.uncapitalize(modelClassName);

		method.addBodyLine("return new ModelAndView(\"" + path + "/toAddView\");");

		method.addAnnotation("@ApiIgnore");
		method.addAnnotation("@RequestMapping(\"service/toAddView\")");
		return method;
	}
	
	private Method methodToMainView(String modelClassName){
		/*@ApiIgnore
		@RequestMapping("service/toListPageView")
		public ModelAndView toMainView() {
			ModelAndView view = new ModelAndView();
			view.setViewName("dictionary/dicGroup/toMainView");
			return view;
		}*/
		
		Method method = new Method();
		method.setName("toMainView");
		method.setVisibility(JavaVisibility.PUBLIC);
		method.setReturnType(new FullyQualifiedJavaType("ModelAndView"));

		String path = this.moduleName + StringUtils.uncapitalize(modelClassName);
		method.addBodyLine("return new ModelAndView(\"" + path + "/toMainView\");");

		method.addAnnotation("@ApiIgnore");
		method.addAnnotation("@RequestMapping(\"service/toMainView\")");
		return method;
	}
	
	public Method methodToUpdateViewById(String modelClassName) {
		/*
		@ApiIgnore
		@RequestMapping("service/toUpdateViewById")
		public ModelAndView toDicGroupById(int id) {
			ModelAndView view = new ModelAndView();
			
			DicGroup dicGroup = this.dicGroupService.queryObjById(id);
			
			view.addObject("dicGroup", dicGroup);
			view.setViewName("dictionary/dicGroup/toAddView");
			return view;
		}*/
		
		Method method = new Method();
		method.setName("toUpdateViewById");
		method.setVisibility(JavaVisibility.PUBLIC);
		method.setReturnType(new FullyQualifiedJavaType("ModelAndView"));
		
		Parameter parameter = new Parameter(new FullyQualifiedJavaType("int"), "id");
		method.addParameter(parameter);

		String littleModel = StringUtils.uncapitalize(modelClassName);
		method.addBodyLine("ModelAndView view = new ModelAndView();");
		method.addBodyLine(modelClassName +" "+littleModel+" = this."+littleModel+"Service.queryObjById(id);");
		method.addBodyLine("view.addObject(\""+littleModel+"\", "+littleModel+");");
		
		String path = this.moduleName + littleModel;
		method.addBodyLine("view.setViewName(\""+path+"/toAddView\");");
		method.addBodyLine("return view;");
		
		method.addAnnotation("@ApiIgnore");
		method.addAnnotation("@RequestMapping(\"service/toUpdateViewById\")");
		return method;
	}
	
	private Method methodTestCreate(IntrospectedTable introspectedTable, String modelClassName) throws SQLException{
		/*
		@Test
		public void test01Create() throws Exception{
			String name = "testName-" + DateFormatUtils.format(new Date(), "yyyy-MM-dd");
			String code = "testCode-" + DateFormatUtils.format(new Date(), "yyyy-MM-dd");
			
			Demo demo = new Demo();
			demo.setCreateTime(new Date());
			demo.setName(name);
			demo.setCode(code);
			
			ResultActions result = this.create("/demo/service/create", demo);
			
			JsonPath jsonPath = JsonPath.compile("$.data");
			id = jsonPath.read(result.andReturn().getResponse().getContentAsString());
			Assert.assertNotNull("id 为空，创建失败", id);
			
			this.findByPage("/demo/service/findByPage", id)
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult").isNotEmpty())
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult[0].id").value(id))
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult[0].name").value(name))
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult[0].code").value(code))
				;
		}
		*/
		
		String littleModel = StringUtils.uncapitalize(modelClassName);
		
		Method method = new Method();
		method.setName("test01Create");
		method.setVisibility(JavaVisibility.PUBLIC);
		method.setReturnType(new FullyQualifiedJavaType("void"));
		method.addException(new FullyQualifiedJavaType("Exception"));
		method.addAnnotation("@Test");
		
		method.addBodyLine(modelClassName + " " + littleModel + " = " + "new " + modelClassName + "();");
		method.addBodyLine("ResultActions result = this.create(\"/"+littleModel+"/service/create\", "+littleModel+");");
		
		method.addBodyLine("JsonPath jsonPath = JsonPath.compile(\"$.data\");");
		method.addBodyLine("id = jsonPath.read(result.andReturn().getResponse().getContentAsString());");
		method.addBodyLine("Assert.assertNotNull(\"id 为空，创建失败\", id);");
		method.addBodyLine("");
		
		method.addBodyLine("this.findByPage(\"/"+littleModel+"/service/findByPage\", id)");
		method.addBodyLine("    .andExpect(MockMvcResultMatchers.jsonPath(\"data.queryResult\").isNotEmpty())");
		method.addBodyLine("    .andExpect(MockMvcResultMatchers.jsonPath(\"data.queryResult[0].id\").value(id));");
		
		return method;
	}
	
	private Method methodTestEdit(IntrospectedTable introspectedTable, String modelClassName) throws SQLException{
		/*
		@Test
		public void test02Edit() throws Exception{
			Assert.assertNotNull("id 不能为空", id);
			
			ResultActions beforeEdit = this.findByPage("/demo/service/findByPage", id);
			
			JsonPath jsonPath = JsonPath.compile("$.data.queryResult[0].code");
			String code = jsonPath.read(beforeEdit.andReturn().getResponse().getContentAsString());
			
			String newName = "demoNewName";
			Demo demo = new Demo();
			demo.setId(id);
			demo.setName(newName);
			
			this.edit("/demo/service/edit", demo);
			
			this.findByPage("/demo/service/findByPage", id)
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult").isNotEmpty())
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult[0].id").value(id))
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult[0].name").value(newName))
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult[0].code").value(code))
				;
		}
		*/
		
		String littleModel = StringUtils.uncapitalize(modelClassName);

		
		Method method = new Method();
		method.setName("test02Edit");
		method.setVisibility(JavaVisibility.PUBLIC);
		method.setReturnType(new FullyQualifiedJavaType("void"));
		method.addException(new FullyQualifiedJavaType("Exception"));
		method.addAnnotation("@Test");
		
		method.addBodyLine("Assert.assertNotNull(\"id 不能为空\", id);");
		method.addBodyLine(modelClassName + " " + littleModel + " = " + "new " + modelClassName + "();");
		method.addBodyLine(littleModel + ".setId(id);");
		method.addBodyLine("");
		
		method.addBodyLine("this.edit(\"/"+littleModel+"/service/edit\", "+littleModel+");");
		
		method.addBodyLine("this.findByPage(\"/"+littleModel+"/service/findByPage\", id)");
		method.addBodyLine("    .andExpect(MockMvcResultMatchers.jsonPath(\"data.queryResult\").isNotEmpty())");
		method.addBodyLine("    .andExpect(MockMvcResultMatchers.jsonPath(\"data.queryResult[0].id\").value(id));");
		
		return method;
	}
	
	private Method methodTestDeleteByIds(IntrospectedTable introspectedTable, String modelClassName) throws SQLException{
		/*
		@Test
		public void test03Delete() throws Exception{
			Assert.assertNotNull("id 不能为空", id);
			
			List<String> ids = new ArrayList<>();
			ids.add(id);
			
			this.deleteByIds("/demo/service/deleteByIds", ids);
			
			this.findByPage("/demo/service/findByPage", id)
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult").isEmpty())
			;
		}
		*/
		
		String littleModel = StringUtils.uncapitalize(modelClassName);

		Method method = new Method();
		method.setName("test03Delete");
		method.setVisibility(JavaVisibility.PUBLIC);
		method.setReturnType(new FullyQualifiedJavaType("void"));
		method.addException(new FullyQualifiedJavaType("Exception"));
		method.addAnnotation("@Test");
		

		method.addBodyLine("List<String> ids = new ArrayList<>();");
		method.addBodyLine("ids.add(id);");
		method.addBodyLine("");
		method.addBodyLine("this.deleteByIds(\"/"+littleModel+"/service/deleteByIds\", ids);");
		method.addBodyLine("");
		
		
		method.addBodyLine("this.findByPage(\"/"+littleModel+"/service/findByPage\", id)");
		method.addBodyLine("    .andExpect(MockMvcResultMatchers.jsonPath(\"data.queryResult\").isEmpty());");
		return method;
		
	}
	
	private Method methodFindByPage(IntrospectedTable introspectedTable, String modelClassName) throws SQLException{
		/*
		protected ResultActions findByPage(String url, String id) throws Exception{

			MockHttpServletRequestBuilder request = MockMvcRequestBuilders.post(url)
				.param("pageNo","1").param("pageSize","10")
				.contentType(MediaType.APPLICATION_JSON);
			
			if (StringUtils.isNotBlank(id)) {
				Map<String, String> queryWapper = new HashMap<>();
				queryWapper.put("id_$_eq", id);
				request.content(JsonUtils.writeValueAsString(queryWapper));
			}
			
			ResultActions actions = getMockMvc().perform(request);
	//		actions.andDo(MockMvcResultHandlers.print());
			
			ResultActions result = actions.andExpect(MockMvcResultMatchers.status().isOk())
				.andExpect(MockMvcResultMatchers.jsonPath("code").value("200"))
				.andExpect(MockMvcResultMatchers.jsonPath("data.queryResult").exists());
			
			return result;
		}*/
		
		Method method = new Method();
		method.setName("findByPage");
		method.setVisibility(JavaVisibility.PUBLIC);
		method.addException(new FullyQualifiedJavaType("Exception"));
		method.setReturnType(new FullyQualifiedJavaType("ResultActions"));
		
		method.addParameter(new Parameter(new FullyQualifiedJavaType("String"), "url"));
		method.addParameter(new Parameter(new FullyQualifiedJavaType("String"), "id"));
		
		method.addBodyLine("MockHttpServletRequestBuilder request = MockMvcRequestBuilders.post(url)");
		method.addBodyLine("    .param(\"pageNo\",\"1\").param(\"pageSize\",\"10\")");
		method.addBodyLine("    .contentType(MediaType.APPLICATION_JSON);");
		
		method.addBodyLine("if (StringUtils.isNotBlank(id)) {");
		method.addBodyLine("    Map<String, String> queryWapper = new HashMap<>();");
		method.addBodyLine("    queryWapper.put(\"id_$_eq\", id);");
		method.addBodyLine("    request.content(JsonUtils.writeValueAsString(queryWapper));");
		method.addBodyLine("ResultActions actions = getMockMvc().perform(request);");
		method.addBodyLine("ResultActions result = actions.andExpect(MockMvcResultMatchers.status().isOk())");
		method.addBodyLine("    .andExpect(MockMvcResultMatchers.jsonPath(\"code\").value(\"200\"))");
		method.addBodyLine("    .andExpect(MockMvcResultMatchers.jsonPath(\"data.queryResult\").exists());");
		method.addBodyLine("return result;");
		
		return method;
	}

	private List<FullyQualifiedJavaType> getImportList(IntrospectedTable introspectedTable, String modelClassName) {
		List<String> packageStrList = Arrays.asList( "java.util.ArrayList",
			 "java.util.List",
			 "org.junit.Assert",
			 "org.junit.FixMethodOrder",
			 "org.junit.Test",
			 "org.junit.runners.MethodSorters",
			 "org.springframework.boot.test.context.SpringBootTest",
			 "org.springframework.test.web.servlet.ResultActions",
			 "org.springframework.test.web.servlet.result.MockMvcResultMatchers",
			 "com.els.base.core.BaseStarterApplication",
			 "com.els.base.test.BaseTest",
			 "com.jayway.jsonpath.JsonPath");
		
		List<FullyQualifiedJavaType> list = packageStrList.stream().map(FullyQualifiedJavaType::new).collect(Collectors.toList());
		
		list.add(this.getModelType(introspectedTable, modelClassName));
//		list.add(this.getExampleType(introspectedTable, modelClassName));
//		list.add(this.getServiceType(introspectedTable, modelClassName));

		return list;
	}

	private FullyQualifiedJavaType getModelType(IntrospectedTable introspectedTable, String modelClassName) {

		String beanPackageStr = introspectedTable.getContext().getJavaModelGeneratorConfiguration().getTargetPackage();
		String fullModelClassName = beanPackageStr + "." + modelClassName;

		FullyQualifiedJavaType modelType = new FullyQualifiedJavaType(fullModelClassName);
		return modelType;
	}

}
