package com.els.base.msg.mail.service.impl;

import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.msg.mail.dao.MailTemplateMapper;
import com.els.base.msg.mail.entity.MailTemplate;
import com.els.base.msg.mail.entity.MailTemplateExample;
import com.els.base.msg.mail.service.MailTemplateService;

@Service("defaultMailTemplateService")
public class MailTemplateServiceImpl implements MailTemplateService {
    @Resource
    protected MailTemplateMapper mailTemplateMapper;

    @CacheEvict(value={"mailTemplate"}, allEntries=true)
    @Override
    public void addObj(MailTemplate t) {
    	Assert.isNotBlank(t.getBusinessType(), "业务类型不能为空");
    	Assert.isNotBlank(t.getTemplateTitle(), "邮件模板的标题不能为空");
    	Assert.isNotBlank(t.getTemplateContent(), "邮件模板的内容不能为空");
    	Assert.isNotBlank(t.getTemplateName(), "邮件模板的名称不能为空");
    	Assert.isNotBlank(t.getTemplateNum(), "邮件模板的编码不能为空");
    	
    	MailTemplateExample example = new MailTemplateExample();
    	example.createCriteria().andBusinessTypeEqualTo(t.getBusinessType());
    	
    	if (this.mailTemplateMapper.countByExample(example) > 0) {
			throw new CommonException("已经存在该业务类型的模板，不能重复创建");
		}
    	
    	example.clear();
    	example.createCriteria().andTemplateNumEqualTo(t.getTemplateNum());
    	if (this.mailTemplateMapper.countByExample(example) > 0) {
    		throw new CommonException("已经存在该编码的模板，不能重复创建");
    	}
    	
    	t.setCreateDate(new Date());
    	t.setModifyDate(new Date());
        this.mailTemplateMapper.insertSelective(t);
    }

    @CacheEvict(value={"mailTemplate"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.mailTemplateMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"mailTemplate"}, allEntries=true)
    @Override
    public void modifyObj(MailTemplate mailTemplate) {
        if (StringUtils.isBlank(mailTemplate.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        
        MailTemplateExample example = new MailTemplateExample();
    	example.createCriteria().andBusinessTypeEqualTo(mailTemplate.getBusinessType()).andIdNotEqualTo(mailTemplate.getId());
    	
    	if (this.mailTemplateMapper.countByExample(example) > 0) {
			throw new CommonException("已经存在该业务类型的模板，不能重复创建");
		}
    	
    	example.clear();
    	example.createCriteria().andTemplateNumEqualTo(mailTemplate.getTemplateNum()).andIdNotEqualTo(mailTemplate.getId());
    	if (this.mailTemplateMapper.countByExample(example) > 0) {
    		throw new CommonException("已经存在该编码的模板，不能重复创建");
    	}
        
    	mailTemplate.setCompanyId(null);
    	mailTemplate.setProjectId(null);
    	mailTemplate.setCreateDate(null);
        mailTemplate.setModifyDate(new Date());
        this.mailTemplateMapper.updateByPrimaryKeySelective(mailTemplate);
    }

    @Cacheable(value="mailTemplate", keyGenerator="redisKeyGenerator")
    @Override
    public MailTemplate queryObjById(String id) {
        return this.mailTemplateMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="mailTemplate", keyGenerator="redisKeyGenerator")
    @Override
    public List<MailTemplate> queryAllObjByExample(MailTemplateExample example) {
        return this.mailTemplateMapper.selectByExample(example);
    }

    @Cacheable(value="mailTemplate", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<MailTemplate> queryObjByPage(MailTemplateExample example) {
        PageView<MailTemplate> pageView = example.getPageView();
        pageView.setQueryResult(this.mailTemplateMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Cacheable(value="mailTemplate", keyGenerator="redisKeyGenerator")
	@Override
	public MailTemplate queryMailTemplateByBusinessType(String businessType) {
		MailTemplateExample mailTemplateExample = new MailTemplateExample();
        mailTemplateExample.createCriteria().andBusinessTypeEqualTo(businessType);

        List<MailTemplate> mailTemplateList = this.mailTemplateMapper.selectByExample(mailTemplateExample);
        if (CollectionUtils.isEmpty(mailTemplateList)) {
			return null;
		}
        return mailTemplateList.get(0);
	}
}